package clinical.web.helpers;

import gnu.trove.TObjectIntHashMap;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.lang.NumberUtils;
import org.jdom.Element;

import clinical.server.vo.Experiment;
import clinical.server.vo.Rawdata;
import clinical.utils.Assertion;
import clinical.utils.BIRNURIUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.download.DataNode;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.FileInfo;
import clinical.web.download.SubjectImageDataBundle;
import clinical.web.vo.ExpSubjectsIDBInfo;
import clinical.web.vo.RawDataObject;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ShoppingCartHelper.java 873 2015-01-07 20:55:20Z jinranc $
 */
public class ShoppingCartHelper {
	public final static int LOCAL = 100;
	public final static int SRB = 200;
	public final static int GRIDFTP = 300;
	
	private static Log log = LogFactory.getLog(ShoppingCartHelper.class);
	public static List<RawDataObject> removeIntermediatePaths(
			List<RawDataObject> rdoList) {
		List<RawDataObject> newList = new ArrayList<RawDataObject>(rdoList.size());
		Map<String, List<DataPathInfo>> seenBaseMap = new LinkedHashMap<String, List<DataPathInfo>>();
		int minLevel = Integer.MAX_VALUE;
		for (Iterator<RawDataObject> iter = rdoList.iterator(); iter.hasNext();) {
			RawDataObject rdo = iter.next();
			Rawdata rd = rdo.getRawData();
			String dataURI = rd.getDatauri();
			String[] parts = dataURI.split("\\/");

			StringBuffer buf = new StringBuffer();
			buf.append(rd.getSegmentid()).append('_');
			buf.append(rd.getComponentid()).append('_');
			buf.append(rd.getNcExperimentUniqueid());

			DataPathInfo dpi = new DataPathInfo(rdo, parts, buf.toString(),
					dataURI);
			if (dpi.numLevels() < minLevel) {
				minLevel = dpi.numLevels();
			}
			List<DataPathInfo> dpiList = seenBaseMap.get(dpi.getIdentifier());
			if (dpiList == null) {
				dpiList = new LinkedList<DataPathInfo>();
				seenBaseMap.put(dpi.getIdentifier(), dpiList);
			}
			dpiList.add(dpi);
		}

		for (Iterator<List<DataPathInfo>> iter = seenBaseMap.values().iterator(); iter
				.hasNext();) {
			List<DataPathInfo> dpiList = iter.next();
			if (areAllSameLevel(dpiList)) {
				for (DataPathInfo dpi : dpiList) {
					newList.add(dpi.rdo);
				}
			} else {
				List<DataPathInfo> niDpiList = findNonIntermediaryPaths(dpiList);
				for (Iterator<DataPathInfo> it = niDpiList.iterator(); it.hasNext();) {
					DataPathInfo dpi = it.next();
					newList.add(dpi.rdo);
				}
			}
		}

		return newList;
	}

	public static List<DataPathInfo> findNonIntermediaryPaths(
			List<DataPathInfo> dpiList) {
		// sort by levelNum ascending
		Collections.sort(dpiList, new Comparator<DataPathInfo>() {
			public int compare(DataPathInfo d1, DataPathInfo d2) {
				return d1.numLevels() - d2.numLevels();
			}
		});

		List<DataPathInfo> niDpiList = new ArrayList<DataPathInfo>(dpiList.size());
		for (Iterator<DataPathInfo> iter = dpiList.iterator(); iter.hasNext();) {
			DataPathInfo dpi = iter.next();
			int numLevel = dpi.numLevels();
			boolean skip = true;
			boolean intermediate = false;
			for (Iterator<DataPathInfo> it = dpiList.iterator(); it.hasNext();) {
				DataPathInfo dpi2 = it.next();
				if (skip) {
					if (dpi2 == dpi) {
						skip = false;
					}
					continue;
				} else {
					if (dpi2.numLevels() > numLevel) {
						String path = GenUtils.join(dpi2.parts, "/", 0, numLevel);
						if (path.equals(dpi.dataURI)) {
							intermediate = true;
							break;
						}
					}
				}
			}
			if (!intermediate) {
				niDpiList.add(dpi);
			}
		}
		return niDpiList;
	}

	public static boolean isIntermediate(DataPathInfo dpi, int minLevel) {
		int level = dpi.numLevels();
		for (int i = level - 2; i >= minLevel; i--) {
			String path = GenUtils.join(dpi.parts, "/", 0, i);
			System.out.println("path:" + path);
		}
		return false;
	}

	public static boolean areAllSameLevel(List<DataPathInfo> dpiList) {
		int level = -1;
		for (Iterator<DataPathInfo> iter = dpiList.iterator(); iter.hasNext();) {
			DataPathInfo dpi = iter.next();
			if (level == -1) {
				level = dpi.numLevels();
				continue;
			}
			if (level != dpi.numLevels())
				return false;
		}
		return true;
	}

	public static boolean isCollection(String objectTypeStr) {
		int idx = objectTypeStr.indexOf('_');
		int typeCategory = NumberUtils.stringToInt(objectTypeStr
				.substring(0, idx));
		if (typeCategory == 2)
			return false;
		String[] toks = objectTypeStr.substring(idx + 1).split("\\s+");
		return (toks.length > 2 && toks[1].equalsIgnoreCase("col"));
	}

	public static int extractFileType(String objectTypeStr) {
		int idx = objectTypeStr.indexOf('_');
		int typeCategory = NumberUtils.stringToInt(objectTypeStr
				.substring(0, idx));
		TObjectIntHashMap map = new TObjectIntHashMap();
		map.put("nifti", FileInfo.NIFTI);
		map.put("nifti gz", FileInfo.GZIPPED_NIFTI);
		map.put("dicom", FileInfo.DICOM);
		map.put("a_7.5", FileInfo.ANALYZE_7_5);
		map.put("html", FileInfo.HTML);
		map.put("eprime", FileInfo.EPRIME);
		map.put("n", FileInfo.NIFTI);

		String[] toks = objectTypeStr.substring(idx + 1).split("\\s+");
		if (typeCategory == 2) {
			String key = toks[0];
			if (toks.length > 1) {
				key = GenUtils.join(toks, " ", null);
			}
			key = key.toLowerCase();
			return map.get(key);
		} else if (typeCategory == 3) {
			if (toks.length != 3)
				return -1;
			String key = toks[toks.length - 1];
			if (key.equalsIgnoreCase("gz")) {
				key = GenUtils.join(toks, " ", toks.length - 2, toks.length);
			} else if (key.toLowerCase().startsWith("n_")) {
				key = "n";
			}
			key = key.toLowerCase();
			return map.get(key);
		}
		return -1;
	}

	public static int extractStorageType(String objectType) {
		int idx = objectType.indexOf('_');
		int lastIdx = objectType.indexOf(' ');
		String st = objectType.substring(idx + 1, lastIdx);
		if (st.equalsIgnoreCase("srb")) {
			return FileInfo.SRB;
		} else if (st.equalsIgnoreCase("gftp")) {
			return FileInfo.GRIDFTP;
		}
		// default
		return FileInfo.LOCAL;
	}

	/**
	 * prepares a {@link FileBundleConfig} object for batch download.
	 * 
	 * @param esidbList
	 * @param bundleName
	 * @param cacheDir
	 * @param expMap
	 * @return
	 * @throws Exception
	 */
	public static FileBundleConfig prepareFileBundleConfig(
			List<ExpSubjectsIDBInfo> esidbList, String bundleName,
			String cacheDir, Map<String, Experiment> expMap) throws Exception {
		FileBundleConfig fbc = new FileBundleConfig(FileBundleConfig.GZIPPED,
				bundleName, cacheDir);

		for (ExpSubjectsIDBInfo esidb : esidbList) {
			List<SubjectImageDataBundle> sidbList = esidb.getAllSIDBs();
			for (Iterator<SubjectImageDataBundle> it = sidbList.iterator(); it
					.hasNext();) {
				SubjectImageDataBundle sidb = it.next();
				List<DataNode> dnList = sidb.getAllPaths();
				for (DataNode dn : dnList) {
					String expName = esidb.getExpName();
					Assertion.assertNotNull(expName);
					String expKey = sidb.getSiteID() + "_" + expName;
					// get source root dir from the experiment baseURI

					Experiment exp = expMap.get(expKey);
					Assertion.assertNotNull(exp);
					String sourceRootDir = exp.getBaseuri();
					Assertion.assertNotNull(sourceRootDir);

					// TODO no longer used. remove?
					if (Constants.USE_BIRN_URIS) {
						sourceRootDir = BIRNURIUtils.getPath(sourceRootDir);
					}

					FileInfo fi = (FileInfo) dn.getUserObject();
					log.info("DYW dn fi host=" + fi.getGridFtpHost());
					fi.setSourceRootDir(sourceRootDir);
					String path = fi.getPath();
					log.info("DYW 1 path=" + path);
					path = path.replaceAll("\\/\\/+", "\\/");
					if (path.indexOf(sidb.getSubjectID()) == -1) {
						// this fileinfo object is detached from its subject so
						// add it back
						path = sidb.getSubjectID() + "/";
						path += fi.getPath();
					}

					log.info("DYW 2 path=" + path);
					String filePath = path;
					// TODO no longer used. remove?
					if (Constants.USE_BIRN_URIS) {
						filePath = BIRNURIUtils.addScheme(path, fi.getStorageType());
					}
					log.info("DYW 3 path=" + filePath);

					FileInfo nfi = new FileInfo(fi.getGridFtpHost()+ filePath, fi.getFileType(), fi
							.getStorageType(), "", fi.isDir());
					nfi.setSourceRootDir(fi.getSourceRootDir());
					nfi.setFileSize(fi.getFileSize());

					fbc.addFileInfo(nfi);
					break;
				}
			}
		}
		return fbc;
	}

	public static FileBundleConfig prepareFileBundleConfig(
			List<SubjectImageDataBundle> sidbList, String selectedItemsXML,
			String bundleName, String cacheDir, Map<String, Experiment> expMap)
			throws Exception {
		FileBundleConfig fbc = new FileBundleConfig(FileBundleConfig.GZIPPED,
				bundleName, cacheDir);

		Element rootElem = FileUtils.loadXML(new StringReader(selectedItemsXML));
		List<?> itemElems = rootElem.getChildren("item");
		for (Iterator<?> iter = itemElems.iterator(); iter.hasNext();) {
			Element itemElem = (Element) iter.next();
			String item = itemElem.getTextTrim();
			for (Iterator<SubjectImageDataBundle> it = sidbList.iterator(); it
					.hasNext();) {
				SubjectImageDataBundle sidb = it.next();
				DataNode dn = sidb.findByPath(item);
				if (dn != null) {
					String expName = sidb.extractExperimentName(item);
					Assertion.assertNotNull(expName);
					String expKey = sidb.getSiteID() + "_" + expName;
					// get source root dir from the experiment baseURI

					Experiment exp = expMap.get(expKey);
					Assertion.assertNotNull(exp);
					String sourceRootDir = exp.getBaseuri();
					Assertion.assertNotNull(sourceRootDir);
					// TODO not used. remove?
					if (Constants.USE_BIRN_URIS) {
						sourceRootDir = BIRNURIUtils.getPath(sourceRootDir);
					}

					FileInfo fi = (FileInfo) dn.getUserObject();
					fi.setSourceRootDir(sourceRootDir);
					log.info("DYW 4 host=" + fi.getGridFtpHost());
					String path = fi.getPath();
					log.info("DYW 4 path=" + path);
					path = path.replaceAll("\\/\\/+", "\\/");
					if(fi.getStorageType()!=LOCAL){
						if (path.indexOf(sidb.getSubjectID()) == -1) {
							// this fileinfo object is detached from its subject so
							// add it back
							path = sidb.getSubjectID() + "/";
							path += fi.getPath();
						}						
					}
					
					String filePath = path;
					
					// TODO not used. remove?
					if (Constants.USE_BIRN_URIS) {
						filePath = BIRNURIUtils.addScheme(path, fi.getStorageType());
					}
					log.info("DYW 5 path=" + filePath);
					FileInfo nfi;
					if(fi.getStorageType() == GRIDFTP){
						nfi = new FileInfo(fi.getGridFtpHost()+ filePath, fi.getFileType(), fi
								.getStorageType(), "", fi.isDir());	
					}else{
						nfi = new FileInfo(filePath, fi.getFileType(), fi
								.getStorageType(), "", fi.isDir());
					}					
					nfi.setSourceRootDir(fi.getSourceRootDir());
					nfi.setFileSize(fi.getFileSize());
					
					nfi.setRootPath(sourceRootDir); //fix files and folder in different folder in downloaded tar ball

					fbc.addFileInfo(nfi);
					break;
				}
			}
		}
		return fbc;
	}

	public static FileBundleConfig prepareFileBundleConfig(
			List<SubjectImageDataBundle> sidbList, String selectedItemsXML,
			String bundleName, String cacheDir, Map<String, Experiment> expMap, String format)
			throws Exception {
		FileBundleConfig fbc = new FileBundleConfig(FileBundleConfig.GZIPPED,
				bundleName, cacheDir);

		Element rootElem = FileUtils.loadXML(new StringReader(selectedItemsXML));
		List<?> itemElems = rootElem.getChildren("item");
		for (Iterator<?> iter = itemElems.iterator(); iter.hasNext();) {
			Element itemElem = (Element) iter.next();
			String item = itemElem.getTextTrim();
			for (Iterator<SubjectImageDataBundle> it = sidbList.iterator(); it
					.hasNext();) {
				SubjectImageDataBundle sidb = it.next();
				DataNode dn = sidb.findByPath(item);
				if (dn != null) {
					String expName = sidb.extractExperimentName(item);
					Assertion.assertNotNull(expName);
					String expKey = sidb.getSiteID() + "_" + expName;
					// get source root dir from the experiment baseURI

					Experiment exp = expMap.get(expKey);
					Assertion.assertNotNull(exp);
					String sourceRootDir = exp.getBaseuri();
					Assertion.assertNotNull(sourceRootDir);
					// TODO not used. remove?
					if (Constants.USE_BIRN_URIS) {
						sourceRootDir = BIRNURIUtils.getPath(sourceRootDir);
					}

					FileInfo fi = (FileInfo) dn.getUserObject();
					fi.setSourceRootDir(sourceRootDir);
					log.info("DYW 4 host=" + fi.getGridFtpHost());
					String path = fi.getPath();
					log.info("DYW 4 path=" + path);
					path = path.replaceAll("\\/\\/+", "\\/");
					if (path.indexOf(sidb.getSubjectID()) == -1) {
						// this fileinfo object is detached from its subject so
						// add it back
						path = sidb.getSubjectID() + "/";
						path += fi.getPath();
					}

					String filePath = path;
					
					// TODO not used. remove?
					if (Constants.USE_BIRN_URIS) {
						filePath = BIRNURIUtils.addScheme(path, fi.getStorageType());
					}
					log.info("DYW 5 path=" + filePath);
					FileInfo nfi = new FileInfo(fi.getGridFtpHost()+ filePath, fi.getFileType(), fi
							.getStorageType(), "", fi.isDir());
					nfi.setSourceRootDir(fi.getSourceRootDir());
					nfi.setFileSize(fi.getFileSize());

					fbc.addFileInfo(nfi);
					break;
				}
			}
		}
		return fbc;
	}

	public static List<SubjectImageDataBundle> prepareAssessmentBundle(
			List<SubjectImageDataBundle> sidbList, String selectedItemsXML,
			Map<String, String> asNameMap) throws Exception {
		Set<SubjectImageDataBundle> caSIDBSet = new HashSet<SubjectImageDataBundle>();
		Element rootElem = FileUtils.loadXML(new StringReader(selectedItemsXML));
		List<?> caElems = rootElem.getChildren("ca");
		for (Iterator<?> iter = caElems.iterator(); iter.hasNext();) {
			Element caElem = (Element) iter.next();
			String item = caElem.getTextTrim();
			String[] parts = item.split("\\/");
			for (Iterator<SubjectImageDataBundle> it = sidbList.iterator(); it
					.hasNext();) {
				SubjectImageDataBundle sidb = it.next();
				String asName = parts[parts.length - 1];
				String subjectId = parts[0];
				if(sidb.getSubjectID().equals(subjectId)){
					DataNode dn = sidb.findByTitle(asName);				
					if (dn != null) {
						caSIDBSet.add(sidb);
						asNameMap.put(asName, asName);
					}	
				}				
			}
		}
		List<SubjectImageDataBundle> caSIDBList = new ArrayList<SubjectImageDataBundle>(
				caSIDBSet);
		return caSIDBList;
	}

	static class DataPathInfo {
		RawDataObject rdo;
		String[] parts;
		String identifier;
		String dataURI;

		public DataPathInfo(RawDataObject rdo, String[] parts, String identifier,
				String dataURI) {
			this.rdo = rdo;
			this.parts = parts;
			this.identifier = identifier;
			this.dataURI = dataURI;
		}

		public String getIdentifier() {
			return identifier;
		}

		public int numLevels() {
			return parts.length;
		}

	}// ;

}// ;
