package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.CacheUtils;
import clinical.cache.DBChangeEvent;
import clinical.cache.DBChangeListener;
import clinical.cache.DBChangeSupport;
import clinical.cache.NullObject;
import clinical.cache.TableChange;
import clinical.server.VisitInfo;
import clinical.server.dao.AnimalspeciesDAO;
import clinical.server.dao.ArgumenttypeDAO;
import clinical.server.dao.AssessmentDAO;
import clinical.server.dao.AssessmentscorecodeDAO;
import clinical.server.dao.CollectionequipmentDAO;
import clinical.server.dao.DatabaseuserDAO;
import clinical.server.dao.DataclassificationDAO;
import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.ExpsegmentDAO;
import clinical.server.dao.ExpstudyDAO;
import clinical.server.dao.MeasurementsystemDAO;
import clinical.server.dao.MeasurementtypeDAO;
import clinical.server.dao.MeasurementunitDAO;
import clinical.server.dao.OntologyconceptDAO;
import clinical.server.dao.OntologysourceDAO;
import clinical.server.dao.ProtocolDAO;
import clinical.server.dao.ResearchgroupDAO;
import clinical.server.dao.ResearchgrouptypeDAO;
import clinical.server.dao.SecurityclassificationDAO;
import clinical.server.dao.SiteDAO;
import clinical.server.dao.SubjexperimentDAO;
import clinical.server.dao.TableidDAO;
import clinical.server.dao.UserclassDAO;
import clinical.server.dao.UserstatusDAO;
import clinical.server.dao.VisittypeDAO;
import clinical.server.vo.Animalspecies;
import clinical.server.vo.Argumenttype;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscorecode;
import clinical.server.vo.Collectionequipment;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Dataclassification;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Expstudy;
import clinical.server.vo.Measurementsystem;
import clinical.server.vo.Measurementtype;
import clinical.server.vo.Measurementunit;
import clinical.server.vo.Ontologyconcept;
import clinical.server.vo.Ontologysource;
import clinical.server.vo.Protocol;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Researchgrouptype;
import clinical.server.vo.Securityclassification;
import clinical.server.vo.Site;
import clinical.server.vo.Subjexperiment;
import clinical.server.vo.Tableid;
import clinical.server.vo.Userclass;
import clinical.server.vo.Userstatus;
import clinical.server.vo.Visittype;
import clinical.web.DAOFactory;
import clinical.web.ISubjectVisitHandler;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.exception.DBCacheException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.helpers.ExpSubjInfo;

/**
 * Implementation of <code>DBCache</code> interface for retrieving and caching
 * nearly static data from the database.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: DBCache.java 873 2015-01-07 20:55:20Z jinranc $
 */
public class DBCache implements IDBCache, DBChangeListener {
	protected IDBPoolService dbPoolService;
	protected String dbID;
	protected Log log = LogFactory.getLog("clinical");
	protected static Map<String, DBCache> instanceMap = new HashMap<String, DBCache>(
			7);
	protected CacheManager cacheMan;

	protected DBCache(String dbID) {
		dbPoolService = MinimalServiceFactory.getPoolService(dbID);
		this.dbID = dbID;
		log.info("DBCache for dbID:" + dbID + " - dbPoolService: "
				+ dbPoolService);
		cacheMan = CacheManager.getInstance();
	}

	public static synchronized DBCache getInstance(String dbID) {
		DBCache dbCache = instanceMap.get(dbID);
		if (dbCache == null) {
			dbCache = new DBCache(dbID);
			instanceMap.put(dbID, dbCache);
			// register itself as a db change listener
			DBChangeSupport.getInstance().addDBChangeListener(dbCache);
		}
		return dbCache;
	}

	@SuppressWarnings("unchecked")
	public synchronized Databaseuser getDatabaseUser(UserInfo ui,
			String userName, String userClass) throws DBCacheException {
		String cacheKey = getKey("databaseuser");

		String key = userName.toUpperCase() + "_" + userClass;

		Object o = getFromCache(cacheKey);
		if (o != null) {
			Map<String, Databaseuser> dbUserMap = (Map<String, Databaseuser>) o;
			Databaseuser dbUser = dbUserMap.get(key);
			if (dbUser != null)
				return dbUser;
		}
		Connection con = null;
		try {
			log.info(">> looking for database user named "
					+ userName.toUpperCase());
			con = dbPoolService.getConnection(ui.getName());

			List<Userclass> ucList = getUserClasses(ui, false);
			Userclass theUC = null;
			Map<BigDecimal, Userclass> ucMap = new HashMap<BigDecimal, Userclass>(
					17);
			for (Userclass uc : ucList) {
				ucMap.put(uc.getUniqueid(), uc);
				if (uc.getName().equals(userClass))
					theUC = uc;
			}

			assert (theUC != null);

			DatabaseuserDAO dao = DAOFactory.createDatabaseuserDAO(dbID);
			List<Databaseuser> dbUsers = dao.find(con, new Databaseuser());
			Map<String, Databaseuser> dbUserMap = new HashMap<String, Databaseuser>(
					17);
			for (Databaseuser aDBUser : dbUsers) {
				Userclass uc = ucMap.get(aDBUser.getUserclass());
				if (uc == null) {
					throw new Exception("No user class for dbuser '"
							+ aDBUser.getName() + "' with userClass:"
							+ aDBUser.getUserclass());
				}
				String aKey = aDBUser.getName().toUpperCase() + "_" + uc.getName();
				dbUserMap.put(aKey, aDBUser);
			}
			dbUserMap = Collections.unmodifiableMap(dbUserMap);
			put2Cache(cacheKey, dbUserMap);
			return dbUserMap.get(key);

		} catch (Exception e) {
			log.error("getDatabaseUser", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized Tableid getTableID(UserInfo ui, String tableName)
			throws DBCacheException {
		String cacheKey = getKey("tableid");
		tableName = tableName.toUpperCase();
		Object o = getFromCache(cacheKey);
		if (o != null) {
			Map<String, Tableid> tableIDMap = (Map<String, Tableid>) o;
			Tableid tid = tableIDMap.get(tableName);
			if (tid != null)
				return tid;
		}

		Connection con = null;
		Map<String, Tableid> tableIDMap = new HashMap<String, Tableid>();
		Tableid tid = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			Tableid criteria = new Tableid();
			TableidDAO dao = DAOFactory.createTableidDAO(dbID);
			List<Tableid> tids = dao.find(con, criteria);
			for (Tableid ti : tids) {
				if (tableIDMap.get(ti.getTablename()) == null) {
					tableIDMap.put(ti.getTablename(), ti);
				}
				if (tableName.equals(ti.getTablename())) {
					tid = ti;
				}
			}
			tableIDMap = Collections.unmodifiableMap(tableIDMap);
			put2Cache(cacheKey, tableIDMap);
			return tid;
		} catch (Exception e) {
			log.error("getTableID", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Collectionequipment> getCollectionEquipments(
			UserInfo ui, boolean forceRecache) throws DBCacheException {
		String key = getKey("collectionequipment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Collectionequipment>) o;
		}
		List<Collectionequipment> ceList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			CollectionequipmentDAO dao = DAOFactory
					.createCollectionequipmentDAO(dbID);
			ceList = dao.find(con, new Collectionequipment());
			ceList = Collections.unmodifiableList(ceList);
			put2Cache(key, ceList);
			return ceList;
		} catch (Exception e) {
			log.error("getCollectionEquipments", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Researchgrouptype> getResearchGroupTypes(
			UserInfo ui, boolean forceRecache) throws DBCacheException {
		String key = getKey("researchgrouptype");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Researchgrouptype>) o;
		}
		List<Researchgrouptype> rgtList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ResearchgrouptypeDAO dao = DAOFactory.createResearchgrouptypeDAO(dbID);
			rgtList = dao.find(con, new Researchgrouptype());
			rgtList = Collections.unmodifiableList(rgtList);
			put2Cache(key, rgtList);
			return rgtList;
		} catch (Exception e) {
			log.error("getResearchGroupTypes", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Researchgroup> getResearchGroups(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("researchgroup");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Researchgroup>) o;
		}
		List<Researchgroup> rgList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ResearchgroupDAO dao = DAOFactory.createResearchgroupDAO(dbID);

			rgList = dao.find(con, new Researchgroup());
			rgList = Collections.unmodifiableList(rgList);
			put2Cache(key, rgList);
			return rgList;
		} catch (Exception e) {
			log.error("getResearchGroups", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Visittype> getVisitTypes(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("visittype");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Visittype>) o;
		}
		List<Visittype> vtList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			VisittypeDAO dao = DAOFactory.createVisittypeDAO(dbID);
			vtList = dao.find(con, new Visittype());
			vtList = Collections.unmodifiableList(vtList);
			put2Cache(key, vtList);
			return vtList;
		} catch (Exception e) {
			log.error("getVisitTypes", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Userclass> getUserClasses(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("userclass");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Userclass>) o;
		}
		List<Userclass> ucList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			UserclassDAO dao = DAOFactory.createUserclassDAO(dbID);

			ucList = dao.find(con, new Userclass());
			ucList = Collections.unmodifiableList(ucList);
			put2Cache(key, ucList);
			return ucList;
		} catch (Exception e) {
			log.error("getUserClasses", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Securityclassification> getSecurityClassifications(
			UserInfo ui, boolean forceRecache) throws DBCacheException {
		String key = getKey("securityclassification");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Securityclassification>) o;
		}
		List<Securityclassification> scList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			SecurityclassificationDAO dao = DAOFactory
					.createSecurityclassificationDAO(dbID);

			scList = dao.find(con, new Securityclassification());
			scList = Collections.unmodifiableList(scList);
			put2Cache(key, scList);
			return scList;
		} catch (Exception e) {
			log.error("getSecurityClassifications", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Assessment> getAssessments(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("assessment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Assessment>) o;
		}
		List<Assessment> asList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			AssessmentDAO dao = DAOFactory.createAssessmentDAO(dbID);

			asList = dao.find(con, new Assessment());
			asList = Collections.unmodifiableList(asList);
			put2Cache(key, asList);
			return asList;
		} catch (Exception e) {
			log.error("getAssessments", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Protocol> getProtocols(UserInfo ui, boolean forceRecache)
			throws DBCacheException {
		String key = getKey("protocol");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Protocol>) o;
		}
		List<Protocol> prList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ProtocolDAO dao = DAOFactory.createProtocolDAO(dbID);
			prList = dao.find(con, new Protocol());
			prList = Collections.unmodifiableList(prList);
			put2Cache(key, prList);
			return prList;
		} catch (Exception e) {
			log.error("getProtocols", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Site> getSites(UserInfo ui, boolean forceRecache)
			throws DBCacheException {
		String key = getKey("site");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Site>) o;
		}
		List<Site> stList = null;

		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			SiteDAO dao = DAOFactory.createSiteDAO(dbID);

			stList = dao.find(con, new Site());
			stList = Collections.unmodifiableList(stList);
			put2Cache(key, stList);
			return stList;
		} catch (Exception e) {
			log.error("getSites", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected String getKey(String prefix) {
		StringBuilder sb = new StringBuilder();
		sb.append(prefix).append(':').append(this.dbID);
		return sb.toString();
	}

	public synchronized Map<String, Experiment> getExperimentMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		List<Experiment> expList = getExperiments(ui, forceRecache);
		Map<String, Experiment> expMap = new HashMap<String, Experiment>(31);
		for (Experiment exp : expList)
			expMap.put(exp.getName(), exp);
		return expMap;
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Experiment> getExperiments(UserInfo ui,
			boolean forceRecache) throws DBCacheException {

		String key = getKey("experiment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Experiment>) o;
		}
		List<Experiment> expList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ExperimentDAO dao = DAOFactory.createExperimentDAO(dbID);

			expList = dao.find(con, new Experiment());
			expList = Collections.unmodifiableList(expList);
			put2Cache(key, expList);
			return expList;
		} catch (Exception e) {
			log.error("getExperiments", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public synchronized List<Experiment> getAllExperiments(UserInfo ui,
			boolean forceRecache) throws DBCacheException {

		String key = getKey("experiment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Experiment>) o;
		}
		List<Experiment> expList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ExperimentDAO dao = DAOFactory.createExperimentDAO(dbID);

			expList = dao.find(con, new Experiment());
			//expList = Collections.unmodifiableList(expList);
			put2Cache(key, expList);
			return expList;
		} catch (Exception e) {
			log.error("getExperiments", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Experiment> getLocalStoredExperiments(UserInfo ui,
			boolean forceRecache) throws DBCacheException {

		String key = getKey("localStoredExperiment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Experiment>) o;
		}
		List<Experiment> expList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ExperimentDAO dao = DAOFactory.createExperimentDAO(dbID);
			Experiment bean = new Experiment();
			bean.setStoragetype("local");
			expList = dao.find(con, bean);
			expList = Collections.unmodifiableList(expList);
			put2Cache(key, expList);
			return expList;
		} catch (Exception e) {
			log.error("getExperiments", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	
	
	@SuppressWarnings("unchecked")
	public synchronized List<Dataclassification> getDataClassications(
			UserInfo ui, boolean forceRecache) throws DBCacheException {
		String key = getKey("dataclassification");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Dataclassification>) o;
		}
		List<Dataclassification> dcList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			DataclassificationDAO dao = DAOFactory
					.createDataclassificationDAO(dbID);

			dcList = dao.find(con, new Dataclassification());
			dcList = Collections.unmodifiableList(dcList);
			put2Cache(key, dcList);
			return dcList;
		} catch (Exception e) {
			log.error("getDataClassifications", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized List<Userstatus> getUserStatusList(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("userstatus");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Userstatus>) o;
		}
		List<Userstatus> usList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			UserstatusDAO dao = DAOFactory.createUserstatusDAO(dbID);
			usList = dao.find(con, new Userstatus());
			usList = Collections.unmodifiableList(usList);
			put2Cache(key, usList);
			return usList;
		} catch (Exception e) {
			log.error("getUserStatusList", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Animalspecies> getAnimalSpecies(UserInfo ui, boolean forceRecache)
			throws DBCacheException {
		List<Animalspecies> asList = null;
		String key = getKey("animalspecies");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Animalspecies>) o;
		}

		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			AnimalspeciesDAO dao = DAOFactory.createAnimalspeciesDAO(dbID);
			asList = dao.find(con, new Animalspecies());
			asList = Collections.unmodifiableList(asList);
			put2Cache(key, asList);
			return asList;
		} catch (Exception e) {
			log.error("getAnimalSpecies", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, List<Assessmentscorecode>> getScoreCodeMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("assessmentscorecode");
		// cache is stale so force populating from the database
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, List<Assessmentscorecode>>) o;
		}

		Connection con = null;
		try {
			Map<String, List<Assessmentscorecode>> scoreCodeMap = new HashMap<String, List<Assessmentscorecode>>(
					19);
			con = dbPoolService.getConnection(ui.getName());
			AssessmentscorecodeDAO dao = DAOFactory
					.createAssessmentscorecodeDAO(dbID);
			List<Assessmentscorecode> scoreCodes = null;
			scoreCodes = dao.find(con, new Assessmentscorecode());
			for (Assessmentscorecode asc : scoreCodes) {
				String aKey = asc.getAssessmentid().toString() + "_"
						+ asc.getScorename();
				List<Assessmentscorecode> scList = scoreCodeMap.get(aKey);
				if (scList == null) {
					scoreCodeMap.put(key,
							scList = new LinkedList<Assessmentscorecode>());
				}
				scList.add(asc);
			}
			scoreCodeMap = Collections.unmodifiableMap(scoreCodeMap);
			put2Cache(key, scoreCodeMap);
			return scoreCodeMap;
		} catch (Exception e) {
			log.error("getScoreMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, Measurementsystem> getMeasurementSystemMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("measurementsystem");
		List<Measurementsystem> msList = null;
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Measurementsystem>) o;
		}

		Connection con = null;
		Map<String, Measurementsystem> measurementSystemMap = new HashMap<String, Measurementsystem>(
				7);
		try {
			con = dbPoolService.getConnection(ui.getName());
			MeasurementsystemDAO dao = DAOFactory.createMeasurementsystemDAO(dbID);

			msList = dao.find(con, new Measurementsystem());
			for (Measurementsystem ms : msList) {
				measurementSystemMap.put(ms.getName(), ms);
			}
			measurementSystemMap = Collections
					.unmodifiableMap(measurementSystemMap);
			put2Cache(key, measurementSystemMap);
			return measurementSystemMap;

		} catch (Exception e) {
			log.error("getMeasurementSystemMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, Measurementtype> getMeasurementTypeMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("measurementtype");
		List<Measurementtype> mtList = null;
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Measurementtype>) o;
		}
		Map<String, Measurementtype> measurementTypeMap = new HashMap<String, Measurementtype>(
				31);
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			MeasurementtypeDAO dao = DAOFactory.createMeasurementtypeDAO(dbID);

			mtList = dao.find(con, new Measurementtype());
			for (Measurementtype mt : mtList) {
				measurementTypeMap.put(mt.getName(), mt);
			}
			measurementTypeMap = Collections.unmodifiableMap(measurementTypeMap);
			put2Cache(key, measurementTypeMap);
			return measurementTypeMap;

		} catch (Exception e) {
			log.error("getMeasurementTypeMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, Measurementunit> getMeasurementUnitMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("measurementunit");
		List<Measurementunit> muList = null;
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Measurementunit>) o;
		}
		Map<String, Measurementunit> measurementUnitMap = new HashMap<String, Measurementunit>();
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			MeasurementunitDAO dao = DAOFactory.createMeasurementunitDAO(dbID);

			muList = dao.find(con, new Measurementunit());
			for (Measurementunit mu : muList) {
				measurementUnitMap.put(mu.getUnit(), mu);
			}
			measurementUnitMap = Collections.unmodifiableMap(measurementUnitMap);
			put2Cache(key, measurementUnitMap);
			return measurementUnitMap;

		} catch (Exception e) {
			log.error("getMeasurementUnitMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, Ontologysource> getOntologySourceMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("ontologysource");
		List<Ontologysource> osList = null;
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Ontologysource>) o;
		}
		Map<String, Ontologysource> ontSourceMap = new HashMap<String, Ontologysource>();

		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			OntologysourceDAO dao = DAOFactory.createOntologysourceDAO(dbID);

			osList = dao.find(con, new Ontologysource());
			for (Ontologysource os : osList) {
				ontSourceMap.put(os.getOntologysource(), os);
			}
			ontSourceMap = Collections.unmodifiableMap(ontSourceMap);
			put2Cache(key, ontSourceMap);
			return ontSourceMap;

		} catch (Exception e) {
			log.error("getOntologySourceMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, Ontologyconcept> getOntologyConceptMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException {
		String key = getKey("ontologyconcept");
		List<Ontologyconcept> ocList = null;
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Ontologyconcept>) o;
		}
		Map<String, Ontologyconcept> ontConceptMap = new HashMap<String, Ontologyconcept>();

		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			OntologyconceptDAO dao = DAOFactory.createOntologyconceptDAO(dbID);

			ocList = dao.find(con, new Ontologyconcept());
			for (Ontologyconcept oc : ocList) {
				ontConceptMap.put(oc.getConceptid(), oc);
			}
			ontConceptMap = Collections.unmodifiableMap(ontConceptMap);
			put2Cache(key, ontConceptMap);
			return ontConceptMap;

		} catch (Exception e) {
			log.error("getOntologyConceptMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	@SuppressWarnings("unchecked")
	public synchronized Map<String, Argumenttype> getArgumentTypeMap(
			UserInfo ui, boolean forceRecache) throws DBCacheException {
		String key = getKey("argumenttype");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (Map<String, Argumenttype>) o;
		}

		List<Argumenttype> argList = null;
		Map<String, Argumenttype> argTypeMap = new HashMap<String, Argumenttype>(
				17);
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ArgumenttypeDAO dao = DAOFactory.createArgumenttypeDAO(dbID);

			argList = dao.find(con, new Argumenttype());
			for (Argumenttype arg : argList) {
				argTypeMap.put(arg.getArgumenttype(), arg);
			}
			argTypeMap = Collections.unmodifiableMap(argTypeMap);
			put2Cache(key, argTypeMap);
			return argTypeMap;

		} catch (Exception e) {
			log.error("getArgumentTypeMap", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	/**
	 * Releases the named user connection back to its corresponding connection
	 * pool.
	 * 
	 * @param con
	 *           a JDBC connection
	 * @param ui
	 *           UserInfo object identifying the named user
	 */

	protected void releaseConnection(Connection con, UserInfo ui) {
		if (con == null)
			return;
		try {
			dbPoolService.releaseConnection(ui.getName(), con);
		} catch (DBPoolServiceException x) {
			log.error("Cannot release connection for user " + ui.getName(), x);
		}
	}

	protected Object getFromCache(String key) {
		Cache cache = cacheMan.getCache("remote");
		Element el = cache.get(key);
		if (el == null || el.getObjectValue() == NullObject.getInstance())
			return null;
		return el.getObjectValue();
	}

	protected void put2Cache(String key, Object value) {
		Cache cache = cacheMan.getCache("remote");
		Element el = new Element(key, value);
		cache.put(el);
	}

	protected void dropFromCache(String key) {
		Cache cache = cacheMan.getCache("remote");
		cache.remove(key);
	}

	@Override
	public void dbContentChanged(DBChangeEvent event) {
		String siteID = event.getDbChange().getSiteID();
		if (siteID == null)
			return;
		String siteDBID = CacheUtils.getDBID4SiteID(siteID);
		if (siteDBID == null || !siteDBID.equals(this.dbID)) {
			return;
		}

		List<TableChange> tcList = event.getDbChange().getChanges();
		for (TableChange tc : tcList) {
			String key = getKey(tc.getTableName());
			dropFromCache(key);
		}
	}
	
	@Override
	public List<Expsegment> getExpsegments(UserInfo ui, boolean forceRecache)
			throws DBCacheException {
		String key = getKey("expsegment");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<Expsegment>) o;
		}
		List<Expsegment> expSegList = null;
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(dbID);
			expSegList = dao.find(con, new Expsegment());
			expSegList = Collections.unmodifiableList(expSegList);
			put2Cache(key, expSegList);
			return expSegList;
		} catch (Exception e) {
			log.error("getExpsegment", e);
			throw new DBCacheException(e.getMessage());
		} finally {
			releaseConnection(con, ui);
		}	
	}
	
	/*
	 * Retrieve all subject and visit information under each experiment
	 */
	@SuppressWarnings("unchecked")
	public synchronized List<ExpSubjInfo> getAllExpSubjInfo(UserInfo ui, boolean forceRecache)
		throws DBCacheException {
		String key = getKey("allexpsubjectinfo");
		if (forceRecache) {
			dropFromCache(key);
		} else {
			Object o = getFromCache(key);
			if (o != null)
				return (List<ExpSubjInfo>) o;
		}		
		
		Connection con = null;
		List<ExpSubjInfo> expSubjInfos = new ArrayList<ExpSubjInfo>();
		
		try {			
			con = dbPoolService.getConnection(ui.getName());
			ISubjectVisitHandler handler = null;
			handler = ServiceFactory.getSubjectVisitHandler();
			
			List<Experiment> exps = getAllExperiments(ui, true);
			//sort exps asc
			Collections.sort(exps, new Comparator<Experiment>(){
				public int compare(Experiment e1, Experiment e2){
					return e1.getName().compareToIgnoreCase(e2.getName());
				}				
			});				
			
			for(Experiment exp : exps)
			{
				ExpSubjInfo expSubj = new ExpSubjInfo();
				expSubj.setExperimentId(exp.getUniqueid().toString());
				expSubj.setExperimentName(exp.getName());
								
				SubjexperimentDAO dao = DAOFactory.createSubjexperimentDAO(dbID);
				Subjexperiment cr = new Subjexperiment();
				cr.setNcExperimentUniqueid(exp.getUniqueid());
	
				List<Subjexperiment> subjExps = dao.find(con, cr);
				Collections.sort(subjExps, new Comparator<Subjexperiment>(){
					public int compare(Subjexperiment s1, Subjexperiment s2){
						return s1.getSubjectid().compareTo(s2.getSubjectid());
					}
				});
				
				for(Subjexperiment subjExp : subjExps){					
					List<VisitInfo> visitInfos = handler.getVisitInfosOnExp(dbID, con, subjExp.getSubjectid(), exp.getUniqueid());
					subjExp.setVisitInfos(visitInfos);
					
					//set study in visitinfo
					for(VisitInfo vi : visitInfos){
						ExpstudyDAO expstudyDao = DAOFactory.createExpstudyDAO(dbID);
						Expstudy bean = new Expstudy();
						bean.setComponentid(vi.getVisit().getComponentid());
						bean.setExperimentid(subjExp.getNcExperimentUniqueid());
						bean.setSubjectid(subjExp.getSubjectid());
						List<Expstudy> studies = expstudyDao.find(con, bean);
						vi.setStudies(studies);
					}
				}
				
				expSubj.setSubjExps(subjExps);				
				expSubjInfos.add(expSubj);		
			}
			put2Cache(key, expSubjInfos);
			return expSubjInfos;
		} catch (Exception x) {			
			log.error("Error in getAllExpSubjInfo", x);
			throw new DBCacheException(x.getMessage());
		}finally{
			releaseConnection(con, ui);	
		}		
	}

}
