package clinical.web.services;

import imagelib.DICOM2JPEGConverter;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.RawdataDAO;
import clinical.server.vo.Experiment;
import clinical.server.vo.Rawdata;
import clinical.utils.Executor;
import clinical.utils.GenUtils;
import clinical.web.DAOFactory;
import clinical.web.IAppConfigService;
import clinical.web.ImageHandler;
import clinical.web.SRBUtils;
import clinical.web.exception.ImageHandlerException;
import edu.sdsc.grid.io.srb.SRBAccount;
import edu.sdsc.grid.io.srb.SRBFile;
import edu.sdsc.grid.io.srb.SRBFileSystem;

/**
 * Implementation of the session facade interface ImageHandler.
 * 
 * @see clinical.web.ImageHandler
 * 
 * @author I. Burak Ozyurt
 * @version Id
 */
public class ImageHandlerImpl implements ImageHandler {
	private String userInfoFile;
	private DICOM2JPEGConverter converter;
	private Log log = LogFactory.getLog("image");
	protected static ImageHandlerImpl instance;

	protected ImageHandlerImpl(String userInfoFile) {
		this.userInfoFile = userInfoFile;
		converter = new DICOM2JPEGConverter();
	}

	public static synchronized ImageHandlerImpl getInstance(String userInfoFile) {
		if (instance == null) {
			instance = new ImageHandlerImpl(userInfoFile);
		}
		return instance;
	}

	public List<Rawdata> getRawDataForSubject(String dbID, Connection con,
			String subjectID, BigDecimal visitID, BigDecimal experimentID,
			BigDecimal segmentID) throws ImageHandlerException {
		Rawdata criteria = new Rawdata();

		criteria.setSubjectid(subjectID);
		criteria.setComponentid(visitID);
		criteria.setNcExperimentUniqueid(experimentID);
		if (segmentID != null) {
			criteria.setSegmentid(segmentID);
		}
		log.info("subjectID=" + subjectID + " visitID=" + visitID
				+ " experimentID=" + experimentID + " segmentID=" + segmentID);
		RawdataDAO dao = DAOFactory.createRawdataDAO(dbID);
		try {
			List<Rawdata> rawDatas = dao.find(con, criteria);
			return rawDatas;
		} catch (Exception x) {
			log.error("getRawDataForSubject", x);
			throw new ImageHandlerException(x.getMessage());
		}
	}

	public String convertToJpeg(String subjectID, String srbSeries,
			String dcmFile, String outPath) throws ImageHandlerException {
		SRBAccount account = null;
		SRBFileSystem sfs = null;
		SRBFile srbFile = null;
		try {
			String base = GenUtils.getBasePart(dcmFile);
			String jpegFilename = subjectID + "_" + base + ".jpg";
			String jpegPath = outPath + File.separator + jpegFilename;
			File f = new File(jpegPath);
			if (f.exists())
				return jpegFilename;

			account = new SRBAccount(userInfoFile);

			String outFile = outPath + File.separator + dcmFile;
			sfs = new SRBFileSystem(account);
			srbFile = new SRBFile(sfs, srbSeries);
			SRBUtils.loadFile(new SRBFile(srbFile, dcmFile), outFile);

			converter.convert(outFile, jpegPath);
			new File(outFile).delete();

			return jpegFilename;
		} catch (IOException x) {
			log.error("convertToJpeg", x);
			throw new ImageHandlerException(x.getMessage());
		} finally {
			if (sfs != null)
				try {
					sfs.close();
				} catch (Exception x) {
				}
		}
	}

	protected String prepareScript(IAppConfigService configService, String scriptName) {
		String scriptPath = configService.getParamValue("script_path");
		StringBuffer buf = new StringBuffer(96);
		if (scriptPath != null) {
			buf.append(scriptPath).append(File.separator).append(scriptName);
		} else
			buf.append(scriptName);
		return buf.toString();
	}

	public boolean doCacheCleanup(String cacheRoot, IAppConfigService configService,
			long maxAgeSecs, long maxSize, boolean silent)
			throws ImageHandlerException {
		Executor executor = prepareExecutor(configService);
		String args = prepareScript(configService, "cleanup.pl") + " " + cacheRoot
				+ " " + maxAgeSecs;
		if (maxSize > 0)
			args += " " + maxSize;
		try {
			int rc = executor.execute(args);
			return (rc == 0);
		} catch (Exception x) {
			if (!silent) {
				log.error("doCacheCleanup", x);
				throw new ImageHandlerException(x.getMessage());
			} else {
				return false;
			}
		}
	}

	public boolean lockImageSeries(String cacheRoot, String srbSeries,
			IAppConfigService configService) throws ImageHandlerException {
		Executor executor = prepareExecutor(configService);
		String containerName = makeContainerName(srbSeries);
		String args = prepareScript(configService, "lock_unlock.pl") + " " + cacheRoot
				+ " " + containerName + " get_rl";
		try {
			int rc = executor.execute(args);
			return (rc == 0);
		} catch (Exception x) {
			log.error("lockImageSeries", x);
			throw new ImageHandlerException(x.getMessage());
		}
	}

	public boolean unlockImageSeries(String cacheRoot, String srbSeries,
			IAppConfigService configService) throws ImageHandlerException {
		Executor executor = prepareExecutor(configService);
		String containerName = makeContainerName(srbSeries);
		String args = prepareScript(configService, "lock_unlock.pl") + " " + cacheRoot
				+ " " + containerName + " rel_rl";
		try {
			int rc = executor.execute(args);
			return (rc == 0);
		} catch (Exception x) {
			log.error("unlockImageSeries", x);
			throw new ImageHandlerException(x.getMessage());
		}
	}

	public static String preparePath(String rootPath, String filename) {
		StringTokenizer stok = new StringTokenizer(rootPath, "/");
		StringBuffer buf = new StringBuffer();
		if (stok.hasMoreTokens()) {
			while (stok.hasMoreTokens()) {
				buf.append(stok.nextToken());
				if (stok.hasMoreTokens())
					buf.append(File.separator);
			}
		} else {
			buf.append(rootPath);
		}
		buf.append(File.separator).append(filename);
		return buf.toString();
	}

	protected Executor prepareExecutor(IAppConfigService configService) {
		String home = configService.getParamValue("proc.env.home");
		String path = configService.getParamValue("proc.env.path");
		String dcmPath = configService.getParamValue("proc.env.dcmpath");
		String pwd = configService.getParamValue("proc.env.pwd");
		String perl5lib = configService.getParamValue("proc.env.perl5lib");
		String perlExec = configService.getParamValue("perl_executable");

		Executor executor = new Executor(perlExec, true);
		executor.addEnvParam("PATH", path);
		executor.addEnvParam("HOME", home);
		executor.addEnvParam("DCMDICTPATH", dcmPath);
		executor.addEnvParam("PERL5LIB", perl5lib);
		executor.addEnvParam("PWD", pwd);

		return executor;
	}

	public String retrieveMRISeriesForSubject(String cacheRoot,
			String srbSeries, IAppConfigService configService) throws ImageHandlerException {
		String home = configService.getParamValue("proc.env.home");
		String path = configService.getParamValue("proc.env.path");
		String dcmPath = configService.getParamValue("proc.env.dcmpath");
		String pwd = configService.getParamValue("proc.env.pwd");
		String perl5lib = configService.getParamValue("proc.env.perl5lib");

		Executor executor = new Executor(prepareScript(configService,
				"mri_converter.pl"));
		executor.addEnvParam("PATH", path);
		executor.addEnvParam("HOME", home);
		executor.addEnvParam("DCMDICTPATH", dcmPath);
		executor.addEnvParam("PERL5LIB", perl5lib);
		executor.addEnvParam("PWD", pwd);

		executor = prepareExecutor(configService);
		String tarFile = null;
		String arg = prepareScript(configService, "mri_converter.pl") + " " + cacheRoot
				+ " " + srbSeries;
		int rc = 0;
		try {
			rc = executor.execute(arg);
			if (rc == 0) {
				String output = executor.getOutput();
				StringTokenizer stok = new StringTokenizer(output, "\n");
				while (stok.hasMoreTokens()) {
					String line = stok.nextToken();
					int idx;
					if ((idx = line.indexOf("saved tar file:")) != -1) {
						tarFile = line.substring(
								idx + "saved tar file:".length()).trim();
						return tarFile;
					}
				}
			}
		} catch (Exception x) {
			x.printStackTrace();
			throw new ImageHandlerException(x.getMessage());
		}
		return null;
	}

	public static String makeContainerName(String srbSeries) {
		if (srbSeries.indexOf("__") != -1) {
			StringTokenizer stok = new StringTokenizer(srbSeries, "/");
			while (stok.hasMoreTokens()) {
				String tok = stok.nextToken();
				if (tok.length() == 12 && Character.isDigit(tok.charAt(0))) {
					return tok;
				}
			}
			return null;
		} else {

			String c = srbSeries.replace('/', '_');
			int idx = c.indexOf("UCSD");
			if (idx != -1) {
				c = c.substring(idx + 5);
			}
			idx = c.indexOf("_RawData");
			if (idx != -1) {
				StringBuffer buf = new StringBuffer(c.length());
				buf.append(c.substring(0, idx));
				buf.append(c.substring(idx + "_RawData".length()));
				c = buf.toString();
			}
			return c;
		}
	}

	public String getExperimentBaseURI(String dbID, Connection con, int expID)
			throws ImageHandlerException {
		ExperimentDAO dao = DAOFactory.createExperimentDAO(dbID);
		Experiment cr = new Experiment();
		cr.setUniqueid(new BigDecimal(String.valueOf(expID)));
		try {
			List<Experiment> expList = dao.find(con, cr);
			if (expList == null || expList.isEmpty()) {
				return null;
			}
			Experiment exp = expList.get(0);
			return exp.getBaseuri();
		} catch (Exception x) {
			throw new ImageHandlerException(x);
		}
	}

	public static void main(String[] args) {
		try {			
			String container = ImageHandlerImpl
					.makeContainerName("/home/BIRN/Human/Clinical/UCSD/000800156805/Visit_001/Study_0001/RawData/005.ser");
			System.out.println("container=" + container);
		} catch (Exception x) {
			x.printStackTrace();
		}
	}
}
