package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.DBChangeNotifySupport;
import clinical.server.dao.AnimalspeciesDAO;
import clinical.server.dao.ExpcomponentDAO;
import clinical.server.dao.ExpsegmentDAO;
import clinical.server.dao.ExpstudyDAO;
import clinical.server.dao.HumansubjectDAO;
import clinical.server.dao.StoredassessmentDAO;
import clinical.server.dao.SubjexperimentDAO;
import clinical.server.vo.Animalspecies;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Expstudy;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Storedassessment;
import clinical.server.vo.Subjexperiment;
import clinical.server.vo.Tableid;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ISequenceHelper;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.SearchCriteria;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.query.TSQLQueryBuilder;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectVisitManagementException;
import clinical.web.vo.Study;
import clinical.web.vo.StudySegment;
import clinical.web.vo.StudySegmentInfo;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import clinical.web.vo.VisitSegment;

/**
 * Implementation of session facade interface
 * <code>ISubjectVisitManagement</code> for subject/visit data management.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SubjectVisitManagementImpl.java,v 1.24 2006/04/19 01:23:27
 *          bozyurt Exp $
 */
public class SubjectVisitManagementImpl extends AbstractServiceImpl implements
		ISubjectVisitManagement {
	/** security service interface */
	protected ISecurityService securityService;
	/**
	 * interface for retrieving and caching nearly static data from the database.
	 */
	protected DBCache dbCache;
	/** unique number generator for primary keys */
	protected ISequenceHelper seqHelper;

	protected String dbType;
	protected String theSiteID;
	private Log log = LogFactory.getLog(SubjectVisitManagementImpl.class);

	public SubjectVisitManagementImpl(String dbID) throws BaseException {
		super(dbID);

		// dbPoolService = ServiceFactory.getPoolService(dbID);
		dbCache = DBCache.getInstance(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(dbID);
		// this.dbID = dbID;
		ISecurityService isec = ServiceFactory.getSecurityService();
		this.dbType = isec.getDBType(dbID);
		this.theSiteID = isec.findSiteIDByDbID(dbID);
	}

	public List<Humansubject> getMatchingSubjects(UserInfo ui, SearchCriteria sc)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			TSQLQueryBuilder qb = new TSQLQueryBuilder(Humansubject.class, sc);
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			String query = qb.buildQuery();

			log.info(" getMatchingSubjects query=" + tsp);
			List<?> subjects = tsp.executeQuery(con, query);

			List<Humansubject> hsList = new ArrayList<Humansubject>(subjects
					.size());
			for (Object item : subjects) {
				Humansubject subject = (Humansubject) item;
				hsList.add(subject);
			}
			return hsList;
		} catch (Exception x) {
			log.error("Error in getMatchingSubjects", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Map<Experiment, List<Subjexperiment>> getExperimentSubjects(
			UserInfo ui) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			List<Experiment> expList = this.dbCache.getExperiments(ui, false);
			SubjexperimentDAO dao = DAOFactory.createSubjexperimentDAO(theDBID);
			Subjexperiment cr = new Subjexperiment();

			List<Subjexperiment> seList = dao.find(con, cr);
			Map<BigDecimal, Experiment> expMap = new HashMap<BigDecimal, Experiment>();
			for (Experiment exp : expList)
				expMap.put(exp.getUniqueid(), exp);
			Map<Experiment, List<Subjexperiment>> map = new HashMap<Experiment, List<Subjexperiment>>(
					17);
			for (Subjexperiment se : seList) {
				Experiment exp = expMap.get(se.getNcExperimentUniqueid());

				List<Subjexperiment> list = map.get(exp);
				if (list == null) {
					list = new ArrayList<Subjexperiment>();
					map.put(exp, list);
				}
				list.add(se);
			}
			return map;
		} catch (Exception x) {
			log.error("Error in getExperimentSubjects", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Subjexperiment> getSubjectsForExperiment(UserInfo ui,
			String expName) throws SubjectVisitManagementException {
		try {
			List<Experiment> expList = this.dbCache.getExperiments(ui, false);
			Experiment theExp = null;
			for (Experiment exp : expList) {
				if (exp.getName().equals(expName)) {
					theExp = exp;
					break;
				}
			}
			if (theExp == null)
				return new ArrayList<Subjexperiment>(0);
			return getSubjectsForExperiment(ui, theExp.getUniqueid().intValue());
		} catch (Exception x) {
			log.error("Error in getSubjectsForExperiment", x);
			if (x instanceof SubjectVisitManagementException)
				throw (SubjectVisitManagementException) x;
			throw new SubjectVisitManagementException(x.getMessage());
		}
	}

	public List<Subjexperiment> getSubjectsForExperiment(UserInfo ui, int expID)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			SubjexperimentDAO dao = DAOFactory.createSubjexperimentDAO(theDBID);
			Subjexperiment cr = new Subjexperiment();
			cr.setNcExperimentUniqueid(new BigDecimal(String.valueOf(expID)));
			List<?> subjExps = dao.find(con, cr);
			List<Subjexperiment> subjExpList = new ArrayList<Subjexperiment>(
					subjExps.size());
			for (Object item : subjExps) {
				Subjexperiment se = (Subjexperiment) item;
				subjExpList.add(se);
			}
			return subjExpList;
		} catch (Exception x) {
			log.error("Error in getSubjectsForExperiment", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Subjexperiment> getMatchingExperimentSubjects(UserInfo ui,
			SearchCriteria sc) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			TSQLQueryBuilder qb = new TSQLQueryBuilder(
					clinical.server.vo.Subjexperiment.class, sc);
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);

			con = pool.getConnection(ui.getName());

			String tsql = qb.buildQuery();
			List<?> expSubjects = tsp.executeQuery(con, tsql);
			List<Subjexperiment> seList = new ArrayList<Subjexperiment>(
					expSubjects.size());
			for (Object o : expSubjects) {
				seList.add((Subjexperiment) o);
			}
			expSubjects = null;
			return seList;
		} catch (Exception x) {
			log.error("Error in getMatchingExperimentSubjects", x);
			throw new SubjectVisitManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Experiment> getAllExperiments(UserInfo ui)
			throws SubjectVisitManagementException {
		try {
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
			return dbCache.getExperiments(ui, false); // use cached info
		} catch (Exception x) {
			log.error("Error in getExperimentsForSubject", x);
			throw new SubjectVisitManagementException(x);
		}
	}

	public List<StudySegmentInfo> findStudyInfosForSubjects(UserInfo ui,
			List<String> subjectIDList) throws SubjectVisitManagementException {
		Connection con = null;
		Statement st = null;
		try {
			con = pool.getConnection(ui.getName());
			if (subjectIDList == null || subjectIDList.isEmpty()) {
				return new LinkedList<StudySegmentInfo>();
			}
			StringBuffer buf = new StringBuffer(200);
			buf
					.append("select b.studyid, b.name, b.subjectid, b.experimentid, b.componentid, ");
			buf.append("a.segmentid  from nc_expsegment a, nc_expstudy b where ");
			buf
					.append("a.studyid = b.studyid and a.nc_experiment_uniqueid = b.experimentid");
			buf
					.append(" and a.componentid = b.componentid and a.subjectid = b.subjectid and ");
			buf.append("a.studyid is not null and a.subjectid in (");
			for (Iterator<String> iter = subjectIDList.iterator(); iter.hasNext();) {
				String subjectID = iter.next();
				buf.append("'").append(subjectID).append("'");
				if (iter.hasNext()) {
					buf.append(',');
				}
			}
			buf.append(')');
			log.info("findStudyInfosForSubjects-query:" + buf.toString());
			st = con.createStatement();
			ResultSet rs = st.executeQuery(buf.toString());
			List<StudySegmentInfo> studyInfos = new LinkedList<StudySegmentInfo>();
			while (rs.next()) {
				StudySegmentInfo sgi = new StudySegmentInfo();
				sgi.setStudyID(rs.getInt(1));
				sgi.setStudyName(rs.getString(2));
				sgi.setSubjectID(rs.getString(3));
				sgi.setExperimentID(rs.getInt(4));
				sgi.setVisitID(rs.getInt(5));
				sgi.setSegmentID(rs.getInt(6));
				studyInfos.add(sgi);
			}
			rs.close();
			return studyInfos;
		} catch (Exception x) {
			log.error("Error in findStudyInfosForSubjects", x);
			throw new SubjectVisitManagementException(x);
		} finally {
			if (st != null) {
				try {
					st.close();
				} catch (Exception x) { /* ignore */
				}
			}
			releaseConnection(con, ui);
		}
	}

	public List<Experiment> getExperimentsForSubject(UserInfo ui,
			String subjectID) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Subjexperiment criteria = new Subjexperiment();
			criteria.setSubjectid(subjectID);
			SubjexperimentDAO dao = DAOFactory.createSubjexperimentDAO(theDBID);
			List<?> subExps = dao.find(con, criteria);
			List<Experiment> experiments = new ArrayList<Experiment>(subExps
					.size());
			if (subExps.isEmpty()) {
				return experiments;
			}

			// get the list of all experiments
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
			// use cached info
			List<Experiment> allExperiments = dbCache.getExperiments(ui, false);

			Map<BigDecimal, Experiment> expMap = new HashMap<BigDecimal, Experiment>();
			for (Experiment exp : allExperiments) {
				expMap.put(exp.getUniqueid(), exp);
			}
			allExperiments = null;
			Map<Experiment, Experiment> seenExpMap = new HashMap<Experiment, Experiment>(
					11);
			for (Object element : subExps) {
				Subjexperiment subExp = (Subjexperiment) element;
				Experiment exp = expMap.get(subExp.getNcExperimentUniqueid());
				// only unique experiments
				if (seenExpMap.get(exp) == null) {
					experiments.add(exp);
					seenExpMap.put(exp, exp);
				}
			}
			return experiments;
		} catch (Exception x) {
			log.error("Error in getExperimentsForSubject", x);
			throw new SubjectVisitManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Researchgroup> getResarchGroupsForSubject(UserInfo ui,
			String subjectID) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Subjexperiment criteria = new Subjexperiment();
			criteria.setSubjectid(subjectID);
			SubjexperimentDAO dao = DAOFactory.createSubjexperimentDAO(theDBID);
			List<?> subExps = dao.find(con, criteria);
			List<Researchgroup> researchGroups = new ArrayList<Researchgroup>();
			if (subExps.isEmpty()) {
				return researchGroups;
			}

			// get the list of all research groups
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
			// use cached info
			List<Researchgroup> allResearchGroups = dbCache.getResearchGroups(ui,
					false);
			Map<BigDecimal, Researchgroup> rgMap = new HashMap<BigDecimal, Researchgroup>();
			for (Researchgroup rg : allResearchGroups) {
				rgMap.put(rg.getUniqueid(), rg);
			}
			allResearchGroups = null;
			for (Iterator<?> iter = subExps.iterator(); iter.hasNext();) {
				Subjexperiment subExp = (Subjexperiment) iter.next();
				Researchgroup rg = rgMap.get(subExp.getNcResearchgroupUniqueid());
				researchGroups.add(rg);
			}
			return researchGroups;
		} catch (Exception x) {
			log.error("Error in getExperimentsForSubject", x);
			throw new SubjectVisitManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Humansubject getSubjectByID(UserInfo ui, String subjectID)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(theDBID);
			Humansubject cr = new Humansubject();
			cr.setSubjectid(subjectID);
			List<Humansubject> hsList = dao.find(con, cr);
			if (hsList.isEmpty())
				return null;
			return hsList.get(0);

		} catch (Throwable t) {
			log.error("Error in getSubjectByID", t);
			throw new SubjectVisitManagementException(t);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			addSubject(con, ui, subject);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"humansubject");
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addSubject", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	// FIXME add siteID
	public void addSubject(Connection con, UserInfo ui, Subject subject)
			throws Exception {
		Humansubject hs = subject.toHumanSubject();

		hs.setModtime(new java.util.Date());

		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), Constants.USERCLASS_ADMIN);

		hs.setModuser(databaseUser.getUniqueid());
		hs.setOwner(databaseUser.getUniqueid());
		Tableid tid = dbCache.getTableID(ui, Constants.HUMANSUBJECT_DB_TABLE);
		hs.setTableid(tid.getTableid());
		hs.setUniqueid(seqHelper.getNextUID(ui, Constants.HUMANSUBJECT_DB_TABLE,
				"uniqueid"));
		Animalspecies asCriteria = new Animalspecies();

		asCriteria.setName("researchSubject");
		AnimalspeciesDAO asDAO = DAOFactory.createAnimalspeciesDAO(theDBID);
		List<?> asList = asDAO.find(con, asCriteria);
		// there should be one record with this name
		Animalspecies as = (Animalspecies) asList.get(0);

		hs.setNcAnimalspeciesUniqueid(as.getUniqueid());
		hs.setExtensionname("humanSubject");

		HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(theDBID);
		log.info(hs.toString());

		dao.insert(con, hs);

	}

	public void updateSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Humansubject criteria = new Humansubject();
			criteria.setSubjectid(subject.getSubjectID());

			Humansubject hs = subject.toHumanSubject();
			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(theDBID);

			dao.update(con, hs, criteria);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"humansubject");
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateSubject", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Humansubject hs = subject.toHumanSubject();

			log.debug(hs.toString());

			Subjexperiment seCriteria = new Subjexperiment();
			seCriteria.setSubjectid(hs.getSubjectid());
			SubjexperimentDAO seDAO = DAOFactory.createSubjexperimentDAO(theDBID);
			List<?> seList = seDAO.find(con, seCriteria);
			Map<BigDecimal, BigDecimal> expIDMap = new LinkedHashMap<BigDecimal, BigDecimal>(
					7);
			for (Iterator<?> iter = seList.iterator(); iter.hasNext();) {
				Subjexperiment se = (Subjexperiment) iter.next();
				if (expIDMap.get(se.getNcExperimentUniqueid()) == null) {
					expIDMap.put(se.getNcExperimentUniqueid(), se
							.getNcExperimentUniqueid());
				}
			}
			if (!expIDMap.isEmpty()) {
				for (Iterator<BigDecimal> iter = expIDMap.values().iterator(); iter
						.hasNext();) {
					BigDecimal expID = iter.next();
					Expcomponent visitCriteria = new Expcomponent();
					visitCriteria.setSubjectid(hs.getSubjectid());
					visitCriteria.setNcExperimentUniqueid(expID);
					ExpcomponentDAO visitDAO = DAOFactory
							.createExpcomponentDAO(theDBID);
					List<?> visits = visitDAO.find(con, visitCriteria);
					for (Iterator<?> it = visits.iterator(); it.hasNext();) {
						Expcomponent visit = (Expcomponent) it.next();
						removeSegmentsForVisit(con, visit);
					}
					visitDAO.delete(con, visitCriteria);
				}
			}

			if (!seList.isEmpty()) {
				// remove the subjexperiment records for this subject
				seDAO.delete(con, seCriteria);
			}

			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(theDBID);

			dao.delete(con, hs);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(
					theSiteID,
					new String[] { "humansubject", "expsegment", "expcomponent",
							"subjexperiment" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in deleteSubject", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void removeSegmentsForVisit(Connection con, Expcomponent visit)
			throws Exception {
		// delete all the assessment data associated with the visit
		deleteAssessmentData(con, visit.getSubjectid(), visit
				.getNcExperimentUniqueid().intValue(), visit.getComponentid()
				.intValue(), -1, -1);

		Expsegment sgCriteria = new Expsegment();
		sgCriteria.setComponentid(visit.getComponentid());
		sgCriteria.setNcExperimentUniqueid(visit.getNcExperimentUniqueid());
		sgCriteria.setSubjectid(visit.getSubjectid());
		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		dao.delete(con, sgCriteria);
	}

	public List<Visit> getAllVisitsForSubject(UserInfo ui, String subjectID)
			throws SubjectVisitManagementException {
		Connection con = null;

		try {
			Expcomponent criteria = new Expcomponent();
			criteria.setSubjectid(subjectID);
			con = pool.getConnection(ui.getName());
			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);

			List<?> visits = dao.find(con, criteria);
			List<Visit> visitList = new ArrayList<Visit>(visits.size());
			for (Iterator<?> iter = visits.iterator(); iter.hasNext();) {
				Expcomponent ec = (Expcomponent) iter.next();
				Visit v = new Visit(ec);
				List<?> studies = getStudies(con, ec.getSubjectid(), ec
						.getComponentid().intValue(), ec.getNcExperimentUniqueid()
						.intValue());
				for (Object element : studies) {
					Expstudy es = (Expstudy) element;
					Study study = new Study(es);
					v.addStudy(study);
					// add study segments
					List<?> segments = getSegments(con, ec.getSubjectid(), ec
							.getComponentid().intValue(), ec.getNcExperimentUniqueid()
							.intValue(), es.getStudyid().intValue());
					for (Iterator<?> it3 = segments.iterator(); it3.hasNext();) {
						Expsegment segment = (Expsegment) it3.next();
						StudySegment ss = new StudySegment(segment);
						study.addStudySegment(ss);
					}
				}
				// the remaining segments
				List<?> segments = getSegments(con, ec.getSubjectid(), ec
						.getComponentid().intValue(), ec.getNcExperimentUniqueid()
						.intValue());
				for (Iterator<?> it = segments.iterator(); it.hasNext();) {
					Expsegment segment = (Expsegment) it.next();
					VisitSegment vs = new VisitSegment(segment);
					v.addVisitSegment(vs);
				}
				visitList.add(v);
			}

			return visitList;
		} catch (Exception x) {
			log.error("Error in getAllVisitsForSubject", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<?> getSegmentsForVisit(UserInfo ui, Expcomponent visit)
			throws SubjectVisitManagementException {
		Connection con = null;
		Expsegment criteria = new Expsegment();
		try {
			con = pool.getConnection(ui.getName());
			criteria.setComponentid(visit.getComponentid());
			criteria.setSubjectid(visit.getSubjectid());
			criteria.setNcExperimentUniqueid(visit.getNcExperimentUniqueid());
			ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
			List<?> segments = dao.find(con, criteria);

			return segments;
		} catch (Exception x) {
			// log
			log.error("Error in getVisitsForSubject", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}

	}

	/**
	 * @deprecated
	 */
	public List<Expcomponent> getVisitsForSubject(UserInfo ui, SearchCriteria sc)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			TSQLQueryBuilder qb = new TSQLQueryBuilder(Expcomponent.class, sc);
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			String query = qb.buildQuery();

			if (log.isDebugEnabled()) {
				log.debug(" getVisitsForSubject query=" + query);
			}
			List<?> visits = tsp.executeQuery(con, query);
			List<Expcomponent> visitList = new ArrayList<Expcomponent>(visits
					.size());
			for (Object item : visits) {
				Expcomponent visit = (Expcomponent) item;
				visitList.add(visit);
			}

			return visitList;
		} catch (Exception x) {
			// log
			log.error("Error in getVisitsForSubject", x);
			throw new SubjectVisitManagementException(x.getMessage());
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addVisitForSubjectNonContiguous(Connection con, UserInfo ui,
			Visit visit) throws SubjectVisitManagementException {
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
			}
			Expcomponent expComp = visit.toExpComponent();
			expComp.setModtime(new java.util.Date());
			/**
			 * TODO timeinterval usage needs to first defined to be correctly
			 * implemented here
			 */
			expComp.setIstimeinterval(new Boolean(false));
			int visitID = visit.getComponentID();
			expComp.setComponentid(GenUtils.toBigDecimal(visitID));
			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);

			// set the owner and moduser
			Map<String, User> userMap = securityService.getAllUsers(theDBID);
			User u = userMap.get(ui.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
					.getName(), Constants.USERCLASS_ADMIN);

			expComp.setModuser(databaseUser.getUniqueid());
			expComp.setOwner(databaseUser.getUniqueid());
			Tableid tid = dbCache.getTableID(ui, Constants.COMPONENT_DB_TABLE);
			expComp.setTableid(tid.getTableid());

			BigDecimal uniqueid = seqHelper.getNextUID(ui,
					Constants.COMPONENT_DB_TABLE, "uniqueid");
			expComp.setUniqueid(uniqueid);

			if (visit.getName() != null) {
				expComp.setName(visit.getName());

			}
			log.info(expComp.toString());
			dao.insert(con, expComp);
			if (visit.getStudies() != null && !visit.getStudies().isEmpty()) {
				// now add the studies also
				for (Study study : visit.getStudies()) {
					study.setComponentID(visitID);
					handleAddStudy(con, ui, study);
				}
			} else {
				// now add the segments also
				List<VisitSegment> visitSegments = visit.getVisitSegments();
				for (VisitSegment vs : visitSegments) {
					// set component id
					vs.setVisitID(visitID);
					handleAddVisitSegment(con, ui, vs);
				}
			}
			// notify any listening HID web apps to update their cache
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment", "expcomponent", "expstudy" });
		} catch (Exception x) {
			throw new SubjectVisitManagementException(x);
		}
	}

	public int addVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException {
		Connection con = null;
		int visitID;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Expcomponent expComp = visit.toExpComponent();
			expComp.setModtime(new java.util.Date());
			/**
			 * TODO timeinterval usage needs to first defined to be correctly
			 * implemented here
			 */
			expComp.setIstimeinterval(new Boolean(false));

			visitID = getMaxVisitID(con, visit.getSubjectID(), visit
					.getExperimentID());
			visitID = (visitID <= 0) ? 1 : ++visitID;

			expComp.setComponentid(GenUtils.toBigDecimal(visitID));
			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);

			// set the owner and moduser
			Map<String, User> userMap = securityService.getAllUsers(theDBID);
			User u = userMap.get(ui.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
					.getName(), Constants.USERCLASS_ADMIN);

			expComp.setModuser(databaseUser.getUniqueid());
			expComp.setOwner(databaseUser.getUniqueid());
			Tableid tid = dbCache.getTableID(ui, Constants.COMPONENT_DB_TABLE);
			expComp.setTableid(tid.getTableid());

			BigDecimal uniqueid = seqHelper.getNextUID(ui,
					Constants.COMPONENT_DB_TABLE, "uniqueid");
			expComp.setUniqueid(uniqueid);

			if (visit.getName() != null) {
				expComp.setName(visit.getName());

			}
			log.info(expComp.toString());
			dao.insert(con, expComp);
			if (visit.getStudies() != null && !visit.getStudies().isEmpty()) {
				// now add the studies also
				for (Study study : visit.getStudies()) {
					study.setComponentID(visitID);
					handleAddStudy(con, ui, study);
				}
			} else {
				// now add the segments also
				List<VisitSegment> visitSegments = visit.getVisitSegments();
				for (VisitSegment vs : visitSegments) {
					// set component id
					vs.setVisitID(visitID);
					handleAddVisitSegment(con, ui, vs);
				}
			}
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment", "expcomponent", "expstudy" });
			return visitID;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitForSubject", x, true);
			return -1;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void updateVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Expcomponent criteria = new Expcomponent();
			criteria.setComponentid(GenUtils.toBigDecimal(visit.getComponentID()));
			criteria.setSubjectid(visit.getSubjectID());
			criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(visit
					.getExperimentID()));

			Expcomponent comp = visit.toExpComponent();

			// cannot update primary key so reset them
			comp.setComponentid(null);
			comp.setSubjectid(null);
			comp.setNcExperimentUniqueid(null);

			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);
			dao.update(con, comp, criteria);

			if (!visit.getStudies().isEmpty()) {
				for (Study study : visit.getStudies()) {
					handleUpdateStudy(con, study);
				}
			} else {
				for (VisitSegment vs : visit.getVisitSegments()) {
					handleUpdateVisitSegment(con, vs);
				}
			}

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment", "expcomponent", "expstudy" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateVisitForSubject", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public void deleteVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Expcomponent criteria = visit.toExpComponent();

			// delete all the assessment data associated with the visit
			deleteAssessmentData(con, visit.getSubjectID(), visit
					.getExperimentID(), visit.getComponentID(), -1, -1);

			List<?> segments = getSegments(con, visit.getSubjectID(), visit
					.getComponentID(), visit.getExperimentID());
			ExpsegmentDAO segDAO = DAOFactory.createExpsegmentDAO(theDBID);
			for (Iterator<?> iter = segments.iterator(); iter.hasNext();) {
				Expsegment segment = (Expsegment) iter.next();
				segDAO.delete(con, segment);
			}

			// delete studies if any
			ExpstudyDAO studyDAO = DAOFactory.createExpstudyDAO(theDBID);
			for (Study study : visit.getStudies()) {
				Expstudy sc = new Expstudy();
				sc.setStudyid(GenUtils.toBigDecimal(study.getStudyID()));
				studyDAO.delete(con, sc);
			}

			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);
			dao.delete(con, criteria);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment", "expcomponent", "expstudy" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitForSubject", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected int getMaxVisitID(Connection con, String subjectID,
			int experimentID) throws Exception {
		int maxVisitID = -1;
		Expcomponent criteria = new Expcomponent();
		criteria.setSubjectid(subjectID);
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(experimentID));
		ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(theDBID);
		List<?> comps = dao.find(con, criteria);
		for (Iterator<?> iter = comps.iterator(); iter.hasNext();) {
			Expcomponent comp = (Expcomponent) iter.next();
			if (comp.getComponentid().intValue() > maxVisitID) {
				maxVisitID = comp.getComponentid().intValue();
			}
		}

		return maxVisitID;
	}

	
	
	public int addVisitSegmentForSubject(UserInfo ui, VisitSegment visitSegment)
			throws SubjectVisitManagementException {
		Connection con = null;

		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();

			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			int newSegmentID = handleAddVisitSegment(con, ui, visitSegment);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment" });
			return newSegmentID;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitSegmentForSubject", x,
					true);
			return -1;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public int addStudySegmentForSubject(UserInfo ui, StudySegment studySegment)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();

			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			int newSegmentID = handleAddStudySegment(con, ui, studySegment);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment" });
			return newSegmentID;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitSegmentForSubject", x,
					true);
			return -1;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Study addStudyForVisit(UserInfo ui, Study study)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();

			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			study = handleAddStudy(con, ui, study);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expstudy", "expsegment" });
			return study;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitSegmentForSubject", x,
					true);
		} finally {
			releaseConnection(con, ui);
		}
		return null;
	}
	
	public Study addNewStudyForVisit(UserInfo ui, Study study)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
		
			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			study = handleAddNewStudyOnly(con, ui, study);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expstudy", "expsegment" });
			return study;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitSegmentForSubject", x,
					true);
		} finally {
			releaseConnection(con, ui);
		}
		return null;
		}

	public void updateStudyForVisit(UserInfo ui, Study study)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();

			}
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			handleUpdateStudy(con, study);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expstudy", "expsegment" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addVisitSegmentForSubject", x,
					true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected List<?> getSegments(Connection con, String subjectID, int visitID,
			int experimentID) throws Exception {
		return getSegments(con, subjectID, visitID, experimentID, -1);
	}

	protected List<?> getSegments(Connection con, String subjectID, int visitID,
			int experimentID, int studyID) throws Exception {
		Expsegment criteria = new Expsegment();
		criteria.setSubjectid(subjectID);
		criteria.setComponentid(GenUtils.toBigDecimal(visitID));
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(experimentID));
		if (studyID != -1) {
			criteria.setStudyid(GenUtils.toBigDecimal(studyID));
		}
		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		List<?> segments = dao.find(con, criteria);
		if (studyID == -1) {
			// remove the segments which belong to any study from the result set
			for (Iterator<?> iter = segments.iterator(); iter.hasNext();) {
				Expsegment es = (Expsegment) iter.next();
				if (es.getStudyid() != null) {
					iter.remove();
				}
			}
		}
		return segments;
	}

	protected List<?> getStudies(Connection con, String subjectID, int visitID,
			int experimentID) throws Exception {
		Expstudy criterion = new Expstudy();
		criterion.setSubjectid(subjectID);
		criterion.setComponentid(GenUtils.toBigDecimal(visitID));
		criterion.setExperimentid(GenUtils.toBigDecimal(experimentID));
		ExpstudyDAO dao = DAOFactory.createExpstudyDAO(theDBID);
		return dao.find(con, criterion);
	}

	public int handleAddVisitSegment(Connection con, UserInfo ui,
			VisitSegment visitSegment) throws Exception {
		Expsegment criteria = new Expsegment();
		criteria.setSubjectid(visitSegment.getSubjectID());
		criteria.setComponentid(GenUtils.toBigDecimal(visitSegment.getVisitID()));
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(visitSegment
				.getExperimentID()));

		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		List<?> segments = dao.find(con, criteria);
		int maxSegmentID = -1;
		for (Object element : segments) {
			Expsegment segment = (Expsegment) element;
			if (segment.getSegmentid().intValue() > maxSegmentID) {
				maxSegmentID = segment.getSegmentid().intValue();
			}
		}

		if (maxSegmentID > 0) {
			maxSegmentID++;
		} else {
			maxSegmentID = 1;

		}
		Expsegment segment = visitSegment.toExpSegment();

		segment.setSegmentid(GenUtils.toBigDecimal(maxSegmentID));
		segment.setModtime(new java.util.Date());

		// set the owner and moduser
		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), Constants.USERCLASS_ADMIN);

		segment.setModuser(databaseUser.getUniqueid());
		segment.setOwner(databaseUser.getUniqueid());
		Tableid tid = dbCache.getTableID(ui, Constants.SEGMENT_DB_TABLE);
		segment.setTableid(tid.getTableid());

		/**
		 * @todo timeinterval usage needs to be defined for the proper
		 *       implementation
		 */
		segment.setIstimeinterval(new Boolean(false));

		segment.setIsbad(new Boolean(false));

		segment.setUniqueid(seqHelper.getNextUID(ui, Constants.SEGMENT_DB_TABLE,
				"uniqueid"));

		log.info(segment.toString());

		dao.insert(con, segment);
		return segment.getSegmentid().intValue();
	}

	protected Study handleAddNewStudyOnly(Connection con, UserInfo ui, Study study)
			throws Exception {
		Expstudy criterion = new Expstudy();
		criterion.setSubjectid(study.getSubjectID());
		criterion.setComponentid(GenUtils.toBigDecimal(study.getComponentID()));
		criterion.setExperimentid(GenUtils.toBigDecimal(study.getExperimentID()));
		ExpstudyDAO dao = DAOFactory.createExpstudyDAO(theDBID);
		List<?> studies = dao.find(con, criterion);
		int maxStudyID = -1;
		for (Iterator<?> iter = studies.iterator(); iter.hasNext();) {
			Expstudy es = (Expstudy) iter.next();
			if (es.getStudyid().intValue() > maxStudyID) {
				maxStudyID = es.getStudyid().intValue();
			}
		}
		if (maxStudyID > 0) {
			maxStudyID++;
		} else {
			maxStudyID = 1;
		}
		Expstudy es = study.toExpStudy();
		es.setStudyid(GenUtils.toBigDecimal(maxStudyID));
		study.setStudyID(maxStudyID);
		
		BigDecimal uniqID = seqHelper.getNextUID(ui, Constants.STUDY_DB_TABLE,
				"uniqueid");
		es.setUniqueid(uniqID);
		
		// set the owner and moduser
		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = userMap.get(ui.getName());
		
		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), Constants.USERCLASS_ADMIN);
		
		es.setModuser(databaseUser.getUniqueid());
		es.setOwner(databaseUser.getUniqueid());
		es.setModtime(new java.util.Date());
		Tableid tid = dbCache.getTableID(ui, Constants.STUDY_DB_TABLE);
		es.setTableid(tid.getTableid());
		log.info(es.toString());
		dao.insert(con, es);
		
		return study;
	}


	
	protected Study handleAddStudy(Connection con, UserInfo ui, Study study)
			throws Exception {
		Expstudy criterion = new Expstudy();
		criterion.setSubjectid(study.getSubjectID());
		criterion.setComponentid(GenUtils.toBigDecimal(study.getComponentID()));
		criterion.setExperimentid(GenUtils.toBigDecimal(study.getExperimentID()));
		ExpstudyDAO dao = DAOFactory.createExpstudyDAO(theDBID);
		List<?> studies = dao.find(con, criterion);
		int maxStudyID = -1;
		for (Iterator<?> iter = studies.iterator(); iter.hasNext();) {
			Expstudy es = (Expstudy) iter.next();
			if (es.getStudyid().intValue() > maxStudyID) {
				maxStudyID = es.getStudyid().intValue();
			}
		}
		if (maxStudyID > 0) {
			maxStudyID++;
		} else {
			maxStudyID = 1;
		}
		Expstudy es = study.toExpStudy();
		es.setStudyid(GenUtils.toBigDecimal(maxStudyID));
		study.setStudyID(maxStudyID);

		BigDecimal uniqID = seqHelper.getNextUID(ui, Constants.STUDY_DB_TABLE,
				"uniqueid");
		es.setUniqueid(uniqID);

		// set the owner and moduser
		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), Constants.USERCLASS_ADMIN);

		es.setModuser(databaseUser.getUniqueid());
		es.setOwner(databaseUser.getUniqueid());
		es.setModtime(new java.util.Date());
		Tableid tid = dbCache.getTableID(ui, Constants.STUDY_DB_TABLE);
		es.setTableid(tid.getTableid());
		log.info(es.toString());
		dao.insert(con, es);

		// add study segments also
		for (StudySegment ss : study.getStudySegments()) {
			log.info("adding study segment");
			ss.setStudyID(study.getStudyID());
			log.info("studySegment " + ss.toString());
			handleAddStudySegment(con, ui, ss);
		}
		return study;
	}
	
	
	

	protected int handleAddStudySegment(Connection con, UserInfo ui,
			StudySegment studySegment) throws Exception {
		Expsegment criteria = new Expsegment();
		criteria.setSubjectid(studySegment.getSubjectID());
		criteria.setComponentid(GenUtils.toBigDecimal(studySegment.getVisitID()));
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(studySegment
				.getExperimentID()));

		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		List<?> segments = dao.find(con, criteria);
		int maxSegmentID = -1;
		for (Iterator<?> iter = segments.iterator(); iter.hasNext();) {
			Expsegment segment = (Expsegment) iter.next();
			if (segment.getSegmentid().intValue() > maxSegmentID) {
				maxSegmentID = segment.getSegmentid().intValue();
			}
		}
		if (maxSegmentID > 0) {
			maxSegmentID++;
		} else {
			maxSegmentID = 1;

		}
		Expsegment segment = studySegment.toExpSegment();

		studySegment.setSegmentID(maxSegmentID);
		segment.setSegmentid(GenUtils.toBigDecimal(maxSegmentID));
		segment.setModtime(new java.util.Date());

		// set the owner and moduser
		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), Constants.USERCLASS_ADMIN);

		segment.setModuser(databaseUser.getUniqueid());
		segment.setOwner(databaseUser.getUniqueid());
		Tableid tid = dbCache.getTableID(ui, Constants.SEGMENT_DB_TABLE);
		segment.setTableid(tid.getTableid());

		segment.setUniqueid(seqHelper.getNextUID(ui, Constants.SEGMENT_DB_TABLE,
				"uniqueid"));
		// set the study id
		segment.setStudyid(GenUtils.toBigDecimal(studySegment.getStudyID()));
		// TODO implement time interval logic
		segment.setIstimeinterval(new Boolean(false));

		segment.setIsbad(new Boolean(false));
		log.info("inserting studySegment " + segment.toString());
		dao.insert(con, segment);
		return segment.getSegmentid().intValue();
	}

	protected void handleUpdateStudy(Connection con, Study study)
			throws Exception {
		Expstudy criteria = new Expstudy();
		criteria.setSubjectid(study.getSubjectID());
		criteria.setComponentid(GenUtils.toBigDecimal(study.getComponentID()));
		criteria.setExperimentid(GenUtils.toBigDecimal(study.getExperimentID()));
		criteria.setStudyid(GenUtils.toBigDecimal(study.getStudyID()));
		ExpstudyDAO dao = DAOFactory.createExpstudyDAO(theDBID);
		Expstudy es = study.toExpStudy();
		// reset the primary key part since primary key can not be updated
		es.setStudyid(null);
		es.setModtime(new java.util.Date());
		log.info("study to update " + es.toString() + "\ncriteria=" + criteria);
		dao.update(con, es, criteria);
		// check dirty flag to determine what to update
		for (StudySegment ss : study.getStudySegments()) {
			if (ss.getDirty()) {
				handleUpdateStudySegment(con, ss);
				ss.setDirty(false);
			}
		}
	}

	protected void handleUpdateStudySegment(Connection con,
			StudySegment studySegment) throws Exception {
		Expsegment criteria = new Expsegment();
		criteria.setSubjectid(studySegment.getSubjectID());
		criteria.setSegmentid(GenUtils.toBigDecimal(studySegment.getSegmentID()));
		criteria.setComponentid(GenUtils.toBigDecimal(studySegment.getVisitID()));
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(studySegment
				.getExperimentID()));
		if (studySegment.getStudyID() >= 0) {
			criteria.setStudyid(GenUtils.toBigDecimal(studySegment.getStudyID()));
		}
		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		Expsegment segment = studySegment.toExpSegment();
		// reset the primary key part since primary key can not be updated
		segment.setComponentid(null);
		segment.setSegmentid(null);
		segment.setSubjectid(null);
		segment.setNcExperimentUniqueid(null);
		segment.setModtime(new java.util.Date());
		log.info("handleUpdateStudySegment - " + segment);
		dao.update(con, segment, criteria);
	}

	public void updateVisitSegmentForSubject(UserInfo ui,
			VisitSegment visitSegment) throws SubjectVisitManagementException {
		Connection con = null;
		try {

			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			handleUpdateVisitSegment(con, visitSegment);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateVisitSegmentForSubject",
					x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void updateStudySegmentForSubject(UserInfo ui,
			StudySegment studySegment) throws SubjectVisitManagementException {
		Connection con = null;
		try {

			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			handleUpdateStudySegment(con, studySegment);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateStudySegmentForSubject",
					x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void handleUpdateVisitSegment(Connection con,
			VisitSegment visitSegment) throws Exception {
		Expsegment criteria = new Expsegment();
		criteria.setSubjectid(visitSegment.getSubjectID());
		criteria.setComponentid(GenUtils.toBigDecimal(visitSegment.getVisitID()));
		criteria.setSegmentid(GenUtils.toBigDecimal(visitSegment.getSegmentID()));
		criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(visitSegment
				.getExperimentID()));

		ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
		Expsegment segment = visitSegment.toExpSegment();
		// reset the primary key part since primary key can not be updated
		segment.setComponentid(null);
		segment.setSegmentid(null);
		segment.setSubjectid(null);
		segment.setNcExperimentUniqueid(null);
		segment.setModtime(new java.util.Date());
		log.info("handleUpdateVisitSegment - " + segment);
		dao.update(con, segment, criteria);
	}

	public void deleteVisitSegmentForSubject(UserInfo ui,
			VisitSegment visitSegment) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			deleteAssessmentData(con, visitSegment.getSubjectID(), visitSegment
					.getExperimentID(), visitSegment.getVisitID(), visitSegment
					.getSegmentID(), -1);

			Expsegment criteria = visitSegment.toExpSegment();

			// no CLOB support in the where part of the delete query
			criteria.setDescription(null);

			ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
			dao.delete(con, criteria);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in deleteVisitSegmentForSubject",
					x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteStudySegmentForSubject(UserInfo ui,
			StudySegment studySegment) throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			deleteAssessmentData(con, studySegment.getSubjectID(), studySegment
					.getExperimentID(), studySegment.getVisitID(), studySegment
					.getSegmentID(), -1);

			Expsegment criteria = studySegment.toExpSegment();
			// no CLOB support in the where part of the delete query
			criteria.setDescription(null);

			ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(theDBID);
			dao.delete(con, criteria);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					new String[] { "expsegment" });
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in deleteStudySegmentForSubject",
					x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteAssessmentFromSegment(UserInfo ui, String subjectID,
			int visitID, int segmentID, int experimentID, int assessmentID)
			throws SubjectVisitManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			deleteAssessmentData(con, subjectID, experimentID, visitID, segmentID,
					assessmentID);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in deleteAssessmentFromSegment", x,
					true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	protected void deleteAssessmentData(Connection con, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID)
			throws Exception {
		StoredassessmentDAO saDAO = DAOFactory.createStoredassessmentDAO(theDBID);
		Storedassessment cr = new Storedassessment();
		if (subjectID != null) {
			cr.setSubjectid(subjectID);
		}
		cr.setNcExperimentUniqueid(GenUtils.toBigDecimal(experimentID));
		if (visitID >= 0) {
			cr.setComponentid(GenUtils.toBigDecimal(visitID));
		}
		if (segmentID >= 0) {
			cr.setSegmentid(GenUtils.toBigDecimal(segmentID));
		}
		if (assessmentID >= 0) {
			cr.setAssessmentid(GenUtils.toBigDecimal(assessmentID));
		}
		List<?> saList = saDAO.find(con, cr);
		if (!saList.isEmpty()) {
			dropAssessmentValues(con, "integer", saList, assessmentID);
			dropAssessmentValues(con, "float", saList, assessmentID);
			dropAssessmentValues(con, "varchar", saList, assessmentID);
			dropAssessmentValues(con, "boolean", saList, assessmentID);
			dropAssessmentValues(con, "timestamp", saList, assessmentID);

			dropStoredAssessments(con, saList);
		}
	}

	protected void dropAssessmentValues(Connection con, String tableType,
			List<?> saList, int assessmentID) throws Exception {
		StringBuffer buf = new StringBuffer(128);
		Map<String, String> map = new HashMap<String, String>(7);
		map.put("integer", "nc_assessmentinteger");
		map.put("float", "nc_assessmentfloat");
		map.put("varchar", "nc_assessmentvarchar");
		map.put("boolean", "nc_assessmentboolean");
		map.put("timestamp", "nc_assessmenttimestamp");

		Statement st = null;
		try {
			buf.append("delete from ").append(map.get(tableType));
			buf.append(" where storedassessmentid in (");
			for (Iterator<?> iter = saList.iterator(); iter.hasNext();) {
				Storedassessment sa = (Storedassessment) iter.next();
				buf.append(sa.getUniqueid());
				if (iter.hasNext()) {
					buf.append(',');
				}
			}
			buf.append(')');
			if (assessmentID != -1) {
				buf.append(" and assessmentid = ").append(assessmentID);
			}
			st = con.createStatement();
			log.info("query=" + buf.toString());
			st.execute(buf.toString());
		} finally {
			DBUtils.close(st);
		}
	}

	protected void dropStoredAssessments(Connection con, List<?> saList)
			throws Exception {
		StringBuffer buf = new StringBuffer(128);
		Statement st = null;
		try {
			buf.append("delete from nc_storedassessment");
			buf.append(" where uniqueid in (");
			for (Iterator<?> iter = saList.iterator(); iter.hasNext();) {
				Storedassessment sa = (Storedassessment) iter.next();
				buf.append(sa.getUniqueid());
				if (iter.hasNext()) {
					buf.append(',');
				}
			}
			buf.append(')');
			st = con.createStatement();
			log.info("query=" + buf.toString());
			st.execute(buf.toString());
		} finally {
			DBUtils.close(st);
		}
	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x, boolean doRollback)
			throws SubjectVisitManagementException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new SubjectVisitManagementException(se.getMessage());
			}
		}
		log.error(msg, x);
		throw new SubjectVisitManagementException(x);
	}

}
