package clinical.web.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import clinical.utils.DateTimeUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentResultSummary.java,v 1.1.2.1 2008/07/24 01:08:35
 *          bozyurt Exp $
 */
public class AssessmentResultSummary implements Serializable {
	private static final long serialVersionUID = -5870226531466636501L;
	private String subjectID;
	private String expID;
	private String expName = "unknown";
	private String siteID;
	private String timeStamp;
	private Map<String, VisitSegAsResultValues> vsarvMap = new LinkedHashMap<String, VisitSegAsResultValues>(
			3);
	private Set<String> uniqSiteIDs = new TreeSet<String>();
	private Map<String, String> paramMap = new HashMap<String, String>(3);
	
	private String selectedVers;	
	private String selectedSegs;
	private String allSegs;
	private String selectedDataTypes;

	public AssessmentResultSummary(){		
	}
	
	public String getSelectedSegs() {
		return selectedSegs;
	}

	public void setSelectedSegs(String selectedSegs) {
		this.selectedSegs = selectedSegs;
	}
	
	public String getSelectedDataTypes() {
		return selectedDataTypes;
	}

	public void setSelectedDataTypes(String selectedDataTypes) {
		this.selectedDataTypes = selectedDataTypes;
	}

	public String getAllSegs() {
		return allSegs;
	}

	public void setAllSegs(String allSegs) {
		this.allSegs = allSegs;
	}
	
	public AssessmentResultSummary(String subjectID, String expID,
			String expName, String siteID, String timeStamp) {
		super();
		this.subjectID = subjectID;
		this.expID = expID;
		this.expName = expName;
		this.siteID = siteID;
		this.timeStamp = timeStamp;
	}

	public String getKey() {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public static String getKey(String subjectID, int expID, String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public VisitSegAsResultValues add(int visitID, int segmentID,
			int experimentID, String siteID) {
		if (!this.expID.equals(experimentID)) {
			throw new RuntimeException("Invalid expID:" + experimentID
					+ " Expected:" + this.expID);
		}
		String key = getKey(visitID, segmentID, experimentID, siteID);
		VisitSegAsResultValues vsarv = new VisitSegAsResultValues(visitID,
				segmentID, experimentID, siteID);
		vsarvMap.put(key, vsarv);
		if (siteID != null)
			uniqSiteIDs.add(siteID);
		return vsarv;
	}

	public VisitSegAsResultValues getOrAdd(int visitID, int segmentID,
			int experimentID, String siteID) {
		String key = getKey(visitID, segmentID, experimentID, siteID);
		VisitSegAsResultValues vsarv = vsarvMap.get(key);
		if (vsarv == null) {
			vsarv = new VisitSegAsResultValues(visitID, segmentID,
					experimentID, siteID);
			vsarvMap.put(key, vsarv);
			if (siteID != null)
				uniqSiteIDs.add(siteID);

		}
		return vsarv;
	}
	
	public VisitSegAsResultValues getOrAdd(int visitID, int segmentID,
			int experimentID, String siteID, String visitName, String visitDate,
			String studyGroup, String segmentName, String segmentDate,
			String studyName, String studyDate, String studyId) {
		String key = getKey(visitID, segmentID, experimentID, siteID);
		VisitSegAsResultValues vsarv = vsarvMap.get(key);
		if (vsarv == null) {
			vsarv = new VisitSegAsResultValues(visitID, segmentID,
					experimentID, siteID, visitName, visitDate, studyGroup, segmentName, segmentDate);
			vsarv.setStudyDate(studyDate);
			vsarv.setStudyName(studyName);
			if(studyId!=null) vsarv.setStudyID(Integer.valueOf(studyId));			
			vsarvMap.put(key, vsarv);
			if (siteID != null)
				uniqSiteIDs.add(siteID);

		}
		return vsarv;
	}

	public VisitSegAsResultValues get(int visitID, int segmentID, int expID,
			String siteID, String asID, String scoreName) {
		String key = getKey(visitID, segmentID, expID, siteID);
		return vsarvMap.get(key);
	}

	public Set<ScoreMetaData> getScoreMetadata() {
		Set<ScoreMetaData> uniqSet = new HashSet<ScoreMetaData>(17);
		for (VisitSegAsResultValues vsarv : vsarvMap.values()) {
			uniqSet.addAll(vsarv.getScoreMetadata());
		}
		return uniqSet;
	}

	/**
	 * if there are matching score values for more than one visit, then use the
	 * newest visit.
	 *
	 * @param varList
	 * @return keyed by scoreName ':' asName ':' visitID
	 */
	public Map<String, String> getMatchingValues(List<ScoreMetaData> varList) {
		Map<String, String> map = new HashMap<String, String>(17);
		VisitSegAsResultValues newest = null;
		Date newestDate = null;
		for (VisitSegAsResultValues vsarv : vsarvMap.values()) {
			if (newest == null) {
				newest = vsarv;
				newestDate = DateTimeUtils.toDate(vsarv.getTimeStamp());
			} else {
				if (newestDate.before(DateTimeUtils
						.toDate(vsarv.getTimeStamp()))) {
					newest = vsarv;
					newestDate = DateTimeUtils.toDate(vsarv.getTimeStamp());
				}
			}
		}

		Map<String, SubjectAsScoreValueSummary> sasvsMap = newest
				.getScore2SavsMap();
		for (ScoreMetaData smd : varList) {
			String key = smd.getScoreName() + ":" + smd.getAsName();
			SubjectAsScoreValueSummary sasvs = sasvsMap.get(key);
			if (sasvs != null && sasvs.getValue() != null) {
				String aKey = smd.getScoreName() + ":" + smd.getAsName() + ":" + sasvs.getVisitID();
				map.put(aKey, sasvs.getValue().toString());
			}
		}
		return map;
	}

	public static String getKey(int visitID, int segmentID, int expID,
			String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(visitID).append(':').append(segmentID).append(':').append(
				expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public Map<String, String> getParamMap() {
		if (paramMap.isEmpty()) {
			paramMap.put("sid", getSubjectID());
			paramMap.put("siteid", getSiteID());
		}
		return paramMap;
	}

	public String getAllSiteIDs() {
		StringBuilder buf = new StringBuilder();
		for (Iterator<String> it = uniqSiteIDs.iterator(); it.hasNext();) {
			String sid = it.next();
			buf.append(sid);
			if (it.hasNext())
				buf.append(',');
		}
		return buf.toString();
	}

	public String getSubjectID() {
		return subjectID;
	}

	public String getExpID() {
		return expID;
	}

	public String getExpName() {
		return expName;
	}

	public String getSiteID() {
		return siteID;
	}
	
	public void setSiteID(String siteId){
		this.siteID = siteId;
	}

	public String getTimeStamp() {
		return timeStamp;
	}

	public Set<String> getUniqSiteIDs() {
		return uniqSiteIDs;
	}

	public Map<String, VisitSegAsResultValues> getVsarvMap() {
		return vsarvMap;
	}

	public List<VisitSegAsResultValues> getVsarvList() {
		return new ArrayList<VisitSegAsResultValues>(vsarvMap.values());
	}

	public String getParamJSON() {
		StringBuilder buf = new StringBuilder();
		buf.append("['subjectID=").append(subjectID).append("',");
		buf.append("'siteID=").append(siteID);
		//buf.append("','expID=").append(expID).append("']");
		buf.append("','expID=").append(expID).append("',");
		buf.append("'selectedSegments=").append(selectedSegs).append("',");
		buf.append("'allSegments=").append(allSegs).append("',");
		buf.append("'selectedDataTypes=").append(selectedDataTypes).append("',");
		buf.append("'selectedVers=").append(selectedVers).append("']");
		return buf.toString();
	}
	
	public void setSelectedVers(String selectedVers) {
		this.selectedVers = selectedVers;
	}

	public String getSelectedVers() {
		return selectedVers;
	}
}
