package clinical.tools;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import clinical.server.vo.ConfParamsGroup;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.services.AppConfigService;

/**
 * @author I. Burak Ozyurt
 * @version $Id: Properties2DBConverter.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class Properties2DBConverter {
	protected ConnectionSupportMixin csm;
	protected List<ParamInfo> piList = new ArrayList<ParamInfo>();
	protected Map<String, ConfParamsGroup> pgMap = new HashMap<String, ConfParamsGroup>();

	public Properties2DBConverter(String propsFile) throws Exception {
		csm = new ConnectionSupportMixin(propsFile, true);
		csm.startup();
	}

	public void shutdown() throws Exception {
		if (csm != null)
			csm.shutdown();
	}

	public void populateDB() throws Exception {
		AppConfigService acs = AppConfigService.getInstance(csm
				.getDbPoolService(), csm.getDbID());

		ConfParamsGroup curPG = null;
		
		List<ConfParamsGroup> pgList = acs.getAllParamGroups();
		
		for (ConfParamsGroup pg : pgList) {
			pgMap.put(pg.getGroupName(), pg);
			curPG = pg;
		}
		BufferedReader console = new BufferedReader(new InputStreamReader(
				System.in));

		for (ParamInfo pi : piList) {
			System.out.println(pi);
			if (curPG == null) {
				curPG = createParamsGroupInteractively(acs, console);
				pgMap.put(curPG.getGroupName(), curPG);
			}
			String ans = getUserInput(console, "Add param '" + pi.name
					+ "' to db (y/n):");
			if (!ans.toLowerCase().equals("y"))
				continue;
			ans = getUserInput(console, "Use current group '"
					+ curPG.getGroupName() + "' (y/n):");
			if (ans.toLowerCase().equals("n")) {
				List<String> choiceList = new ArrayList<String>(pgMap.keySet());
				Collections.sort(choiceList);
				String groupName = userSelectFromList(console, choiceList);
				curPG = null;
				if (groupName != null) {
					curPG = pgMap.get(groupName);
				}
				if (curPG == null) {
					curPG = createParamsGroupInteractively(acs, console);
					pgMap.put(curPG.getGroupName(), curPG);
				}
			}
			createParamInteractively(acs, console, pi, curPG);
		}
	}

	protected void createParamInteractively(AppConfigService acs,
			BufferedReader console, ParamInfo pi, ConfParamsGroup pg)
			throws Exception {
		System.out.println("Param Creation");
		System.out.println("--------------");
		String ans = getUserInput(console, "New description (y/[n]):", "n");
		if (ans.toLowerCase().equals("y")) {
			pi.description = getUserInput(console, "Description:");
		}
		String type = getUserInput(console,
				"Param Type ([string],int,real,bool):", "string");

		acs.addParameter(pi.name, pi.value, type, pi.description, pg
				.getGroupName());
	}

	protected ConfParamsGroup createParamsGroupInteractively(
			AppConfigService acs, BufferedReader console) throws Exception {
		System.out.println("Param Group Creation");
		System.out.println("--------------------");
		String groupName = getUserInput(console, "Group Name:");
		String descr = getUserInput(console, "Description:");

		return acs.addParameterGroup(groupName, descr);
	}

	protected String userSelectFromList(BufferedReader console,
			List<String> choiceList) throws IOException {
		int idx = 0;
		for (String choice : choiceList) {
			System.out.println("\t" + idx + ") " + choice);
			idx++;
		}
		System.out.print("Choice:");
		String ans = console.readLine();
		int ci = GenUtils.toInt(ans, -1);
		if (ci >= 0) {
			return choiceList.get(ci);
		}
		return null;
	}

	protected String getUserInput(BufferedReader console, String msg)
			throws IOException {
		while (true) {
			System.out.print(msg);
			String value = console.readLine();
			value = value.trim();
			if (value.length() > 0) {
				return value;
			}
		}
	}

	protected String getUserInput(BufferedReader console, String msg,
			String defaultVal) throws IOException {
		while (true) {
			System.out.print(msg);
			String value = console.readLine();
			value = value.trim();
			if (value.length() > 0) {
				return value;
			} else if (defaultVal != null) {
				return defaultVal;
			}
		}
	}

	public void loadProperties(String propertiesFile) throws IOException {
		BufferedReader in = null;
		StringBuffer descBuf = null;
		try {
			in = new BufferedReader(new FileReader(propertiesFile));
			String line = null;
			while ((line = in.readLine()) != null) {
				line = line.trim();
				if (line.length() == 0)
					continue;
				if (line.startsWith("#")) {
					if (descBuf == null)
						descBuf = new StringBuffer(128);
					descBuf.append(line.substring(1)).append('\n');
				} else {
					String[] toks = line.split("\\s*=\\s*");
					String name = toks[0];
					String value = " ";
					if (toks.length > 1) {
						value = toks[1];
					}

					if (value.indexOf(',') != -1) {
						String[] values = value.split("\\s*,\\s*");
						for (int i = 0; i < values.length; i++) {
							String pName = name + "." + (i + 1);
							ParamInfo pi = new ParamInfo(pName, values[i]);
							if (descBuf != null)
								pi.description = descBuf.toString().trim();
							piList.add(pi);
						}
						descBuf = null;
					} else {
						ParamInfo pi = new ParamInfo(name, value);
						if (descBuf != null) {
							pi.description = descBuf.toString().trim();
							descBuf = null;
						}
						piList.add(pi);
					}
				}
			}
			for (ParamInfo pi : piList) {
				System.out.println(pi);
			}
		} finally {
			FileUtils.close(in);
		}
	}

	public static class ParamInfo {
		String name;
		String value;
		String description;

		public ParamInfo(String name, String value) {
			this.name = name;
			this.value = value;
		}

		public String toString() {
			StringBuffer buf = new StringBuffer();
			buf.append("ParamInfo::[");
			buf.append("name=").append(name);
			buf.append(",value=").append(value);
			buf.append(",descr=").append(description);
			buf.append(']');
			return buf.toString();
		}
	}

	public static void main(String[] args) throws Exception {
		Properties2DBConverter conv = null;
		String appPropsFile = "/home/bozyurt/dev/java/BIRN/clinical/conf/clinical.properties";
		try {
			conv = new Properties2DBConverter("test.properties");
			conv.loadProperties(appPropsFile);
			conv.populateDB();

		} finally {
			if (conv != null)
				conv.shutdown();
		}
	}
}
