package clinical.tools.install;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.input.InputRequest;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: UsersFileCreator.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class UsersFileCreator {
	protected BufferedReader in;
	protected Project project;

	public UsersFileCreator() {
		in = new BufferedReader(new InputStreamReader(System.in));
	}

	public UsersFileCreator(Project project) {
		this.project = project;
	}

	public String getInput(String prompt) throws IOException {
		if (in != null) {
			System.out.print(prompt);
			return in.readLine().trim();
		} else {
			InputRequest request = new InputRequest(prompt);
			project.getInputHandler().handleInput(request);
			String value = request.getInput();
			return value.trim();
		}
	}

	public String getUserInput(String prompt, String[] posValues)
			throws IOException {
		StringBuffer buf = new StringBuffer(128);
		buf.append(prompt);
		if (posValues != null) {
			buf.append(" (");
			for (int i = 0; i < posValues.length; i++) {
				if (i == 0) {
					buf.append('[');
				}
				buf.append(posValues[i]);
				if (i == 0) {
					buf.append(']');
				}

				if ((i + 1) < posValues.length)
					buf.append(",");
			}
			buf.append(" ) ");
		}
		buf.append(" >> ");

		String answer = getInput(buf.toString());
		boolean ok = true;
		do {
			if (answer.length() > 0) {
				if (posValues != null && validAnswer(answer, posValues)) {
					break;
				} else if (posValues == null) {
					// if there are no choices return the first nonempty answer
					break;
				}
			} else {
				if (posValues != null) {
					// return the default answer
					return posValues[0];
				}
			}
			answer = getInput(buf.toString());
		} while (!ok);

		return answer;
	}

	public static boolean validAnswer(String answer, String[] posValues) {
		for (int i = 0; i < posValues.length; i++) {
			if (answer.equals(posValues[i]))
				return true;
		}
		return false;
	}

	public static String[] getDbUsers(List<DBUserInfo> dbUsers, String dbID) {
		List<String> l = new LinkedList<String>();
		for (Iterator<DBUserInfo> iter = dbUsers.iterator(); iter.hasNext();) {
			DBUserInfo dbui = iter.next();
			if (dbui.dbID.equals(dbID))
				l.add(dbui.name);
		}
		String[] dbUsersArr = new String[l.size()];
		return l.toArray(dbUsersArr);
	}

	public static String[] getPrivilegeNames(List<PrivilegeInfo> privileges) {
		String[] privNames = new String[privileges.size()];
		int i = 0;
		for (Iterator<PrivilegeInfo> iter = privileges.iterator(); iter
				.hasNext();) {
			PrivilegeInfo pi = iter.next();
			privNames[i++] = pi.name;
		}
		return privNames;
	}

	public static String[] updatePrivilegeNames(String[] privNames,
			String usedPriv) {
		if (privNames.length == 1)
			return null;
		String[] npnArr = new String[privNames.length - 1];
		int idx = 0;
		for (int i = 0; i < privNames.length; i++) {
			if (!privNames[i].equals(usedPriv)) {
				npnArr[idx++] = privNames[i];
			}
		}
		return npnArr;
	}

	public static boolean testJDBCConnection(DatabaseInfo dbi,
			DBUserInfo dbUserInfo) {
		Connection con = null;
		Statement st = null;
		String testQuery = "select 2 from dual";
		try {
			if (dbi.dbType.equalsIgnoreCase("oracle")) {
				Class.forName("oracle.jdbc.driver.OracleDriver");
			} else if (dbi.dbType.equalsIgnoreCase("postgres")) {
				Class.forName("org.postgresql.Driver");
				testQuery = "select 2";
			}
			con = DriverManager.getConnection(dbi.getDBURL(), dbUserInfo.name,
					dbUserInfo.password);
			st = con.createStatement();
			ResultSet rs = st.executeQuery(testQuery);
			rs.close();
		} catch (Exception x) {
			x.printStackTrace();
			return false;
		} finally {
			if (st != null)
				try {
					st.close();
				} catch (Exception ex) { /* ignore */
				}
			if (con != null)
				try {
					con.close();
				} catch (Exception ex) { /* ignore */
				}
		}
		return true;
	}

	public void prepareUsersFile(String xmlFilename) throws IOException {
		UsersFileInfo ui = new UsersFileInfo();
		String ans = "";

		System.out.println("Preparing users.xml file...");
		System.out
				.println("------------------ Database Section -----------------");
		String[] yesNo = { "y", "n" };
		do {
			DatabaseInfo dbi = new DatabaseInfo();
			dbi.dbType = getUserInput("Please enter the database type",
					new String[] { "oracle", "postgres" });

			dbi.hostName = getUserInput(
					"In most cases, the fully qualified hostname of your gpop is the host name\n"
							+ " Please enter the database server host name",
					null);

			dbi.dbID = getUserInput(
					"The database id is expected of this format <site-name>_[mbirn|fbirn]\n"
							+ "A valid database ID is, for example, ucsd_fbirn or mgh_mbirn.\n "
							+ "The database id is used to identify different data sources by the web app.\n\n"
							+ " Please enter the database id", null);
			dbi.defaultDB = getUserInput(
					"The default database flag determines to which database the web app will be connected by default.\n\n"
							+ " Please enter if this database is the default database",
					yesNo).equals("y");

			if (dbi.dbType.equalsIgnoreCase("oracle")) {
				dbi.port = Integer
						.parseInt(getUserInput(
								"In most cases, the Oracle database connection port is 1521.\n\n"
										+ " Please enter the database server connection port",
								null));

				dbi.SID = getUserInput(
						"For most default BIRN rack Oracle installation the SID is orcl1\n\n"
								+ "Please enter the SID of your database", null);
			} else if (dbi.dbType.equalsIgnoreCase("postgres")) {
				/*
				 * dbi.port = Integer.parseInt(getUserInput( "In most cases, the
				 * Postgres database connection port is 5432.\n\n" + " Please
				 * enter the database server connection port", null));
				 */
				dbi.SID = getUserInput(
						"Please enter the name of your Postgres database", null);
			}

			ui.addDatabase(dbi);

			ans = getUserInput("\nDo you want to enter another database", yesNo);
		} while (ans.equals("y"));

		PrivilegeInfo pi = new PrivilegeInfo(
				"manageExperiment",
				"Any user with this privilege can add/or update\n"
						+ "experiment(s), enroll subjects to an experiment and/or change"
						+ "their study group");
		ui.addPrivilege(pi);
		pi = new PrivilegeInfo("manageSubject", "");
		ui.addPrivilege(pi);

		String[] dbIDs = new String[ui.databases.size()];
		int i = 0;
		for (Iterator<DatabaseInfo> iter = ui.databases.iterator(); iter
				.hasNext();) {
			DatabaseInfo dbi = iter.next();
			dbIDs[i++] = dbi.dbID;
		}

		System.out
				.println("------------------ Database Users Section -----------------");
		do {
			DBUserInfo dbui = new DBUserInfo();

			dbui.name = getUserInput(
					"Please enter the named database connection user name",
					null);
			dbui.password = getUserInput(
					"Please enter the named database connection password", null);
			dbui.dbID = getUserInput(
					"Please enter the database id this database user belongs",
					dbIDs);

			ans = getUserInput(
					"\nDo you want to test the database connection for this user",
					yesNo);
			if (ans.equals("y")) {
				// test database
				if (testJDBCConnection(ui.findDatabase(dbui.dbID), dbui)) {
					System.out
							.println("Database connection test is successful for user "
									+ dbui.name + ".");
					System.out
							.println("==============================================================");
					System.out.println();
				} else {
					continue;
				}
			}
			ui.addDBUser(dbui);
			ans = getUserInput("\nDo you want to enter another database user ",
					yesNo);
		} while (ans.equals("y"));

		System.out
				.println("------------------ Web App Mandatory Users Section -----------------\n");
		System.out
				.println("IMPORTANT: A mandatory 'admin' user with all privileges (including admin privilege for web based'");
		System.out
				.println("database/user/app configuration) will be created first with minimal user input\n");

		addAdminUsers(ui, dbIDs);
		
		System.out.println("------------------ Web App Users Section -----------------\n");
		do {
			WebUserInfo wui = new WebUserInfo();

			wui.name = getUserInput("Please enter the web user name", null);
			wui.password = getUserInput("Please enter the web user password",
					null);
			wui.dbID = getUserInput(
					"Please enter the database id for the database this web user uses",
					dbIDs);
			String[] dbUsers = getDbUsers(ui.dbUsers, wui.dbID);
			wui.dbUser = getUserInput(
					"Please enter the database user, this web user will be associated",
					dbUsers);

			if (ui.findWebUser(wui.name, wui.dbID, wui.dbUser) != null) {
				System.out
						.println("DUPLICATE webUser:"
								+ wui.name
								+ "! Please enter a unique web user name for the corresponding database!");
				continue;
			}
			ui.addWebUser(wui);
			String[] privNames = getPrivilegeNames(ui.privileges);
			do {
				ans = getUserInput(
						"\nDo you want to enter a privilege for this web user ",
						yesNo);
				if (ans.equals("y")) {
					String privName = getUserInput(
							"Please enter a privilege for this user", privNames);
					wui.addPrivilege(new PrivilegeInfo(privName, null));
					privNames = updatePrivilegeNames(privNames, privName);
				}
				if (privNames == null)
					break;
			} while (ans.equals("y"));

			ans = getUserInput("\nDo you want to enter another web user ",
					yesNo);
		} while (ans.equals("y"));

		BufferedOutputStream bout = null;
		try {
			bout = new BufferedOutputStream(new FileOutputStream(xmlFilename));
			saveAsXML(bout, ui.toXML());
		} finally {
			if (bout != null)
				try {
					bout.close();
				} catch (Exception ex) { /* ignore */
				}
		}
	}

	protected void addAdminUsers(UsersFileInfo ui, String[] dbIDs)
			throws IOException {

		for (int i = 0; i < dbIDs.length; i++) {

			WebUserInfo wui = new WebUserInfo();
			wui.name = "admin";
			wui.dbID = dbIDs[i];

			String[] dbUsers = getDbUsers(ui.dbUsers, wui.dbID);
			if (dbUsers.length == 0) {
				wui.dbUser = dbUsers[0];
			} else {
				wui.dbUser = getUserInput(
						"Please enter the database user, the admin web user for database '"
								+ wui.dbID + "' will be associated", dbUsers);
			}
			wui.password = getUserInput(
					"Please enter the admin web user password for database user '"
							+ wui.dbUser + "' for database '" + wui.dbID + "' ", null);
			String[] privNames = getPrivilegeNames(ui.privileges);
			for (int j = 0; j < privNames.length; j++) {
				wui.addPrivilege(new PrivilegeInfo(privNames[j], null));
			}
			if ( ui.findWebUser(wui.name, wui.dbID, wui.dbUser) == null ) {
				ui.addWebUser(wui);
			}
		}
	}

	protected void saveAsXML(OutputStream out, Element parent)
			throws IOException {
		Format format = Format.getPrettyFormat();
		format.setLineSeparator(System.getProperty("line.separator"));
		XMLOutputter xmlOut = new XMLOutputter(format);
		xmlOut.output(parent, out);
	}

	public static void main(String[] args) {
		try {
			UsersFileCreator inst = new UsersFileCreator();
			inst.prepareUsersFile("/home/bozyurt/users.xml");
		} catch (Exception x) {
			x.printStackTrace();
		}
	}

}
