package clinical.web;

import java.util.List;
import java.util.Map;

import clinical.server.vo.Brainsegmentdata;
import clinical.web.common.UserInfo;
import clinical.web.common.query.Operator;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.vo.SubjectDerivedDataStats;
import clinical.web.vo.SubjectDerivedDataValueSummary;

/**
 * The session facade interface for subcortical derived data retrieval related
 * data services.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: IDerivedDataService.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public interface IDerivedDataService {

	/**
	 * Returns all available subcortical variable type information from the
	 * persistent store.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @return a list of <code>Brainsegmentdata</code> value objects
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.server.vo.Brainsegmentdata
	 */
	public List<Brainsegmentdata> getAllSubCorticalVariables(UserInfo userInfo) throws Exception;

	/**
	 * Returns the corresponding subcortical volume measurements for the given
	 * subject set and subcortical variable list which matches the encoded
	 * search criteria in the operator tree.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param subjectList
	 *            a list subject BIRN IDs
	 * @param derivedDataList
	 *            a list of <code>SubCorticalVarInfo</code> value objects
	 *            holding the subcortical variables for which the the values for
	 *            each subject will be retrieved.
	 * @param rootOp
	 *            the root node of the search criteria operator tree.
	 * @return a list <code>SubjectDerivedDataValueSummary</code> objects
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.web.vo.SubjectDerivedDataValueSummary
	 */
	public List<SubjectDerivedDataValueSummary> getSubCorticalValuesForSubjects(
			UserInfo userInfo, List<String> subjectList,
			List<SubCorticalVarInfo> derivedDataList, Operator rootOp)
			throws Exception;

	/**
	 * Returns the corresponding subcortical volume measurements for all
	 * subjects given a subcortical variable list which matches the encoded
	 * search criteria in the operator tree.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param dbID
	 *            a the unique id of the database to connect
	 * @param derivedDataList
	 *            a list of <code>SubCorticalVarInfo</code> value objects
	 *            holding the subcortical variables for which the the values for
	 *            each subject will be retrieved.
	 * @param rootOp
	 *            the root node of the search criteria operator tree.
	 * @return a list <code>SubjectDerivedDataValueSummary</code> objects
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.web.vo.SubjectDerivedDataValueSummary
	 */

	public List<SubjectDerivedDataValueSummary> getSubCorticalValuesForSubjects(
			UserInfo userInfo, String dbID,
			List<SubCorticalVarInfo> derivedDataList, Operator rootOp)
			throws Exception;

	/**
	 * Returns the corresponding subcortical volume measurements for the given
	 * subject set and subcortical variable list which matches the encoded
	 * search criteria in the operator tree using the mediator
	 * 
	 * @param subjectList
	 *            a list subject BIRN IDs
	 * @param derivedDataList
	 *            a list of <code>SubCorticalVarInfo</code> value objects
	 *            holding the subcortical variables for which the the values for
	 *            each subject will be retrieved.
	 * @param rootOp
	 *            the root node of the search criteria operator tree.
	 * @return a list <code>SubjectDerivedDataValueSummary</code> objects
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.web.vo.SubjectDerivedDataValueSummary
	 */

	public List<SubjectDerivedDataValueSummary> getMediatedSubCorticalValuesForSubjects(
			List<String> subjectList, List<SubCorticalVarInfo> derivedDataList,
			Operator rootOp) throws Exception;

	/**
	 * Returns the corresponding subcortical volume measurements for the given
	 * subcortical variable list which matches the encoded search criteria in
	 * the operator tree using the mediator.
	 * 
	 * @param derivedDataList
	 *            a list of <code>SubCorticalVarInfo</code> value objects
	 *            holding the subcortical variables
	 * @param rootOp
	 *            the root node of the search criteria operator tree.
	 * @return a list <code>SubjectDerivedDataValueSummary</code> objects
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.web.vo.SubjectDerivedDataValueSummary
	 */
	public List<SubjectDerivedDataValueSummary> getMediatedSubCorticalValues(
			List<SubCorticalVarInfo> derivedDataList, Operator rootOp)
			throws Exception;

	/**
	 * Returns all the available subcortical brain region volumes for a given
	 * subject participating in a given experiment.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param subjectID
	 *            the BIRN ID of the subject
	 * @param experimentID
	 *            the ID of the <code>clinical.server.vo.Experiment</code>
	 *            object
	 * @return an association table containing volume measurements keyed by
	 *         laterality (left or right) followed by "_" followed by brain
	 *         regionName.
	 * @throws java.lang.Exception
	 */
	public Map<String, Float> getAllSubcorticalRegionVolumesForSubject(UserInfo userInfo,
			String subjectID, int experimentID) throws Exception;

	/**
	 * Returns all the available subcortical brain region volumes and min, max,
	 * average and standard deviation of the whole population for the experiment
	 * for a given subject participating in a given experiment.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param subjectID
	 *            the BIRN ID of the subject
	 * @param experimentID
	 *            the ID of the <code>clinical.server.vo.Experiment</code>
	 *            object
	 * @return a <code>SubjectDerivedDataStats</code> containing the
	 *         subcortical volumes and the descriptive statistics relative to
	 *         the whole population.
	 * @throws java.lang.Exception
	 * @see clinical.web.vo.SubjectDerivedDataStats
	 */
	public SubjectDerivedDataStats getAllSubcorticalVolumeStatsForSubject(
			UserInfo userInfo, String subjectID, int experimentID)
			throws Exception;

	/**
	 * Returns all the available subcortical brain region volumes and min, max,
	 * average and standard deviation of the population as specified by the
	 * <code>SubjectDerivedDataStats</code> for the experiment for a given
	 * subject participating in a given experiment.
	 * 
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param subjectID
	 *            the BIRN ID of the subject
	 * @param experimentID
	 *            the ID of the <code>clinical.server.vo.Experiment</code>
	 *            object
	 * @param subjectsInPop
	 *            list of BIRN IDS representing a population on which the
	 *            descriptive classifications are calculated
	 * @return a <code>SubjectDerivedDataStats</code> containing the
	 *         subcortical volumes and the descriptive statistics relative to
	 *         population denoted by <code>subjectsInPop</code> argument
	 * @throws java.lang.Exception
	 * 
	 * @see clinical.web.vo.SubjectDerivedDataStats
	 */
	public SubjectDerivedDataStats getAllSubcorticalVolumeStatsForSubject(
			UserInfo userInfo, String subjectID, int experimentID,
			String[] subjectsInPop) throws Exception;

}
