package clinical.web;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.List;
import java.util.Map;

import clinical.server.vo.Assessment;
import clinical.server.vo.Storedassessment;
import clinical.web.common.UserInfo;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.helpers.MissingEntry;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ReconScoreValueInfo;
import clinical.web.vo.ScoreValue;

/**
 * The session facade interface for the management of the assessments for a
 * subject.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ISubjectAssessmentManagement.java,v 1.24 2007/04/12 19:40:06
 *          bozyurt Exp $
 */
public interface ISubjectAssessmentManagement {
	/**
	 * Returns the clinical assessments for a given subject and clinical
	 * experiment.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @param experimentID
	 *           the ID of the experiment the subject is involved in
	 * @return a list of value objects of type
	 *         {@link clinical.web.vo.AssessmentInfo}
	 * @throws SubjectAssessmentManagementException
	 * 
	 * @see clinical.server.vo.Assessment
	 * @see clinical.web.vo.AssessmentInfo
	 */
	public List<AssessmentInfo> getAssessmentsForSubject(UserInfo ui,
			String subjectID, int experimentID)
			throws SubjectAssessmentManagementException;

	public List<AssessmentInfo> getAssessmentsForSubject(Connection con,
			UserInfo ui, String subjectID, int experimentID) throws Exception;

	/**
	 * 
	 * Returns the values of clinical assessments for a given subject and
	 * clinical experiment and a list of assessments.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @param experimentID
	 *           the ID of the experiment the subject is involved in
	 * @param assessments
	 *           a list of {@link clinical.web.vo.AssessmentInfo} objects
	 * @param entryID
	 *           if less than 0 disregard
	 * @param validated
	 * @return a list of {@link clinical.web.vo.AssessmentScoreValues} objects
	 * @throws SubjectAssessmentManagementException
	 * @see #getAssessmentsForSubject
	 */
	public List<AssessmentScoreValues> getAssessmentValuesForSubject(
			UserInfo ui, String subjectID, int experimentID,
			List<AssessmentInfo> assessments, int entryID, boolean validated)
			throws SubjectAssessmentManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @param experimentID
	 *           the ID of the experiment the subject is involved in
	 * @param visitID
	 *           the visit id
	 * @param segmentID
	 *           the id of the visit segment
	 * @param assessmentID
	 * @return
	 * @throws SubjectAssessmentManagementException
	 */
	public java.util.Date getTimeStampForAssessment(UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID,
			int assessmentID) throws SubjectAssessmentManagementException;

	/**
	 * Returns the values of clinical assessment scores for a given subject,
	 * clinical experiment, visit and segment and list of assessments.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           subjectID the BIRN ID of the subject
	 * @param experimentID
	 *           the ID of the experiment the subject is involved in
	 * @param visitID
	 *           the visit ID
	 * @param segmentID
	 *           the id of the visit segment
	 * @param assessments
	 *           a list of {@link clinical.web.vo.AssessmentInfo} objects.
	 * @param entryID
	 *           entry ID for the set of values to be returned (id -1 in
	 *           combination with validated flag set to true means to retrieve
	 *           the validated version
	 * @param validated
	 *           if set to true returns only validated (reconciled) assessment
	 *           values
	 * @return a list of {@link clinical.web.vo.AssessmentScoreValues} objects
	 * @throws SubjectAssessmentManagementException
	 * 
	 * @see clinical.web.vo.AssessmentScoreValues
	 * @see #getAssessmentsForSubject
	 */
	public List<AssessmentScoreValues> getAssessmentValuesForSubject(
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, List<AssessmentInfo> assessments, int entryID,
			boolean validated) throws SubjectAssessmentManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param dbID
	 * @param asv
	 * @param dbVarMetaDataMap
	 *           keyed by the score name each entry holds a hashtable of metadata
	 *           name string - value string pairs
	 * @throws SubjectAssessmentManagementException
	 * @see clinical.web.vo.AssessmentScoreValues
	 */
	public void insertAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	public void insertAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv, BigDecimal entryID, boolean validated,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	public void insertAssessmentValues(Connection con, UserInfo ui, String dbID,
			AssessmentScoreValues asv, BigDecimal entryID, boolean validated,
			Map<String, Map<String, String>> dbVarMetaDataMap) throws Exception;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param dbID
	 * @param asv
	 * @param scoresToBeUpdated
	 * @param entryID
	 *           the id of the assessment entry
	 * @throws SubjectAssessmentManagementException
	 */
	public void updateAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv, List<String> scoresToBeUpdated,
			int entryID, Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	/**
	 * 
	 * @param dbID
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param asv
	 * @param scoreValues
	 * @param entryID
	 * @param validated
	 * @param dbVarMetaDataMap
	 *           keyed by the score name each entry holds a hashtable of metadata
	 *           name string - value string pairs
	 * @throws SubjectAssessmentManagementException
	 */
	public void insertScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues,
			BigDecimal entryID, boolean validated,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	public void insertScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	public void deleteScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues, int entryID)
			throws SubjectAssessmentManagementException;

	/**
	 * 
	 * @param dbID
	 * @param ui
	 * @param informantID
	 * @param informantRelation
	 * @param asv
	 *           {@link clinical.web.vo.AssessmentScoreValues} object
	 * @throws SubjectAssessmentManagementException
	 */
	public void persisInformantInfo(String dbID, UserInfo ui,
			String informantID, String informantRelation, AssessmentScoreValues asv)
			throws SubjectAssessmentManagementException;

	public List<Storedassessment> getStoredAssessmentRecs(String dbID,
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException;

	/**
	 * executes an SQL query returning a list of columns values that can be
	 * converted to string and returns the result as a list of
	 * <code>String</code> objects.
	 * 
	 * @param ui
	 *           UserInfo
	 * @param sqlQuery
	 *           String
	 * @throws SubjectAssessmentManagementException
	 * @return List
	 */
	public List<String> getDynamicQueryResults(UserInfo ui, String sqlQuery)
			throws SubjectAssessmentManagementException;

	public Timestamp getSegmentTimeStamp(String dbID, UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID)
			throws SubjectAssessmentManagementException;

	/**
	 * 
	 * @param dbID
	 *           String
	 * @param ui
	 *           UserInfo
	 * @param asv
	 *           {@link clinical.web.vo.AssessmentScoreValues} object
	 * 
	 * @param reconMap
	 *           {@link clinical.web.vo.ReconScoreValueInfo} objects keyed by
	 *           scoreName
	 * @throws SubjectAssessmentManagementException
	 */
	public void saveReconciledAssessmentData(String dbID, UserInfo ui,
			AssessmentScoreValues asv, Map<String, ReconScoreValueInfo> reconMap,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException;

	public Map<String, ReconScoreValueInfo> getReconciliationDataForAssessment(
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException;

	public boolean isAssessmentReadyForReconcile(UserInfo ui, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException;

	public boolean isAssessmentReconciled(UserInfo ui, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException;

	public int getNumOfEntries(UserInfo ui, String subjectID, int experimentID,
			int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException;

	public List<MissingEntry> getMissingEntryItems(UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID,
			int assessmentID) throws SubjectAssessmentManagementException;

	public Assessment getAssessment(UserInfo ui, String asName)
			throws SubjectAssessmentManagementException;

	/**
	 * persists an assessment with its scores (excluding scorecodes and question
	 * info). Mainly used for tablet input.
	 * 
	 * @param ui
	 * @param asInfo
	 * @throws SubjectAssessmentManagementException
	 */
	public Assessment addAssessment(UserInfo ui, AssessmentInfo asInfo)
			throws SubjectAssessmentManagementException;
}
