package clinical.web.download;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import clinical.utils.GenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SubjectImageDataBundle.java,v 1.5 2008/04/25 19:36:01 bozyurt
 *          Exp $
 */
public class SubjectImageDataBundle implements Serializable {
	private static final long serialVersionUID = -2472430723916394827L;
	protected String subjectID;
	protected String siteID;
	protected String treeId;
	protected DataNode root;
	protected List<String> expIDs = new ArrayList<String>(1);
	protected Set<String> segmentNameSet = new HashSet<String>(17);
	protected Map<String, String> expId2NameMap = new HashMap<String, String>(7);

	/**
	 * Contains image paths, excluding the common rootDir, which are terminal
	 * nodes.
	 */
	protected Map<String, DataNode> pathMap = new HashMap<String, DataNode>();
	protected Map<String, List<DataNode>> segPathMap = new HashMap<String, List<DataNode>>(
			23);

	protected Map<String, DataNode> asMap = null;

	public SubjectImageDataBundle(String subjectID, String siteID, String treeId) {
		super();
		this.subjectID = subjectID;
		this.siteID = siteID;
		this.treeId = treeId;
		root = new DataNode(subjectID, subjectID, "subject");
	}

	public DataNode addVisit(String visitID) {
		DataNode child = new DataNode(visitID, visitID, "visit");
		root.addChild(child);
		return child;
	}

	public void removeSegmentUnless(Set<String> runTypeSet) {
		if (root == null)
			return;
		removeSegmentUnless(root, runTypeSet);
	}

	protected void removeSegmentUnless(DataNode parent, Set<String> runTypeSet) {
		if (parent == null || !parent.hasChildren())
			return;
		List<DataNode> toBeRemoved = new ArrayList<DataNode>(5);
		for (DataNode child : parent.getChildren()) {
			if (child.getUserObject() instanceof ObjectInfo) {
				ObjectInfo oi = (ObjectInfo) child.getUserObject();

				if (oi.getType().equals("segment")) {
					if (!runTypeSet.contains(oi.getLabel())) {
						boolean found = false;
						for (String runType : runTypeSet) {
							if (oi.getLabel().startsWith(runType)) {
								found = true;
								break;
							}
						}
						if (found) {
							continue;
						}
						toBeRemoved.add(child);
					}
				}
			}
		}
		if (toBeRemoved.isEmpty()) {
			for (DataNode child : parent.getChildren()) {
				removeSegmentUnless(child, runTypeSet);
			}
		} else {
			for (DataNode dn : toBeRemoved) {
				parent.removeChild(dn);
			}
			return;
		}
	}

	public DataNode addObject(ObjectInfo oi, DataNode parent) {
		if (parent == null)
			return null;
		if (oi.getType().equals("segment")) {
			segmentNameSet.add(oi.getLabel());
		} else if (oi.getType().equals("experiment")) {
			expId2NameMap.put(oi.getId(), oi.getLabel());
		}
		DataNode child = new DataNode(oi.getLabel(), oi, "object");
		parent.addChild(child);
		return child;
	}

	public String getExperimentName(String expID) {
		return expId2NameMap.get(expID);
	}

	public DataNode addSubDir(String dirName, DataNode parent) {
		if (parent == null)
			return null;
		DataNode child = new DataNode(dirName, dirName, "dir");
		parent.addChild(child);
		return child;
	}

	public void addFiles(List<FileInfo> fiList, DataNode parentDir,
			boolean useRelativePath) {
		for (FileInfo fi : fiList) {
			addFile(fi, parentDir, useRelativePath);
		}
	}

	public void addFile(FileInfo fi, DataNode parentDir, boolean useRelativePath) {
		DataNode child = null;
		if (useRelativePath) {
			child = new DataNode(fi.getRelativePath(), fi, "file");
		} else {
			child = new DataNode(fi.getPath(), fi, "file");
		}
		parentDir.addChild(child);
	}

	public DataNode findByPath(String path) {
		String[] parts = path.split("\\/");
		if (!parts[0].equals(subjectID)) {
			return null;
		}
		// first four levels are subjectid , experiment, visit , segment|
		// assessment
		if (pathMap.isEmpty()) {
			preparePathMap(root);
		}
		String key = GenUtils.join(parts, "/", 4, parts.length);
		return (DataNode) pathMap.get(key);
	}

	public List<DataNode> findBySegment(String birnID, String expName,
			int visitID, String segName) {
		if (!birnID.equals(subjectID)) {
			return null;
		}
		if (segPathMap == null)
			return null;
		if (segPathMap.isEmpty()) {
			prepareSegPathMap(root);
			if (segPathMap.isEmpty()) {
				segPathMap = null;
				return null;
			}
		}
		StringBuilder sb = new StringBuilder();
		sb.append(birnID).append(':');
		sb.append(expName).append(":Visit ").append(visitID);
		sb.append(':').append(segName);
		String key = sb.toString();

		return segPathMap.get(key);
	}

	protected void prepareSegPathMap(DataNode parent) {
		String type = parent.getType();
		if ((type.equals("file") || type.equals("dir"))
				&& !parent.hasChildren()) {
			String segKey = prepareSegmentKey(parent);
			List<DataNode> list = segPathMap.get(segKey);
			if (list == null) {
				list = new ArrayList<DataNode>(1);
				segPathMap.put(segKey, list);
			}
			list.add(parent);
		}
		if (parent.hasChildren()) {
			for (DataNode child : parent.getChildren()) {
				prepareSegPathMap(child);
			}
		}

	}

	protected String prepareSegmentKey(DataNode parent) {
		StringBuilder sb = new StringBuilder();
		Stack<String> stack = new Stack<String>();
		DataNode p = parent;
		while (p != null && p.getType().equals("file")
				|| p.getType().equals("dir")) {
			p = p.getParent();
		}
		while (p != null) {
			stack.push(p.getTitle());
			p = p.getParent();
		}
		while (!stack.isEmpty()) {
			String s = stack.pop();
			sb.append(s);
			if (!stack.isEmpty())
				sb.append(':');
		}
		return sb.toString();
	}

	public List<DataNode> getAllPaths() {
		if (pathMap.isEmpty()) {
			preparePathMap(root);
		}
		List<DataNode> pathNodes = new ArrayList<DataNode>(pathMap.values());
		return pathNodes;
	}

	public String extractExperimentName(String path) {
		String[] parts = path.split("\\/");
		if (!parts[0].equals(subjectID)) {
			return null;
		}
		return parts[1];
	}

	protected void preparePathMap(DataNode parent) {
		String type = parent.getType();
		if ((type.equals("file") || type.equals("dir"))
				&& !parent.hasChildren()) {
			String path = parent.buildFilePath();
			pathMap.put(path, parent);
		}
		if (parent.hasChildren()) {
			for (DataNode child : parent.getChildren()) {
				preparePathMap(child);
			}
		}
	}

	public DataNode findByTitle(String title) {
		if (asMap == null) {
			asMap = new HashMap<String, DataNode>(11);
			prepareAsMap(root);
		}
		return (DataNode) asMap.get(title);
	}

	protected void prepareAsMap(DataNode parent) {
		String type = parent.getType();
		if (type.equals("object")) {
			ObjectInfo oi = (ObjectInfo) parent.getUserObject();
			if (oi.getType().equals("assessment")) {
				asMap.put(oi.getLabel(), parent);
			}
		}
		if (parent.hasChildren()) {
			for (DataNode child : parent.getChildren()) {
				prepareAsMap(child);
			}
		}
	}

	public String getJavascriptTree() {
		return buildJSTree();
	}

	public String buildJSTree() {
		StringBuffer buf = new StringBuffer(2048);
		buf.append("var ").append(treeId).append(" = new dTree('").append(
				treeId).append("'); ").append("\n");
		buf.append(treeId).append(".config.useIcons=false; ");
		buf.append(treeId).append(".config.useCookies=false; ").append('\n');
		IDGenerator idGen = new IDGenerator();
		build(root, idGen, -1, buf);
		buf.append("document.write(").append(treeId).append("); ");
		buf.append(treeId).append(".registerCBObservers();\n");
		buf.append(treeId).append(
				".registerCBEventListener(totalSizeHandler);\n");

		return buf.toString();
	}

	protected void build(DataNode node, IDGenerator idGen, int parentId,
			StringBuffer buf) {
		int nodeId = idGen.nextID();
		buf.append(treeId).append(".add(").append(nodeId).append(',').append(
				parentId);
		buf.append(",'").append(node.getTitle()).append("'");
		if (node.getParent() != null
				&& node.getParent().getTitle().toLowerCase().startsWith(
						"assessment")) {
			buf.append(",'ca');\n");
		} else {
			if (node.getUserObject() instanceof FileInfo) {
				FileInfo fi = (FileInfo) node.getUserObject();
				buf.append(",null,").append(fi.getFileSize()).append(");\n");
			} else {
				buf.append(");\n");
			}
		}
		if (node.hasChildren()) {
			for (DataNode child : node.getChildren()) {
				build(child, idGen, nodeId, buf);
			}
		}
	}

	public String getTreeId() {
		return treeId;
	}

	public String getSiteID() {
		return siteID;
	}

	public String getSubjectID() {
		return subjectID;
	}

	public DataNode getRoot() {
		return root;
	}

	public void addExpID(String expID) {
		if (!expIDs.contains(expID)) {
			expIDs.add(expID);
		}
	}

	public List<String> getExpIds() {
		return expIDs;
	}

	public boolean isEmpty() {
		return (root == null || !root.hasChildren());
	}

	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((expIDs == null) ? 0 : expIDs.hashCode());
		result = PRIME * result + ((siteID == null) ? 0 : siteID.hashCode());
		result = PRIME * result
				+ ((subjectID == null) ? 0 : subjectID.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final SubjectImageDataBundle other = (SubjectImageDataBundle) obj;
		if (expIDs == null) {
			if (other.expIDs != null)
				return false;
		} else if (!expIDs.equals(other.expIDs))
			return false;
		if (siteID == null) {
			if (other.siteID != null)
				return false;
		} else if (!siteID.equals(other.siteID))
			return false;
		if (subjectID == null) {
			if (other.subjectID != null)
				return false;
		} else if (!subjectID.equals(other.subjectID))
			return false;
		return true;
	}

	public synchronized Set<String> getSegmentNameSet() {
		return segmentNameSet;
	}

}
