package clinical.web.helpers;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: SearchResultsIterator.java 62 2009-05-29 23:54:50Z bozyurt $
 */

public class SearchResultsIterator implements Serializable{
	private static final long serialVersionUID = 1L;
	private int windowSize;
	private int startIdx = 0;
	private List<?> data;
	private ListIterator<?> iter;
	private int idx = 0;
	private int globIdx = 0;
	private boolean atEnd;

	public SearchResultsIterator(List<?> data, int windowSize) {
		this.data = data;
		iter = data.listIterator();
		this.windowSize = windowSize;
	}

	/**
	 * return the total number of results
	 */
	public int getSize() {
		if (data != null)
			return data.size();
		else
			return 0;
	}

	public boolean hasNext() {
		if (iter == null) {
			iter = data.listIterator();
		}
		if (idx >= windowSize) {
			idx = 0;
			return false;
		}
		boolean d = iter.hasNext();
		atEnd = !d;
		return d;
	}

	public Object next() {
		Object o = iter.next();
		++idx;
		++globIdx;
		return o;
	}

	public Object prev() {
		Object o = iter.previous();
		--idx;
		--globIdx;
		return o;
	}

	/**
	 * returns true if we are currently at the first window of results
	 *
	 * @return true if we are currently at the first window of results
	 */
	public boolean isAtStart() {
		return (globIdx <= windowSize);
	}

	/**
	 * returns true if we are currently at last window of results
	 *
	 * @return true if we are currently at last window of results
	 */
	public boolean isAtEnd() {
		return globIdx + windowSize >= data.size();
	}

	public int getNumberOfWindows() {
		int numWindows = (int) Math.ceil(this.data.size()
				/ (double) this.windowSize);
		return numWindows;
	}

	public int getCurrentWindow() {
		int curWindow = this.globIdx / this.windowSize + 1;
		return curWindow;
	}

	/**
	 * returns the windowSize elements of the underlying result list starting
	 * from the current position of the iterator
	 *
	 * @return the windowSize elements of the underlying result list starting
	 *         from the current position of the iterator
	 */
	public List<Object> getCurrentWindowContents() {
		List<Object> wList = new ArrayList<Object>();
		while (hasNext()) {
			wList.add(next());
		}
		boolean b = this.atEnd;
		// roll back to the current window start
		moveToPrevWindow();
		if (b) {
			this.atEnd = true;
		}
		// log.debug("wlist size="+ wList.size());
		return wList;
	}

	public void reset() {
		iter = null;
		iter = data.listIterator();
	}

	public void moveToPrevWindow() {
		int i = 0;
		int wsize = windowSize;
		// the last window may have less items than window size
		if (!iter.hasNext()) {
			// in case the last window is also the only window
			if (data.size() == windowSize) {
				wsize = windowSize;
			} else {
				wsize = data.size() % windowSize;
			}
		}
		while (iter.hasPrevious()) {
			if (i >= wsize)
				break;
			prev();
			++i;
		}
		idx = 0;
	}

	public void moveToNextWindow() {
		int i = 0;
		while (iter.hasNext()) {
			if (i >= windowSize)
				break;
			next();
			++i;
		}
		idx = 0;
	}

	/**
	 *
	 * @param windowNumber
	 *            the one based window index
	 */
	public void moveToWindowAt(int windowNumber) {
		int nextIdx = iter.nextIndex();
		int wStartGlobIdx = (windowNumber - 1) * windowSize;
		if (nextIdx == -1 || nextIdx > wStartGlobIdx) {
			// we are at the end, so move back
			rewindTo(wStartGlobIdx);
		} else if (nextIdx <= wStartGlobIdx) {
			forwardTo(wStartGlobIdx);
		}

		this.idx = 0;
		this.globIdx = wStartGlobIdx;

	}

	protected void rewindTo(int posIdx) {
		while (iter.previousIndex() > (posIdx - 1)) {
			iter.previous();
		}
	}

	protected void forwardTo(int posIdx) {
		while (iter.nextIndex() < posIdx) {
			iter.next();
		}
	}

	// ---------------------- setters --------------
	public void setStartIdx(int newStartIdx) {
		this.startIdx = newStartIdx;
		idx = 0;
	}

	// ---------------------- getters --------------
	public int getWindowSize() {
		return this.windowSize;
	}

	public int getStartIdx() {
		return this.startIdx;
	}

	public int getWindowStartIdx() {
		return globIdx + 1;
	}

	public int getWindowEndIdx() {
		if (isAtEnd())
			return data.size();
		else
			return globIdx + windowSize;
	}

	static void dumpList(List<Object> l) {
		System.out.println("list dump ");
		for (Iterator<Object> it = l.iterator(); it.hasNext();) {
			System.out.println(it.next().toString());
		}
		System.out.println("-------");
	}

	public static void main(String[] args) {
		final int SIZE = 62;
		List<Integer> list = new ArrayList<Integer>(SIZE);
		for (int i = 0; i < SIZE; i++) {
			list.add(new Integer(i));
		}
		SearchResultsIterator sri = new SearchResultsIterator(list, 20);
		List<Object> wl = sri.getCurrentWindowContents();
		dumpList(wl);
		// simulating visiting of the same page again
		System.out.println("second visit of the same list");
		wl = sri.getCurrentWindowContents();
		dumpList(wl);

		System.out.println("num of windows= " + sri.getNumberOfWindows());

		System.out.println("moving to window 4");
		sri.moveToWindowAt(4);
		wl = sri.getCurrentWindowContents();
		dumpList(wl);

		System.out.println("moving to window 2");
		sri.moveToWindowAt(2);
		wl = sri.getCurrentWindowContents();
		dumpList(wl);

		/*
		 * sri.moveToNextWindow(); wl = sri.getCurrentWindowContents();
		 * System.out.println("next window"); dumpList(wl);
		 * System.out.println("end=" + sri.getWindowEndIdx());
		 * sri.moveToPrevWindow(); System.out.println("prev window"); wl =
		 * sri.getCurrentWindowContents(); dumpList(wl);
		 */
	}
}
