package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.DatabaseuserDAO;
import clinical.server.vo.Assessmentboolean;
import clinical.server.vo.Assessmentfloat;
import clinical.server.vo.Assessmentinteger;
import clinical.server.vo.Assessmenttimestamp;
import clinical.server.vo.Assessmentvarchar;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Dataclassification;
import clinical.server.vo.Userclass;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.services.AbstractSubjectAssessmentManImpl.Params;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;

public class SubjectAssessmentManHelper {
	protected static Log log = LogFactory
			.getLog(SubjectAssessmentManHelper.class);

	public static String findAssessmentDataTablename(String scoreType) {
		if (scoreType.equals("integer")) {
			return Constants.ASSESSMENT_INTEGER_DB_TABLE;
		} else if (scoreType.equals("varchar")) {
			return Constants.ASSESSMENT_VARCHAR_DB_TABLE;
		} else if (scoreType.equals("float")) {
			return Constants.ASSESSMENT_FLOAT_DB_TABLE;
		} else if (scoreType.equals("boolean")) {
			return Constants.ASSESSMENT_BOOLEAN_DB_TABLE;
		} else if (scoreType.equals("timestamp")) {
			return Constants.ASSESSMENT_TIMESTAMP_DB_TABLE;
		} else {
			throw new RuntimeException("Not a supported score type:"
					+ scoreType);
		}
	}

	public static Assessmentinteger prepareIntegerScoreValueCriteria(
			AssessmentInfo.ScoreInfo si, BigDecimal storedAsID,
			int assessmentID, int entryID, int scoreOrder) {
		Assessmentinteger ai = new Assessmentinteger();
		ai.setAssessmentid(new BigDecimal(String.valueOf(assessmentID)));
		ai.setStoredassessmentid(storedAsID);
		ai.setScorename(si.getName());
		if (entryID >= 1) {
			ai.setEntryid(GenUtils.toBigDecimal(entryID));
		}
		ai.setScoreorder(GenUtils.toBigDecimal(scoreOrder));
		return ai;
	}

	public static Assessmentfloat prepareFloatScoreValueCriteria(
			AssessmentInfo.ScoreInfo si, BigDecimal storedAsID,
			int assessmentID, int entryID, int scoreOrder) {
		Assessmentfloat af = new Assessmentfloat();
		af.setAssessmentid(new BigDecimal(String.valueOf(assessmentID)));
		af.setStoredassessmentid(storedAsID);
		af.setScorename(si.getName());
		if (entryID >= 1) {
			af.setEntryid(GenUtils.toBigDecimal(entryID));
		}
		af.setScoreorder(GenUtils.toBigDecimal(scoreOrder));
		return af;
	}

	public static Assessmentvarchar prepareVarcharScoreValueCriteria(
			AssessmentInfo.ScoreInfo si, BigDecimal storedAsID,
			int assessmentID, int entryID, int scoreOrder) {
		Assessmentvarchar av = new Assessmentvarchar();
		av.setAssessmentid(new BigDecimal(String.valueOf(assessmentID)));
		av.setStoredassessmentid(storedAsID);
		av.setScorename(si.getName());
		if (entryID >= 1) {
			av.setEntryid(GenUtils.toBigDecimal(entryID));
		}
		av.setScoreorder(GenUtils.toBigDecimal(scoreOrder));
		return av;
	}

	public static Assessmentboolean prepareBooleanScoreValueCriteria(
			AssessmentInfo.ScoreInfo si, BigDecimal storedAsID,
			int assessmentID, int entryID, int scoreOrder) {
		Assessmentboolean ab = new Assessmentboolean();
		ab.setAssessmentid(new BigDecimal(String.valueOf(assessmentID)));
		ab.setStoredassessmentid(storedAsID);
		ab.setScorename(si.getName());
		if (entryID >= 1) {
			ab.setEntryid(GenUtils.toBigDecimal(entryID));
		}
		ab.setScoreorder(GenUtils.toBigDecimal(scoreOrder));
		return ab;
	}

	public static Assessmenttimestamp prepareTimestampScoreValueCriteria(
			AssessmentInfo.ScoreInfo si, BigDecimal storedAsID, int assessmentID) {
		Assessmenttimestamp at = new Assessmenttimestamp();
		at.setAssessmentid(new BigDecimal(String.valueOf(assessmentID)));
		at.setStoredassessmentid(storedAsID);
		at.setScorename(si.getName());

		return at;
	}

	public static int findClinicalRaterID(AssessmentScoreValues asv,
			Connection con, String dbID, UserInfo ui) throws Exception {
		return findClinicalRaterID(ui, dbID, con, asv.getClinicalRater());
	}

	public static int findClinicalRaterID(UserInfo ui, String dbID,
			Connection con, String clinicalRater) throws Exception {
		int raterID = -1;
		if (clinicalRater != null) {
			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Userclass> ucList = dbCache.getUserClasses(ui, false);
			Userclass raterUC = null;
			for (Userclass uc : ucList) {
				if (uc.getUserclass().equals(Constants.USERCLASS_RESEARCHER)) {
					raterUC = uc;
					break;
				}
			}
			assert (raterUC != null);

			// ASSUMPTION: clinical rater data comes from nc_databaseuser table
			DatabaseuserDAO dao = DAOFactory.createDatabaseuserDAO(dbID);
			Databaseuser criterion = new Databaseuser();
			criterion.setName(clinicalRater.toUpperCase());
			criterion.setIsgroup(new Boolean(false));
			criterion.setUserclass(raterUC.getUniqueid());

			List<Databaseuser> dbUsers = dao.find(con, criterion);
			if (!dbUsers.isEmpty()) {
				Databaseuser rater = dbUsers.get(0);
				raterID = rater.getUniqueid().intValue();
			} else {
				log.error("No matching clinical rater id for "
						+ clinicalRater.toUpperCase());
			}
		}
		return raterID;
	}

	public static Integer toInteger(BigDecimal bd) {
		if (bd == null) {
			return null;
		}
		return new Integer(String.valueOf(bd.toString()));
	}

	public static BigDecimal getClassificationID(UserInfo ui, String dbID,
			String classificationLabel) throws BaseException {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		List<Dataclassification> dcList = dbCache.getDataClassications(ui,
				false);
		for (Dataclassification dc : dcList) {
			if (dc.getName().equals(classificationLabel)) {
				return dc.getUniqueid();
			}
		}
		return null;
	}

	public static Assessmenttimestamp prepareTimestampScoreValueRecord(
			Params params, Map<String, Map<String, String>> dbVarMetaDataMap)
			throws ParseException, Exception {
		Assessmenttimestamp at = new Assessmenttimestamp();
		at.setNcAssessmentdataUniqueid(params.uniqueID);
		at.setUniqueid(params.uniqueID);
		at.setAssessmentid(new BigDecimal(String.valueOf(params.assessmentID)));
		at.setTableid(params.tableID);
		at.setScoreorder(new BigDecimal(String.valueOf(params.sv
				.getScoreOrder())));
		at.setModuser(params.databaseUser.getUniqueid());
		at.setOwner(params.databaseUser.getUniqueid());
		at.setModtime(new java.util.Date());

		at.setTextnormvalue("");
		if (params.sv.getComments() != null) {
			at.setComments(params.sv.getComments());
		} else {
			at.setComments("");
		}
		checkSetMissingValue(params.sv, params.missClassificationID);
		if (params.sv.getClassification() != null) {
			at.setClassification(new BigDecimal(params.sv.getClassification()
					.toString()));
		}

		at.setDatavalue(null);

		if (params.sv.getClassification() == null) {
			if (params.sv.getUncorrectedValue() != null) {
				Map<String, String> metaDataMap = (Map<String, String>) dbVarMetaDataMap
						.get(params.sv.getName());
				assert (metaDataMap.get("format") != null);
				String tsFormat = (String) metaDataMap.get("format");
				SimpleDateFormat df = new SimpleDateFormat(tsFormat);
				df.setLenient(false);

				java.util.Date d = df.parse(params.sv.getUncorrectedValue());
				at.setDatavalue(new java.sql.Timestamp(d.getTime()));
				log.info("****** setting timestamp dataValue:"
						+ at.getDatavalue());
			}
		}

		at.setTextvalue(params.sv.getValue());
		at.setStoredassessmentid(params.storedAssessmentID);
		at.setScorename(params.si.getName());
		at.setScoretype(params.si.getType());
		/** @todo isRanked needs to be supplied instead of defaults */
		at.setIsranked(new Boolean(false));
		at.setIsvalidated(new Boolean(params.validated));
		at.setScoreorder(GenUtils.toBigDecimal(params.sv.getScoreOrder()));
		if (params.entryID != null) {
			at.setEntryid(params.entryID);
		} else {
			at.setEntryid(GenUtils.toBigDecimal(1));
		}
		return at;
	}

	/**
	 * checks if the provided scorevalue is missing and if a reason provided for
	 * it.
	 * 
	 * @param sv
	 *            ScoreValue
	 * @throws Exception
	 */
	public static void checkSetMissingValue(ScoreValue sv,
			BigDecimal missClassID) throws Exception {
		if (sv.getClassification() != null) {
			if (sv.getUncorrectedValue() == null
					|| sv.getUncorrectedValue().length() == 0) {
				sv.setValue(null);
			}
		} else {
			if (sv.getUncorrectedValue() == null
					|| sv.getUncorrectedValue().length() == 0) {
				sv.setValue(null);
				sv.setClassification(SubjectAssessmentManHelper
						.toInteger(missClassID));
				// throw new Exception("Score value for score '" + sv.getName()
				// + "' was missing without any reason provided!");
			}
		}
	}

	public static String checkAndPrepareIntScoreValue(ScoreValue sv,
			Assessmentinteger ai, BigDecimal missClassID) throws Exception {

		checkSetMissingValue(sv, missClassID);
		String value = sv.getUncorrectedValue();

		if (value != null) {
			// in case boolean value true or false convert it to 1 or 0
			// respectively
			if (value.equalsIgnoreCase("true")
					|| value.equalsIgnoreCase("false")) {
				value = (value.equalsIgnoreCase("true")) ? "1" : "0";
			}

			if (!GenUtils.isInteger(value)) {
				throw new RuntimeException("Score value for score "
						+ sv.getName() + " was not an integer:" + sv.toString());
			}
			ai.setTextvalue(value);
			ai.setDatavalue(new BigDecimal(value));
		} else {
			ai.setTextvalue(null);
			ai.setNull("textvalue", true);
			ai.setDatavalue(null);
			// indicate explicitly a null value is set, since otherwise during
			// update the corresponding SQL will not generated since implicit
			// null values
			// indicate that the property will be not updated
			ai.setNull("datavalue", true);
		}
		return value;
	}

	public static Assessmentfloat prepareFloatScoreValueRecord(Params params)
			throws Exception {
		Assessmentfloat af = new Assessmentfloat();
		af.setNcAssessmentdataUniqueid(params.uniqueID);
		af.setUniqueid(params.uniqueID);
		af.setAssessmentid(new BigDecimal(String.valueOf(params.assessmentID)));
		af.setTableid(params.tableID);
		af.setScoreorder(new BigDecimal(String.valueOf(params.sv
				.getScoreOrder())));
		af.setModuser(params.databaseUser.getUniqueid());
		af.setOwner(params.databaseUser.getUniqueid());
		af.setModtime(new java.util.Date());
		af.setTextnormvalue("");
		if (params.sv.getComments() != null) {
			af.setComments(params.sv.getComments());
		} else {
			af.setComments("");
		}
		checkSetMissingValue(params.sv, params.missClassificationID);
		if (params.sv.getClassification() != null) {
			af.setClassification(new BigDecimal(params.sv.getClassification()
					.toString()));
		}

		af.setTextvalue(params.sv.getValue());
		// for POSTGRES new Float(sv.getUncorrectedValue()) -> new
		// Double(sv.getUncorrectedValue())
		af.setDatavalue(params.sv.getUncorrectedValue() != null ? new Float(
				params.sv.getUncorrectedValue()) : null);

		log.info("###### af.dataValue=" + af.getDatavalue());
		af.setStoredassessmentid(params.storedAssessmentID);
		af.setScorename(params.si.getName());
		af.setScoretype(params.si.getType());
		/** @todo isRanked needs to be supplied instead of defaults */
		af.setIsranked(new Boolean(false));
		af.setIsvalidated(new Boolean(params.validated));
		af.setScoreorder(GenUtils.toBigDecimal(params.sv.getScoreOrder()));
		if (params.entryID != null) {
			af.setEntryid(params.entryID);
		} else {
			af.setEntryid(GenUtils.toBigDecimal(1));
		}
		return af;
	}

	public static Assessmentvarchar prepareVarcharScoreValueRecord(Params params)
			throws Exception {
		Assessmentvarchar av = new Assessmentvarchar();
		av.setNcAssessmentdataUniqueid(params.uniqueID);
		av.setUniqueid(params.uniqueID);
		av.setAssessmentid(new BigDecimal(String.valueOf(params.assessmentID)));
		av.setTableid(params.tableID);
		av.setScoreorder(new BigDecimal(String.valueOf(params.sv
				.getScoreOrder())));
		av.setModuser(params.databaseUser.getUniqueid());
		av.setOwner(params.databaseUser.getUniqueid());
		av.setModtime(new java.util.Date());

		checkSetMissingValue(params.sv, params.missClassificationID);
		if (params.sv.getClassification() != null) {
			av.setClassification(new BigDecimal(params.sv.getClassification()
					.toString()));
		}
		av.setTextvalue(params.sv.getUncorrectedValue());
		av.setTextnormvalue(null);
		if (params.sv.getComments() != null) {
			av.setComments(params.sv.getComments());
		} else {
			av.setComments("");
		}
		av.setDatavalue(params.sv.getValue());
		av.setStoredassessmentid(params.storedAssessmentID);
		av.setScorename(params.si.getName());
		av.setScoretype(params.si.getType());
		/** @todo isRanked needs to be supplied instead of defaults */
		av.setIsranked(new Boolean(false));
		av.setIsvalidated(new Boolean(params.validated));
		av.setScoreorder(GenUtils.toBigDecimal(params.sv.getScoreOrder()));

		if (params.entryID != null) {
			av.setEntryid(params.entryID);
		} else {
			av.setEntryid(GenUtils.toBigDecimal(1));
		}
		return av;
	}

	public static Assessmentboolean prepareBooleanScoreValueRecord(Params params)
			throws Exception {
		Assessmentboolean ab = new Assessmentboolean();
		ab.setNcAssessmentdataUniqueid(params.uniqueID);
		ab.setUniqueid(params.uniqueID);
		ab.setAssessmentid(new BigDecimal(String.valueOf(params.assessmentID)));
		ab.setTableid(params.tableID);
		ab.setScoreorder(new BigDecimal(String.valueOf(params.sv
				.getScoreOrder())));
		ab.setModuser(params.databaseUser.getUniqueid());
		ab.setOwner(params.databaseUser.getUniqueid());
		ab.setModtime(new java.util.Date());
		ab.setTextnormvalue("");
		if (params.sv.getComments() != null) {
			ab.setComments(params.sv.getComments());
		} else {
			ab.setComments("");
		}
		checkSetMissingValue(params.sv, params.missClassificationID);

		if (params.sv.getClassification() != null) {
			ab.setClassification(new BigDecimal(params.sv.getClassification()
					.toString()));
		}

		if (params.sv.getUncorrectedValue() != null) {
			if (!GenUtils.isBoolean(params.sv.getUncorrectedValue())) {
				throw new Exception("Score value for score "
						+ params.sv.getName() + " was not an boolean:"
						+ params.sv.toString());
			}
		}

		ab.setTextvalue(params.sv.getUncorrectedValue());
		ab.setDatavalue(params.sv.getUncorrectedValue() != null ? new Boolean(
				GenUtils.toBoolean(params.sv.getValue(), false)) : null);

		ab.setStoredassessmentid(params.storedAssessmentID);
		ab.setScorename(params.si.getName());
		ab.setScoretype(params.si.getType());
		/** @todo isRanked needs to be supplied instead of defaults */
		ab.setIsranked(new Boolean(false));
		ab.setIsvalidated(new Boolean(params.validated));
		ab.setScoreorder(GenUtils.toBigDecimal(params.sv.getScoreOrder()));
		if (params.entryID != null) {
			ab.setEntryid(params.entryID);
		} else {
			ab.setEntryid(GenUtils.toBigDecimal(1));
		}
		return ab;
	}

	public static Assessmentinteger prepareIntegerScoreValueRecord(Params params)
			throws Exception {
		Assessmentinteger ai = new Assessmentinteger();
		ai.setNcAssessmentdataUniqueid(params.uniqueID);
		ai.setUniqueid(params.uniqueID);
		ai.setAssessmentid(new BigDecimal(String.valueOf(params.assessmentID)));
		ai.setTableid(params.tableID);
		ai.setScoreorder(new BigDecimal(String.valueOf(params.sv
				.getScoreOrder())));
		ai.setModuser(params.databaseUser.getUniqueid());
		ai.setOwner(params.databaseUser.getUniqueid());
		ai.setModtime(new java.util.Date());
		ai.setTextnormvalue("");
		if (params.sv.getComments() != null) {
			ai.setComments(params.sv.getComments());
		} else {
			ai.setComments("");
		}

		checkAndPrepareIntScoreValue(params.sv, ai, params.missClassificationID);
		if (params.sv.getClassification() != null) {
			ai.setClassification(new BigDecimal(params.sv.getClassification()
					.toString()));
		}
		ai.setStoredassessmentid(params.storedAssessmentID);
		ai.setScorename(params.si.getName());
		ai.setScoretype(params.si.getType());
		/** @todo isRanked needs to be supplied instead of defaults */
		ai.setIsranked(new Boolean(false));
		ai.setIsvalidated(new Boolean(params.validated));
		ai.setScoreorder(GenUtils.toBigDecimal(params.sv.getScoreOrder()));

		if (params.entryID != null) {
			ai.setEntryid(params.entryID);
		} else {
			ai.setEntryid(GenUtils.toBigDecimal(1));
		}
		return ai;
	}

	public static boolean isIntValueValid(String value) {
		if (value == null || value.length() == 0) {
			return false;
		}
		Number numVal = GenUtils.toNumber(value);
		if (numVal != null) {
			if (numVal instanceof Integer) {
				// log.info("numVal.intValue = " + numVal.intValue());
				return true;
			} else {
				return false;
			}
		}
		return true;
	}

	public static boolean isFloatValueValid(String value) {
		if (value == null || value.length() == 0) {
			return false;
		}
		Number numVal = GenUtils.toNumber(value);
		if (numVal != null) {
			if (numVal instanceof Float) {
				// log.info("numVal.floatValue = " + numVal.floatValue());
				return true;
			} else {
				return false;
			}
		}
		return true;
	}

	public static String prepareScoreNameOrderKey(String scoreName,
			int scordeOrder) {
		StringBuffer buf = new StringBuffer();
		buf.append(scoreName).append('_').append(scordeOrder);
		return buf.toString();
	}

	public static Map<String, Integer> getScoreNameMap(Connection con,
			int assessmentID) throws SQLException {
		StringBuffer buf = new StringBuffer();
		buf
				.append("select scorename, scoresequence from nc_assessmentscore where assessmentid =");
		buf.append(assessmentID);
		Map<String, Integer> scoreNameMap = new HashMap<String, Integer>(23);
		Statement st = null;
		ResultSet rs = null;
		try {
			st = con.createStatement();
			rs = st.executeQuery(buf.toString());
			while (rs.next()) {
				String scoreName = rs.getString(1);
				int scoreSequence = rs.getInt(2);
				if (scoreNameMap.get(scoreName) != null) {
					throw new RuntimeException("Duplicate scorename '"
							+ scoreName + "' for assessmentid " + assessmentID);
				}
				scoreNameMap.put(scoreName, new Integer(scoreSequence));
			}
			return scoreNameMap;
		} finally {
			DBUtils.close(st, rs);
		}
	}

	public static String prepareInnerQuery(String subjectID,
			List<AssessmentInfo> assessments, int experimentID, int visitID,
			int segmentID) {
		StringBuffer buf = new StringBuffer(128);
		buf
				.append("select uniqueid from nc_storedassessment where subjectid = '");
		buf.append(subjectID).append("'");

		if (experimentID >= 0) {
			buf.append(" and nc_experiment_uniqueid = ").append(experimentID);
		}
		if (visitID >= 0) {
			buf.append(" and componentid = ").append(visitID);
		}
		if (segmentID >= 0) {
			buf.append(" and segmentid = ").append(segmentID);

		}
		buf.append(" and assessmentid in (");
		Set<Integer> asIDSet = new TreeSet<Integer>();
		for (AssessmentInfo asi : assessments) {
			asIDSet.add(new Integer(asi.getAssessmentID()));
		}
		for (Iterator<Integer> iter = asIDSet.iterator(); iter.hasNext();) {
			Integer asID = iter.next();
			buf.append(asID);
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		buf.append(')');
		return buf.toString();
	}

	public static void getScores(Connection con,
			List<AssessmentInfo> assessmentInfos) throws Exception {
		StringBuilder queryBuf = new StringBuilder(200);
		Statement st = null;
		try {
			queryBuf.append("select assessmentid, scorename, scoretype, ");
			queryBuf
					.append("scorelevel, parentasid, parentscore,scoresequence ");
			queryBuf.append("from nc_assessmentscore where  assessmentid in (");

			Set<Integer> asInfoSet = new TreeSet<Integer>();
			for (AssessmentInfo asi : assessmentInfos) {
				asInfoSet.add(new Integer(asi.getAssessmentID()));
			}

			for (Iterator<Integer> iter = asInfoSet.iterator(); iter.hasNext();) {
				Integer asID = iter.next();
				queryBuf.append(asID);
				if (iter.hasNext()) {
					queryBuf.append(',');
				}
			}
			queryBuf.append(")");
			log.info("query=" + queryBuf.toString());
			st = con.createStatement();

			ResultSet rs = st.executeQuery(queryBuf.toString());

			int[] as_ids = new int[assessmentInfos.size()];
			int idx = 0;
			Map<Integer, List<AssessmentInfo>> asiMap = new HashMap<Integer, List<AssessmentInfo>>();
			for (AssessmentInfo asi : assessmentInfos) {
				as_ids[idx++] = asi.getAssessmentID();
				Integer asID = new Integer(asi.getAssessmentID());
				// we can have multiple assessments of same type for a subject
				List<AssessmentInfo> asList = asiMap.get(asID);
				if (asList == null) {
					asList = new LinkedList<AssessmentInfo>();
					asiMap.put(asID, asList);
				}
				asList.add(asi);
			}

			while (rs.next()) {
				int asID = rs.getInt(1);
				String scoreName = rs.getString(2);
				String scoreType = rs.getString(3);
				int scoreLevel = rs.getInt(4);
				int parentAsID = rs.getInt(5);
				String parentScore = rs.getString(6);
				int scoreOrder = rs.getInt(7);
				AssessmentInfo.ScoreInfo si = new AssessmentInfo.ScoreInfo(
						scoreName, scoreType, scoreLevel, parentAsID,
						parentScore, scoreOrder);

				List<AssessmentInfo> asiList = asiMap.get(new Integer(asID));

				for (AssessmentInfo asi : asiList) {
					asi.addScore(si);
				}
				// log.info("adding score to " + currentAsi.getName() + " : "+
				// si.toString());
			}
			rs.close();
		} finally {
			if (st != null) {
				try {
					st.close();
				} catch (Exception x) {
				}
			}
		}
	}

}
