package clinical.web.vo;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import clinical.server.vo.Humansubject;
import clinical.utils.DateTimeUtils;
import clinical.web.Constants;

public class Subject implements Serializable {
	protected String subjectID;
	protected String extensionName;
	protected int animalSpeciesID;
	protected List<Visit> clinicalVisits = new LinkedList<Visit>();
	protected List<Visit> scanVisits = new LinkedList<Visit>();
	protected String birthDate = "";
	protected String name = "";
	protected String localID = "";
	/** visits are grouped by experiment id at the first level and by visit ty */
	protected Map<Integer, List<Visit>> visitMap = new TreeMap<Integer, List<Visit> >();
	private static final long serialVersionUID = -5913689434375485324L;
	
	public Subject(Humansubject hs) {
		subjectID = hs.getSubjectid();
		extensionName = hs.getExtensionname();
		animalSpeciesID = hs.getNcAnimalspeciesUniqueid().intValue();
		if (hs.getLocalid() != null)
			localID = hs.getLocalid();
		if (hs.getName() != null)
			name = hs.getName();
		if (hs.getBirthdate() != null)
			birthDate = DateTimeUtils.formatDate(hs.getBirthdate());
	}

	public Subject(String subjectID) {
		this.subjectID = subjectID;
	}

	public Subject() {
	}

	public void addClinicalVisit(Visit vi) {
		clinicalVisits.add(vi);
	}

	public void addScanVisit(Visit vi) {
		scanVisits.add(vi);
	}

	public void addVisit(Visit vi, boolean sortVisits) {
		Integer expID = new Integer(vi.getExperimentID());
		List<Visit> visits = visitMap.get(expID);
		if (visits == null) {
			visits = new LinkedList<Visit>();
			visitMap.put(expID, visits);
		}
		visits.add(vi);
		/** @todo remove the harcoded visit types dependency in search drill-down */
		if (vi.getVisitType().equalsIgnoreCase(Constants.VISIT_TYPE_SCAN)) {
			addScanVisit(vi);
		} else {
			addClinicalVisit(vi);
		}

		if (visits.size() == 1) {
			// no sorting necessary
			return;
		}
		if (sortVisits) {
			// sort the visits by date
			Collections.sort(visits, new Comparator<Visit>() {
				public int compare(Visit v1, Visit v2) {
					return v2.getTs().compareTo(v1.getTs());
				}
			});
		}
	}

	public int getFirstExperimentID() {
		if (visitMap.isEmpty())
			return -1;
		List<Visit> visits = visitMap.values().iterator().next();
		Visit visit = (Visit) visits.get(0);
		return visit.getExperimentID();
	}

	public List<Visit> getVisitsForExperiment(String expIDStr) {
		Integer expID = new Integer(expIDStr);
		return visitMap.get(expID);
	}

	public List<Visit> getScanVisits() {
		return scanVisits;
	}

	public List<Visit> getClinicalVisits() {
		return clinicalVisits;
	}

	public Visit findVisit(int visitID, int experimentID) {
		List<Visit> visits = visitMap.get(new Integer(experimentID));
		if (visits == null)
			return null;
		for(Visit visit : visits) {
			if (visit.getComponentID() == visitID) {
				return visit;
			}
		}
		
		return null;
	}

	public Visit findFirstVisit(int experimentID) {
		List<Visit> visits = visitMap.get(new Integer(experimentID));
		if (visits == null)
			return null;
		return (Visit) visits.get(0);
	}

	public void clearVisits() {
		scanVisits.clear();
		clinicalVisits.clear();
		visitMap.clear();
	}

	public Humansubject toHumanSubject() {
		Humansubject hs = new Humansubject();
		hs.setExtensionname(extensionName);
		hs.setSubjectid(subjectID);
		hs.setNcAnimalspeciesUniqueid(new BigDecimal(String
				.valueOf(animalSpeciesID)));
		if (name != null && name.trim().length() > 0) {
			hs.setName(name);
		}
		if (localID != null && localID.trim().length() > 0) {
			hs.setLocalid(localID);
		}
		if (birthDate != null) {
			hs.setBirthdate(DateTimeUtils.toDate(birthDate));
		}
		return hs;
	}

	// ---------------------- setters --------------
	public void setExtensionName(String newExtensionName) {
		this.extensionName = newExtensionName;
	}

	public void setAnimalSpeciesID(int newAnimalSpeciesID) {
		this.animalSpeciesID = newAnimalSpeciesID;
	}

	public void setBirthDate(String newBirthDate) {
		this.birthDate = newBirthDate;
	}

	public void setName(String newName) {
		this.name = newName;
	}

	public void setSubjectID(String newSubjectID) {
		this.subjectID = newSubjectID;
	}

	public void setLocalID(String newLocalID) {
		this.localID = newLocalID;
	}

	// ---------------------- getters --------------
	public String getSubjectID() {
		return this.subjectID;
	}

	public String getExtensionName() {
		return this.extensionName;
	}

	public int getAnimalSpeciesID() {
		return this.animalSpeciesID;
	}

	public String getBirthDate() {
		return this.birthDate;
	}

	public String getName() {
		return this.name;
	}

	public String getLocalID() {
		return this.localID;
	}

	public int hashCode() {
		if (subjectID == null) {
			return super.hashCode();
		} else {
			return subjectID.hashCode();
		}
	}

	public boolean equals(Object other) {
		if (other == null || !(other instanceof Subject))
			return false;
		return subjectID.equals(((Subject) other).subjectID);
	}

}
