package clinical.web.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import clinical.web.common.vo.Row;

/**
 * groups assessment score values by visit-segment.
 *
 * @author I. Burak Ozyurt
 * @version $Id: VisitSegAsResultValues.java,v 1.1.2.1 2008/07/24 01:08:35
 *          bozyurt Exp $
 */
public class VisitSegAsResultValues implements Serializable {
	private static final long serialVersionUID = 1L;
	private int visitID;
	private int segmentID;
	private int expID;
	private String siteID;
	private int studyID = -1;
	private String studyName;
	private String timeStamp;
	private Map<ASValueKey, SubjectAsScoreValueSummary> sasvsMap = new LinkedHashMap<ASValueKey, SubjectAsScoreValueSummary>(
			7);
	private List<Row<ValueWrapper>> rows;
	private int numberOfCols = 2;

	public VisitSegAsResultValues(int visitID, int segmentID, int expID,
			String siteID) {
		this(visitID, segmentID, expID, siteID, -1, null);
	}

	public VisitSegAsResultValues(int visitID, int segmentID, int expID,
			String siteID, int studyID, String studyName) {
		super();
		this.visitID = visitID;
		this.segmentID = segmentID;
		this.expID = expID;
		this.siteID = siteID;
		this.studyID = studyID;
		this.studyName = studyName;
	}

	public void add(ASValueKey key, SubjectAsScoreValueSummary sasvs) {
		sasvsMap.put(key, sasvs);
	}

	public Map<String,SubjectAsScoreValueSummary> getScore2SavsMap() {
		Map<String, SubjectAsScoreValueSummary> map = new HashMap<String, SubjectAsScoreValueSummary>();
		for(ASValueKey asvKey : sasvsMap.keySet()) {
			String key = asvKey.scoreName + ":" + asvKey.asName;
			map.put(key, sasvsMap.get(asvKey));
		}
		return map;
	}

	public List<Row<ValueWrapper>> getRows() {
		if (rows == null) {
			int count = 0;
			String curAsName = null;
			Row<ValueWrapper> row = null;
			rows = new ArrayList<Row<ValueWrapper>>(sasvsMap.size()
					/ numberOfCols + 1);
			List<ASValueKey> sortedList = getSortedKeyList();
			for (ASValueKey key : sortedList) {
				SubjectAsScoreValueSummary sasvs = sasvsMap.get(key);
				if (curAsName == null || !curAsName.equals(key.getAsName())) {
					curAsName = key.getAsName();
					row = new Row<ValueWrapper>(new ArrayList<ValueWrapper>(1),
							numberOfCols);
					row.addColumn(new ValueWrapper(new KeyView(key, true), ""));
					rows.add(row);
					count = 0;
				}
				if (sasvs.isMultiValued() && count > 0) {
					count = 0;
				} else if (count == numberOfCols) {
					count = 0;
				}
				if (count == 0) {
					if (sasvs.isMultiValued()) {
						row = new Row<ValueWrapper>(
								new ArrayList<ValueWrapper>(1), numberOfCols);
					} else {
						row = new Row<ValueWrapper>(
								new ArrayList<ValueWrapper>(numberOfCols),
								numberOfCols);
					}
					rows.add(row);
				}
				if (sasvs.isMultiValued()) {
					StringBuilder sb = new StringBuilder();
					for (Object o : sasvs.getValues()) {
						sb.append(o.toString()).append(" &nbsp; ");
					}
					row.getColumns().add(
							new ValueWrapper(new KeyView(key, false), sb
									.toString()));
				} else {
					row.getColumns().add(
							new ValueWrapper(new KeyView(key, false), sasvs
									.getValue()));
				}
				count++;
			}
		}
		return rows;
	}

	public Set<ScoreMetaData> getScoreMetadata() {
		Set<ScoreMetaData> uniqSet = new HashSet<ScoreMetaData>(17);
		for (ASValueKey asvKey : sasvsMap.keySet()) {
			SubjectAsScoreValueSummary sasvs = sasvsMap.get(asvKey);
			ScoreMetaData smd = new ScoreMetaData(asvKey.getScoreName(), asvKey
					.getAsName(), sasvs.getScoreType());
			uniqSet.add(smd);
		}
		return uniqSet;
	}

	public static class KeyView extends ASValueKey implements Serializable {
		private static final long serialVersionUID = 1L;
		protected boolean title;

		public KeyView(int asID, String asName, String scoreName, boolean title) {
			super(asID, asName, scoreName);
			this.title = title;
		}

		public KeyView(ASValueKey avk, boolean title) {
			super(avk.asID, avk.asName, avk.scoreName);
			this.title = title;
		}

		public boolean isTitle() {
			return title;
		}
	}// ;

	protected List<ASValueKey> getSortedKeyList() {
		List<ASValueKey> list = new ArrayList<ASValueKey>(sasvsMap.keySet());
		Collections.sort(list, new Comparator<ASValueKey>() {

			@Override
			public int compare(ASValueKey o1, ASValueKey o2) {
				int cv = o1.getAsName().compareTo(o2.getAsName());
				if (cv == 0) {
					cv = o1.getScoreName().compareTo(o2.getScoreName());
					SubjectAsScoreValueSummary sasvs = sasvsMap.get(o1);
					if (sasvs.isMultiValued())
						return -1;
				}
				return cv;
			}
		});
		return list;
	}

	public int getVisitID() {
		return visitID;
	}

	public int getSegmentID() {
		return segmentID;
	}

	public int getExpID() {
		return expID;
	}

	public String getSiteID() {
		return siteID;
	}

	public int getStudyID() {
		return studyID;
	}

	public String getStudyName() {
		return studyName;
	}

	public String getTimeStamp() {
		return timeStamp;
	}

	public Map<ASValueKey, SubjectAsScoreValueSummary> getSasvsMap() {
		return sasvsMap;
	}

	public int getNumberOfCols() {
		return numberOfCols;
	}

	public static class ValueWrapper implements Serializable {
		private static final long serialVersionUID = 1L;
		ASValueKey key;
		Object value;

		public ValueWrapper() {
		}

		public ValueWrapper(ASValueKey key, Object value) {
			super();
			this.key = key;
			this.value = value;
		}

		public ASValueKey getKey() {
			return key;
		}

		public Object getValue() {
			return value;
		}

		public boolean isEmptyObject() {
			return value == null;
		}

	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + expID;
		result = prime * result + segmentID;
		result = prime * result + ((siteID == null) ? 0 : siteID.hashCode());
		result = prime * result + visitID;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final VisitSegAsResultValues other = (VisitSegAsResultValues) obj;
		if (expID != other.expID)
			return false;
		if (segmentID != other.segmentID)
			return false;
		if (siteID == null) {
			if (other.siteID != null)
				return false;
		} else if (!siteID.equals(other.siteID))
			return false;
		if (visitID != other.visitID)
			return false;
		return true;
	}

	public void setTimeStamp(String timeStamp) {
		this.timeStamp = timeStamp;
	}

}
