# $Id: srbutil.pm,v 1.1 2003/12/04 20:03:45 bozyurt Exp $
package srbutil;


use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS $VERSION);

use Exporter;

$VERSION = 1.00; 
@ISA = qw(Exporter);

@EXPORT = qw(sinit sexit scd  add_attribute_to_collection add_attribute_to_files remove_files list_files put_files create_SRB_collections);
@EXPORT_OK = qw(sinit sexit scd  add_attribute_to_collection add_attribute_to_files remove_files list_files put_files create_SRB_collections ); 
%EXPORT_TAGS = ();

use strict;
use File::Path;

my $verbose = 0;

sub set_verbose {
   my $flag = $_[0];
   $verbose = $flag  if ($flag == 1 || $flag == 0);
}

sub sinit() {
   my $rc = system("Sinit"); 
    if ($rc == -1) {
	print STDERR "Cannot start Sinit:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}

sub sexit() {
   my $rc = system("Sexit"); 
    if ($rc == -1) {
	print STDERR "Cannot start Sexit:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}

sub scd {
    my ($collection) = @_;
    my $rc = 0; 
    if ($collection){
       $rc = system("Scd $collection");
    } else {
	$rc = system("Scd");
    }
    if ($rc == -1) {
	print STDERR "Cannot start Scd:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}

sub add_attribute_to_collection {
   my ($collection, $name, $value) = @_;
   my $rc = smeta("-c -i -I UDSMD_COLL0='$name' -I UDSMD_COLL1='$value' $collection");
   return $rc; 
}

sub change_attribute_for_collection {
 my ($collection, $name, $value,$op_type) = @_;
 my $cmd="-c ";
 if (defined($op_type) ) {
     $cmd .= ($op_type =~/[iud]/) ?  "-$op_type " : "-i ";
  } else {
          $cmd = "-i ";
  }
   $cmd .= "-I UDSMD_COLL0='$name' -I UDSMD_COLL1='$value' $collection";
   print("$cmd\n");
   my $rc = smeta($cmd);
   return $rc;
}

sub change_attr_for_file {
    my ($collection, $fname, $name, $value, $op_type) = @_;
    my $cmd;
    if (defined($op_type) ) {
         if ( $op_type =~/[iud]/) {
	     $cmd = "-$op_type ";
         } else {
	     $cmd = "-i ";
         }   
     } else {
          $cmd = "-i ";
     }
    $cmd .= "-I  UDSMD0='$name' -I UDSMD1='$value' $collection/$fname";
    my $rc = smeta($cmd);
    return $rc;
}

sub apply_attribute_to_each_file {
    my ($collection, $extension, %nv_map_map) = @_;
    my @files = list_files("$collection/*.$extension");
    my $rc = scd($collection);
    foreach my $file (@files) {
       my $base = $file;
       if ( $file =~ /\/([^\/]+)$/ ) {
           $base = $1;
       }
       unless ( exists $nv_map_map{$base} ) {
           warn("skipping file $base while applying attributes!\n");
	   next;
       } 
       my $nv_ref = $nv_map_map{$base};
       my %nv_map = %{$nv_ref};
       while( my ($key, $value) = each(%nv_map) ) { 
	   $rc = smeta("-i -I  UDSMD0='$key' -I UDSMD1='$value' $collection/$base");
	   return $rc unless ($rc == 0); 
       }
    }
}

sub add_attribute_to_files {
    my ($collection, $extension, $name, $value, ) = @_;
    my @files = list_files("$collection/*.$extension");
    my $rc = scd($collection);
    return $rc unless ($rc == 0); 
    foreach my $file (@files) {
      $rc = smeta("-i -I  UDSMD0='$name' -I UDSMD1='$value' $collection/$file"); 
      return $rc unless ($rc == 0); 
    }
    scd(); 
    return 0; 
}


sub smeta {
    my $args = $_[0];
    # my $rc = system("Smeta $args");
    my $rc = 0; 
    my @all = `Smeta $args 2>&1`;
    foreach my $line (@all) {
      print "$line";
      if ($line =~ /CLI_ERR_SOCKET/) {
            $rc = 1;
      }
    }
    return $rc;
}


sub does_SRB_collection_exists {
    my $collection = $_[0];
    my $flag = 1;
    my $pid = open(PH, "Scd $collection  |") or warn("Cannot execute Scd:$!\n");
    
    while(<PH>) {
	chomp;
        #print "$_\n";
        if (/not in cat$/ ) {
	    $flag = 0;
	}
    } 
    return $flag;
}

sub remove_files {
    my ($collection, $extension) = @_;
    my $rc = system("Srm  $collection/*.$extension");
    if ($rc == -1) {
	print STDERR "Cannot start Srm:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;   
}

sub list_files {
    my $path = $_[0];
    my @objects = ();
    my $pid = open(PH,"Sls $path |");
    while(<PH>) {
       chomp;
       if (/^[^\/]/) {
	   my $x = $_;
	   $x =~ s/\s+//g;
	   push @objects, $x;
       } 
    }
    return @objects;
}


sub get_containers {
  my %containers = ();
  my $pid = open(PH, "Slscont |");
  while(<PH>) {
      chomp;
      next if (/^Containers:/);
      s/^\s+//;
      my $c = $_;
      if ( $c =~ /^\/container\/([^\.\?]+\.[^\.\?]+)\/(\w+)/) {
          $containers{$2} = $1;
      }      
  }   
  return %containers;
}

sub create_container {
    my ($container, $resource_name) = @_;
    my $cmd = "";
    my %container_map =  get_containers();
    return if ( exists $container_map{$container} );
    if (defined($resource_name) ) {
	$cmd = "Smkcont -S $resource_name $container"; 
    } else {
	$cmd = "Smkcont $container";
    }
    my $rc = system($cmd);
    if ($rc == -1) {
	print STDERR "Cannot start Smkcont:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}


sub bulk_load_files {
    my ($local_path, $target_path, $extension, $container, $create_path) = @_;
    my $cmd = "";
    my $rc;
    if ( $create_path ) {
        $rc =  create_SRB_collections($target_path, $container);           
        die("Cannot create collections:$target_path") unless ($rc == 0);
    } 
    $cmd = "Sbload -c $container $local_path/*.$extension $target_path";
    $rc = system($cmd);
    if ($rc == -1) {
       warn("Cannot start Sbload:$!");
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;

}

sub bulk_unload_files {
    my($container, $local_path_root) = @_;
    my @toks = split(/\_/, $container);
    my $path = $local_path_root;
    $path =~ s/\/+$//;
    # a workaround the SRB bug in sbunload
    $path .= "/$toks[0]/$toks[1]/$toks[2]/RawData";
    mkpath($path);
    return -1 unless(-e $path);
    my $cmd = "Sbunload $container $path"; 
    my $rc = system($cmd);
    if ($rc == -1) {
	print STDERR "Cannot start Sbunload:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc; 
}

sub get_data_object_sizes {
   my ($srb_coll, $pattern) = @_;
   $pattern = "*" if (!defined($pattern) or $pattern =~ /^\s*$/ );
   $srb_coll .= "/" unless ( $srb_coll =~ /\/$/ );
   my $pid = open(PH,"SgetD -Y -1 -L 10000 ${srb_coll}$pattern |") or warn("Cannot execute SgetD!");
   my $in_block = 0;
   my %file_info_map = ();
   while (<PH>) {
     chomp;
     my @tokens = split;
     foreach my $token (@tokens) {
       $token =~ s/'//g;
     }  
       my %info_map = ("data_name" => $tokens[1], "data_size" => $tokens[5]);
       $file_info_map{$tokens[1]} = \%info_map;
   }  
   close(PH);
   return %file_info_map; 
}

sub get_data_object_info {
   my ($srb_coll, $pattern) = @_;
   $pattern = "*" if (!defined($pattern) or $pattern =~ /^\s*$/ );
   $srb_coll .= "/" unless ( $srb_coll =~ /\/$/ );
   # print(">> SgetD ${srb_coll}$pattern\n");
   #
   # like most Scommanda SgetD pages results (not best suited for scripting)
   # Here the page size is set to 1,0000 lines hoping the output will be less than ten thousand lines
   # Apparently SgetD allocates memory to prepare the output so we can not be too conservative with this
   # value
   #

   my $pid = open(PH,"SgetD -L 10000 ${srb_coll}$pattern |") or warn("Cannot execute SgetD!");
   my $in_block = 0;
   my %file_info_map = ();
   my @info_list = ();
   while (<PH>) {
     chomp;
     if (/^----/) {
       my %info_map = ();
       push @info_list, \%info_map;
       next;
     }  
   
     if (/(\w+) :(.+)/) {
       #print("$1=$2\n"); 
       my $info_map_ref = $info_list[$#info_list];
       $info_map_ref->{$1} = $2;    
     }
   
   }
   close(PH);
   %file_info_map = map { $_->{data_name} => $_ } @info_list;

   return %file_info_map; 
 }


sub put_files {
    my ($local_path, $target_path, $extension) = @_;
    
    # create SRB collections in target path if necessary

    my $rc = system("Sput $local_path/*.$extension $target_path");
    if ($rc == -1) {
	print STDERR "Cannot start Sput:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}

sub create_collection {
    my ($collection, $container) = @_;
    
    my $cmd = "Smkdir $collection ";
    if ( defined($container) ) {
      $cmd = "Smkdir -c $container $collection" 
    }
    my $rc = system($cmd);
    if ($rc == -1) {
	print STDERR "Cannot start Smkdir:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
}

sub create_SRB_collections {
    my ($collection,$container)  = @_;
    my $rc = 0;
    scd("/");
    if ( $collection =~ /\// ) {
        $collection =~ s/^\/+//;
	my @cols = split("/", $collection);
     	# print join("\n", @cols), "\n"; 
	my $first = 1;  
	my $first_create = 1;
	foreach my $colpart (@cols) {
	    if ($first) {
		$first= 0;
		$colpart = "/" . $colpart;
	    }	
	    if ( does_SRB_collection_exists($colpart) == 0 ) {
		if ( defined( $container) && $first_create) {
		    $first_create = 0;  
		    $rc = create_collection($colpart, $container); 
		} else {
		    $rc = create_collection($colpart);
		}
		return $rc unless ($rc == 0);
		scd($colpart); 
	    } 	   
	}
	scd("/");
    }
    return $rc;
}

sub remove_collection {
    my $collection = $_[0];
    my $rc = system("Srm -r $collection");
    if ($rc == -1) {
	print STDERR "Cannot start Srm:$!\n";
    } else {
	$rc /= 256; # actual exit value
    }
    return $rc;
} 


sub query_coll_meta {
   my ($key, $value_exp, $collection, $max_result_size) = @_;
   my @result_set = ();
   my $cmd = "Smeta -Rc -L $max_result_size -I\"UDSMD_COLL0='$key'\" ";
   $cmd .= " -I\"UDSMD_COLL1 ${value_exp}\" $collection";
   print("cmd=$cmd\n");
   my $pid = open(PH,"$cmd |");
   my $in_record = 0;
    while(<PH>) {
       chomp;
       if (/^data_grp_name :(\S+)/) {
           my $record = { COLLECTION => $1 };
           push @result_set, $record;
           $in_record = 1;
       }
       if (/userdef_metastr0 :(\w+)/) {
           my $record = $result_set[$#result_set];
           $record->{NAME} = $1;
       }
      if (/userdef_metastr1 :(\w+)/) {
           my $record = $result_set[$#result_set];
           $record->{VALUE} = $1;
           $in_record = 0;
       }
    }
   close(PH);
   return @result_set;
}

1;

=head1 NAME

srbutil.pm

=cut

=head1 SYNOPSIS

    use srbutil;
 
    # to establish a connection with SRB server

    srbutil::sinit();

    # for collection(directory)/file operations
    
    srbutil::does_SRB_collection_exists($collection);
    srbutil::remove_files($collection, $extension);
    srbutil::list_files($path);
    srbutil::put_files($local_path, $target_path, $extension);
    srbutil::create_SRB_collections($collection);
    srbutil::remove_collection($collection);

    # for metadata

    srbutil::add_attribute_to_collection($collection, $name, $value);
    srbutil::add_attribute_to_files($collection, $extension, $name, $value);

    # to close connection to SRB server

    srbutil::sexit();   

=cut

=head1 DESCRIPTION

A semi-thin wrapper around SRB Scommands. The main purpose of this library is to abstract away 
details and provide convenience functions to the client programs 

=cut

=head1 AUTHOR

Written by I. Burak Ozyurt

=cut

=head1 NOTES

This library does not contain wrappers for all SRB commands, only for those which are necessary for data/metadata 
upload and cleanup.

=cut
