package clinical.web.common.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.struts.util.LabelValueBean;
import org.jdom.Element;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.GenUtils;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.helpers.AssessmentSelectionInfoHelper;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.vo.RangeInfo;

/**
 * Represents the model for the query part (single search predicate) view of the
 * assessment/derived data query builder screen.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: QueryPartInfo.java 91 2009-08-17 23:38:26Z bozyurt $
 */

public class QueryPartInfo extends AbstractQueryPartInfo {
	protected AssessmentSelectionInfo asi;
	protected AsScoreInfo scoreInfo;
	protected SubCorticalVarInfo subCorticalVarInfo;
	protected RangeInfo rangeInfo;
	protected String connector;
	public final static String NONE = "NONE";
	public final static String AND = "AND";
	public final static String OR = "OR";
	private static final long serialVersionUID = 1L;

	public QueryPartInfo() {
	}

	public QueryPartInfo(AssessmentSelectionInfo asi, AsScoreInfo si) {
		this.asi = asi;
		this.scoreInfo = si;
		prepareLogicalOps();
	}

	public QueryPartInfo(QueryPartInfo other, AssessmentSelectionInfo asi,
			AsScoreInfo si) {
		super(other);
		this.connector = other.connector;
		if (asi != null)
			this.asi = asi;
		else
			this.asi = other.asi;
		if (si != null)
			this.scoreInfo = si;
		else
			this.scoreInfo = other.scoreInfo;
	}

	private void prepareLogicalOps() throws RuntimeException {
		if (scoreInfo.getType().equals("integer")
				|| scoreInfo.getType().equals("float")
				|| scoreInfo.getType().equals("timestamp")) {
			prepareNumericLogicalOps();
		} else if (scoreInfo.getType().equals("varchar")) {
			prepareVarcharLogicalOps();
		} else {
			throw new RuntimeException(scoreInfo.getType()
					+ " is not a supported type!");
		}
	}

	public QueryPartInfo(AssessmentSelectionInfo asi, AsScoreInfo si,
			List<String> possibleValues, List<String> scoreCodes) {
		this.asi = asi;
		this.scoreInfo = si;

		prepareRHSEnumerations(possibleValues, scoreCodes);
		prepareLogicalOps();
	}

	public QueryPartInfo(SubCorticalVarInfo scvi) {
		this.subCorticalVarInfo = scvi;
		prepareNumericLogicalOps();
	}

	// to fool buggy Struts bean tags
	public Collection<LabelValueBean> getLogicalOps() {
		return super.getLogicalOps();
	}

	// ---------------------- setters --------------
	public void setAsi(AssessmentSelectionInfo newAsi) {
		this.asi = newAsi;
	}

	public void setScoreInfo(AsScoreInfo newScoreInfo) {
		this.scoreInfo = newScoreInfo;
	}

	public void setSubCorticalVarInfo(SubCorticalVarInfo newSubCorticalVarInfo) {
		this.subCorticalVarInfo = newSubCorticalVarInfo;
	}

	public void setRangeInfo(RangeInfo newRangeInfo) {
		this.rangeInfo = newRangeInfo;
	}

	// ---------------------- getters --------------
	public AssessmentSelectionInfo getAsi() {
		return this.asi;
	}

	public AsScoreInfo getScoreInfo() {
		return this.scoreInfo;
	}

	public String getConnector() {
		return connector;
	}

	public SubCorticalVarInfo getSubCorticalVarInfo() {
		return this.subCorticalVarInfo;
	}

	public RangeInfo getRangeInfo() {
		return this.rangeInfo;
	}

	public String getFormattedRange() {
		if (rangeInfo == null)
			return "";
		StringBuffer sb = new StringBuffer();
		if (rangeInfo.getIncludeLowBound()) {
			sb.append("[");
		} else {
			sb.append("(");
		}
		sb.append(rangeInfo.getLowBound()).append(" - ").append(
				rangeInfo.getUppBound());
		if (rangeInfo.getIncludeUppBound()) {
			sb.append("]");
		} else {
			sb.append(")");
		}
		return sb.toString();
	}

	@Override
	public Collection<LabelValueBean> getRhsEnums() {
		return rhsEnums;
	}

	public JSONObject toJSON(JSONObject js) throws JSONException {
		js = super.toJSON(js);
		js.put("connector", connector);
		if (asi != null) {
			js.put("asi", asi.toJSON(new JSONObject()));
		}
		if (scoreInfo != null) {
			js.put("score", scoreInfo.toJSON(new JSONObject()));
		}
		return js;
	}

	public static QueryPartInfo initializeFromJSON(JSONObject js)
			throws JSONException {
		AssessmentSelectionInfo asi = AssessmentSelectionInfo
				.initializeFromJSON(js.getJSONObject("asi"));
		AsScoreInfo si = AsScoreInfo.initializeFromJSON(asi.getAssessmentID()
				.intValue(), js.getJSONObject("score"));
		QueryPartInfo qpi = new QueryPartInfo(asi, si);
		// FIXME handle hasEnums
		qpi.operator = js.getInt("operator");
		if (js.has("lowBound") && js.has("uppBound")) {
			qpi.lowBound = js.getString("lowBound");
			qpi.uppBound = js.getString("uppBound");
		}
		if (js.has("rhs"))
			qpi.rhs = js.getString("rhs");

		qpi.connector = js.getString("connector");
		return qpi;
	}

	public Element toXML() {
		Element elem = super.toXML();
		if (asi != null) {
			// skip scores
			Element asiElem = asi.toXML(true);
			elem.addContent(asiElem);
		}
		if (scoreInfo != null) {
			Element siElem = scoreInfo.toXML();
			elem.addContent(siElem);
		}
		if (subCorticalVarInfo != null) {
			Element scvElem = subCorticalVarInfo.toXML();
			elem.addContent(scvElem);
		}
		return elem;
	}

	public static QueryPartInfo initializeFromXML(Element elem,
			AssessmentSelectionInfoHelper helper) throws Exception {
		Element asiElem = elem.getChild("as-select-info");
		Element siElem = elem.getChild("score-info");
		Element scvElem = elem.getChild("subcortical-var-info");
		List<String> possibleValues = null;
		QueryPartInfo qpi = null;

		Element enumsElem = elem.getChild("rhs-enums");
		if (enumsElem != null) {
			List<?> enumList = enumsElem.getChildren("enum");
			possibleValues = new ArrayList<String>();
			for (Iterator<?> iter = enumList.iterator(); iter.hasNext();) {
				Element child = (Element) iter.next();
				String valueStr = child.getAttributeValue("value");
				possibleValues.add(valueStr);
			}
		}

		if (scvElem != null) {
			SubCorticalVarInfo scvi = SubCorticalVarInfo
					.initializeFromXML(scvElem);
			qpi = new QueryPartInfo(scvi);
		} else {
			AssessmentSelectionInfo asi = AssessmentSelectionInfo
					.initializeFromXML(asiElem, helper);
			AsScoreInfo si = AsScoreInfo.initializeFromXML(siElem, asi
					.getAssessmentID());
			AsScoreInfo theSi = helper.findAsScoreInfo(asi.getAssessmentID(),
					si.getName());

			// set the selected state of the actual AsScoreInfo
			theSi.setSelected(si.isSelected());
			if (possibleValues != null) {
				List<String> scoreCodes = helper.getScoreCodesForScore(asi
						.getAssessmentID(), theSi.getName());
				qpi = new QueryPartInfo(asi, theSi, possibleValues, scoreCodes);
				// set the selected enumeration
				qpi.setRhsIdx(GenUtils.getAttributeValueAsString(elem, "rhs"));
			} else {
				qpi = new QueryPartInfo(asi, theSi);
			}
		}

		int value = GenUtils.toInt(elem.getAttributeValue("operator"), -1);
		if (value >= 0) {
			qpi.setOperator(value);
		}

		qpi.setRhs(GenUtils.getAttributeValueAsString(elem, "rhs"));
		qpi.setLowBound(GenUtils
				.getAttributeValueAsString(elem, "lowBound", ""));
		qpi.setUppBound(GenUtils
				.getAttributeValueAsString(elem, "uppBound", ""));

		qpi.setUseBetween(GenUtils.toBoolean(elem
				.getAttributeValue("useBetween"), false));
		qpi.setHasEnums(GenUtils.toBoolean(elem.getAttributeValue("hasEnums"),
				false));

		return qpi;
	}

	public String toString() {
		StringBuffer sb = new StringBuffer(128);
		sb.append("AbstractQueryPartInfo::[");
		if (getScoreInfo() != null) {
			sb.append("scoreName=").append(getScoreInfo().getName());
			sb.append(", selected=").append(getScoreInfo().isSelected());
		} else if (getSubCorticalVarInfo() != null) {
			sb.append("brainRegionName=").append(
					getSubCorticalVarInfo().getBrainRegionName());
			sb.append(", laterality=").append(
					getSubCorticalVarInfo().getLaterality());
			sb.append(", selected=").append(
					getSubCorticalVarInfo().isSelected());
		}
		sb.append(", rhs=").append(getRhs());
		sb.append(']');
		return sb.toString();
	}

	public int hashCode() {
		if (this.scoreInfo != null) {
			return this.scoreInfo.hashCode();
		} else {
			return subCorticalVarInfo.hashCode();
		}
	}

	/**
	 * relies on the assumption that a score or subcortical var is not repeated
	 * thus a querypartifo is uniquely identified by the score name or
	 * subcortical var name and laterality
	 * 
	 * @param other
	 * @return
	 */
	public boolean equals(Object other) {
		if (other == this) {
			return true;
		}
		if (!(other instanceof QueryPartInfo)) {
			return false;
		}
		QueryPartInfo that = (QueryPartInfo) other;

		if (this.scoreInfo != null) {
			if (that.scoreInfo == null)
				return false;
			if (this.useBetween) {
				if (that.useBetween == false)
					return false;
				return (this.scoreInfo.equals(that.scoreInfo)
						&& this.rhs.equals(that.rhs)
						&& this.lowBound.equals(that.lowBound) && this.uppBound
						.equals(that.uppBound));

			}

			return (this.scoreInfo.equals(that.scoreInfo));

		} else {
			if (that.subCorticalVarInfo == null)
				return false;
			return (this.subCorticalVarInfo.equals(that.subCorticalVarInfo));
		}
	}

}
