package clinical.web.common.query;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.StringTokenizer;

import clinical.utils.DateTimeUtils;

/**
 * A collection of static utility methods for query builders.
 * 
 * @author I. Burak Ozyurt
 * @version Id
 */
public class QueryUtils {
	private static SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");

	static {
		df.setLenient(false);
	}

	private QueryUtils() {
	}

	/**
	 * Given data type string returns the type code as defined in
	 * <code>SearchPredicate</code>.
	 * 
	 * @param type
	 *            a type string (varchar,float,boolean or date)
	 * @return type code as defined in <code>SearchPredicate</code>. Can be
	 *         <code>INTEGER</code>, <code>STRING</code>,
	 *         <code>FLOAT</code>, <code>BOOLEAN</code> or
	 *         <code>DATE</code>
	 */
	public static int getSearchPredicateType(String type) {
		int typ = SearchPredicate.INTEGER;
		if (type.equals("varchar"))
			typ = SearchPredicate.STRING;
		else if (type.equals("float"))
			typ = SearchPredicate.FLOAT;
		else if (type.equals("boolean"))
			typ = SearchPredicate.BOOLEAN;
		else if (type.equals("date"))
			typ = SearchPredicate.DATE;
		else if (type.equals("timestamp")) {
			typ = SearchPredicate.DATE;
		}
		return typ;
	}

	/**
	 * 
	 * @param spType
	 *            search predicate type
	 * @return
	 */
	public static boolean isNumeric(int spType) {
		return (spType == SearchPredicate.INTEGER || spType == SearchPredicate.FLOAT);
	}

	/**
	 * Given a Java data type wrapper class returns the type code as defined in
	 * <code>SearchPredicate</code>.
	 * 
	 * @param type
	 *            Java data type wrapper class (one of the
	 *            <code>java.lang.String.class</code>,
	 *            <code>java.lang.Integer.class</code>,<code>java.lang.Float.class</code>,
	 *            <code>java.lang.Boolean.class</code> or
	 *            <code>java.util.Date.class</code>
	 * @return type type code as defined in <code>SearchPredicate</code>. Can
	 *         be <code>INTEGER</code>, <code>STRING</code>,
	 *         <code>FLOAT</code>, <code>BOOLEAN</code> or
	 *         <code>DATE</code>
	 */
	public static int getSearchPredicateTypeForProperty(Class<?> type) {
		if (type == java.lang.String.class)
			return SearchPredicate.STRING;
		if (type == java.lang.Integer.class || type == int.class)
			return SearchPredicate.INTEGER;
		if (type == java.lang.Float.class || type == float.class)
			return SearchPredicate.FLOAT;
		if (type == java.lang.Boolean.class || type == boolean.class)
			return SearchPredicate.BOOLEAN;
		if (type == java.util.Date.class)
			return SearchPredicate.DATE;
		return -1;
	}

	/**
	 * Given a string and a type code, converts the string to the corresponding
	 * data type.
	 * 
	 * @param value
	 *            the value (number, string or date) as a string
	 * @param type
	 *            type code as defined in <code>SearchPredicate</code>. Can
	 *            be <code>INTEGER</code>, <code>STRING</code>,
	 *            <code>FLOAT</code>, <code>BOOLEAN</code> or
	 *            <code>DATE</code>
	 * @return corresponding Java data type wrapper object ( Integer, String,
	 *         Float, Boolean or java.util.Date
	 * @throws java.lang.NumberFormatException
	 *             if
	 */
	public static Object convertToType(String value, int type)
			throws NumberFormatException {
		if (value == null || value.length() == 0)
			return null;
		switch (type) {
		case SearchPredicate.INTEGER:
			return new Integer(value);
		case SearchPredicate.STRING:
			return value;
		case SearchPredicate.FLOAT:
			return new Float(value);
		case SearchPredicate.BOOLEAN:
			return new Boolean(value);
		case SearchPredicate.DATE:
			try {
				return df.parse(value);
			} catch (ParseException pe) {
				pe.printStackTrace();
				return null;
			}
		}

		throw new RuntimeException("Not a supported type " + type + " for "
				+ value);
	}

	/**
	 * 
	 * @param sourceName
	 * @param variableName
	 * @param value
	 * @param opType
	 *            operation type
	 * @return
	 * @throws java.lang.Exception
	 */
	public static String createPredicate(String sourceName,
			String variableName, Object value, int opType) throws Exception {
		StringBuffer buf = new StringBuffer(64);
		boolean done = false;
		buf.append(sourceName).append(".eval_boolean(");

		if (value instanceof String && ((String) value).equals("*")) {
			buf.append("like,");
			buf.append(variableName).append(',');
			buf.append("'%') ");
			return buf.toString();
			// throw new Exception("Wildcard is not supported in mediated
			// queries!");
		}

		switch (opType) {
		case SearchPredicate.EQUAL:
			buf.append("eq,");
			buf.append(variableName).append(',');
			if (value instanceof String) {
				buf.append("'").append((String) value).append("') ");
			} else if (value instanceof Number) {
				buf.append((Number) value).append(") ");
			} else
				throw new Exception("Not an supported value type:"
						+ value.getClass().getName());
			done = true;
			break;
		case SearchPredicate.LESS:
			buf.append("lt,");
			buf.append(variableName).append(',');
			break;
		case SearchPredicate.GREATER:
			buf.append("gt,");
			buf.append(variableName).append(',');
			break;
		case SearchPredicate.LESS_EQUAL:
			buf.append("le,");
			buf.append(variableName).append(',');
			break;
		case SearchPredicate.GREATER_EQUAL:
			buf.append("ge,");
			buf.append(variableName).append(',');
			break;
		case SearchPredicate.NOT_EQUAL:
			buf.append("ne,");
			buf.append(variableName).append(',');
			break;
		case SearchPredicate.STARTS_WITH:
			if (!(value instanceof String))
				throw new Exception(
						"Only varchar type can be used with Starts_With comparator!");
			buf.append("like,");
			buf.append(variableName).append(',');
			buf.append("'").append((String) value).append("%') ");
			done = true;
			break;
		case SearchPredicate.ENDS_WITH:
			if (!(value instanceof String))
				throw new Exception(
						"Only varchar type can be used with Ends_With comparator!");
			buf.append("like,");
			buf.append(variableName).append(',');
			buf.append("'%").append((String) value).append("') ");
			done = true;
			break;
		case SearchPredicate.ANY:
			throw new Exception(
					"Any comparator is not supported with mediated queries!");
		}

		if (!done) {
			if (!(value instanceof Number))
				throw new Exception(
						"Only numeric type can be used with comparison operators!");
			buf.append((Number) value).append(") ");
		}
		return buf.toString();
	}

	/**
	 * Given a operator code returns the corresponding SQL relational operator.
	 * 
	 * @param op
	 *            relational operator codes as defined in
	 *            <code>SearchPredicate</code>
	 * @return the corresponding SQL relational operator
	 * @see SearchPredicate
	 */
	public static String getRelationalOperator(int op) {
		switch (op) {
		case SearchPredicate.EQUAL:
			return " = ";
		case SearchPredicate.GREATER:
			return " > ";
		case SearchPredicate.LESS:
			return " < ";
		case SearchPredicate.NOT_EQUAL:
			return " <> ";
		case SearchPredicate.GREATER_EQUAL:
			return " >= ";
		case SearchPredicate.LESS_EQUAL:
			return " <= ";
		case SearchPredicate.BETWEEN:
			return " BETWEEN ";
		case SearchPredicate.ENDS_WITH:
		case SearchPredicate.STARTS_WITH:
			return " LIKE ";
		default:
			throw new RuntimeException("Not a valid operator " + op);
		}
	}

	public static String formatDate(java.sql.Date date) {
		return df.format(date);
	}

	public static boolean condSatisfied(SearchPredicate.Range range,
			String value2Check) {
		Number v2c = convert2Number(value2Check);
		if (v2c == null) {
			return false;
		}
		return (range.getLowBound().doubleValue() <= v2c.doubleValue() && range
				.getUppBound().doubleValue() >= v2c.doubleValue());
	}

	public static boolean condSatisfied(SearchPredicate.DateRange range,
			String value2Check) {
		Date v2c = DateTimeUtils.toDate(value2Check);
		if (v2c == null) {
			return false;
		}
		return range.getLowBound().compareTo(v2c) <= 0
				&& range.getUppBound().compareTo(v2c) >= 0;
	}

	public static boolean condSatisfied(int opType, double v1, double v2) {
		switch (opType) {
		case SearchPredicate.GREATER:
			return v2 > v1;
		case SearchPredicate.GREATER_EQUAL:
			return v2 >= v1;
		case SearchPredicate.LESS:
			return v2 < v1;
		case SearchPredicate.LESS_EQUAL:
			return v2 <= v1;
		case SearchPredicate.NOT_EQUAL:
			return v2 != v1;
		default:
			return false;
		}
	}

	public static boolean condSatisfied(int opType, Date v1, Date v2) {
		switch (opType) {
		case SearchPredicate.GREATER:
			return v2.after(v1);
		case SearchPredicate.GREATER_EQUAL:
			return v2.compareTo(v1) >= 0; // >= v1;
		case SearchPredicate.LESS:
			return v2.before(v1);
		case SearchPredicate.LESS_EQUAL:
			return v2.compareTo(v1) <= 0;
		case SearchPredicate.NOT_EQUAL:
			return v2.compareTo(v1) != 0;
		default:
			return false;
		}
	}

	public static Number convert2Number(String value2Check) {
		if (value2Check == null) return null;
		try {
			Integer num = new Integer(value2Check);
			return num;
		} catch (NumberFormatException nfe) {
			try {
				Double num = new Double(value2Check);
				return num;
			} catch (NumberFormatException nfe2) {
				return null;
			}
		}
	}

	public static boolean condSatisfied(Object value, int opType, int dataType,
			String value2Check) {
		if (value instanceof String && ((String) value).trim().equals("*")) {
			return true;
		}
		switch (opType) {
		case SearchPredicate.EQUAL:
			if (value instanceof String) {
				return value.equals(value2Check);
			} else if (value instanceof Number) {
				Number v2c = convert2Number(value2Check);
				if (v2c == null) {
					return false;
				}
				Number v = (Number) value;
				return v.doubleValue() == v2c.doubleValue();
			} else if (value instanceof Date) {
				Date v2c = DateTimeUtils.toDate(value2Check);
				if (v2c == null) {
					return false;
				}
				return ((Date) value).compareTo(v2c) == 0;
			} else {
				return false;
			}
		case SearchPredicate.GREATER:
		case SearchPredicate.GREATER_EQUAL:
		case SearchPredicate.LESS:
		case SearchPredicate.LESS_EQUAL:
		case SearchPredicate.NOT_EQUAL:
			if (value instanceof Number) {
				Number v = (Number) value;
				Number v2c = convert2Number(value2Check);
				if (v2c == null) {
					return false;
				}
				return condSatisfied(opType, v.doubleValue(), v2c.doubleValue());
			} else if (value instanceof Date) {
				Date v = (Date) value;
				Date v2c = DateTimeUtils.toDate(value2Check);
				if (v2c == null) {
					return false;
				}
				return condSatisfied(opType, v, v2c);
			} else {
				throw new RuntimeException("Unsupported data type: "
						+ value.getClass().getName());
			}
		case SearchPredicate.STARTS_WITH:
			if (!(value instanceof String)) {
				return false;
			}
			return value2Check.startsWith((String) value);
		case SearchPredicate.ENDS_WITH:
			if (!(value instanceof String)) {
				return false;
			}
			return value2Check.endsWith((String) value);
		case SearchPredicate.ANY:
			if (!(value instanceof String)) {
				return false;
			}
			StringTokenizer stok = new StringTokenizer((String) value, " ,");
			while (stok.hasMoreTokens()) {
				if (value.equals(stok.nextToken())) {
					return true;
				}
			}
			return false;
		default:
			return false;
		}
	}
}
