package clinical.web.services;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.NamedUserPool;
import clinical.web.Constants;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.User;
import clinical.web.exception.DBPoolServiceException;

/**
 * Implements <code>IDBPoolService</code> interface to provide database
 * connection pooling service.
 * 
 * @see clinical.web.common.IDBPoolService
 * @author I. Burak Ozyurt
 * @version $Id: DBPoolService.java 90 2009-08-17 23:37:45Z bozyurt $
 */

public class DBPoolService implements IDBPoolService {
	private NamedUserPool pool;
	private static Log log = LogFactory.getLog("clinical");
	private String driverClass;
	private String dbURL;
	private String dbID;
	private Map<String, User> userMap;
	private boolean showStats = false;
	private static Map<String, String> driverClassMap;

	private static DBPoolService instance;
	private static Map<String, DBPoolService> instanceMap = null;

	static {
		driverClassMap = new HashMap<String, String>(3);
		driverClassMap.put("oracle", "oracle.jdbc.driver.OracleDriver");
		driverClassMap.put("postgres", "org.postgresql.Driver");
	}

	private DBPoolService(String driverClass, String dbURL,
			Map<String, User> userMap) {
		this.driverClass = driverClass;
		this.dbURL = dbURL;
		this.userMap = userMap;
	}

	public synchronized static DBPoolService getInstance(
			ISecurityService securityService, Map<String, DBConfig> dbConfigMap) {
		if (instanceMap == null) {
			List<String> badDBList = new ArrayList<String>(1);
			log.debug("creating a new DBPoolService");
			instanceMap = new LinkedHashMap<String, DBPoolService>(7);
			for (DBConfig dbConfig : dbConfigMap.values()) {

				String driverClassName = driverClassMap.get(dbConfig
						.getDbType());
				DBPoolService poolService = new DBPoolService(driverClassName,
						dbConfig.getDbURL(), dbConfig.getUserMap());

				// test if the database for this particular DBPoolService is
				// reachable
				User adminUser = dbConfig.getUser(Constants.ADMIN_USER);
				if (adminUser == null) {
					log.error("No admin web user for database:"
							+ dbConfig.getId() + "! Skipping...");
					badDBList.add(dbConfig.getId());
					continue;
				}
				Connection con = null;
				try {
					poolService.startup();
					con = poolService.getConnection(adminUser.getName());
				} catch (Throwable t) {
					log.error("No connection to database " + dbConfig.getId(),
							t);
					badDBList.add(dbConfig.getId());
					continue;
				} finally {
					if (con != null)
						try {
							poolService.releaseConnection(adminUser.getName(),
									con);
						} catch (DBPoolServiceException e1) {
						}
				}
				poolService.dbID = dbConfig.getId();
				instanceMap.put(dbConfig.getId(), poolService);

				if (dbConfig.isDefaultDB())
					instance = poolService;
			}
			if (!badDBList.isEmpty()) {
				for (Iterator<String> it = badDBList.iterator(); it.hasNext();) {
					String dbID = it.next();
					dbConfigMap.remove(dbID);
				}
			}
		}
		return instance;
	}

	public synchronized static DBPoolService getInstance(String dbID) {
		if (instanceMap == null)
			throw new RuntimeException("DBPoolService not properly initiated!");
		DBPoolService poolService = instanceMap.get(dbID);
		if (poolService == null)
			throw new RuntimeException("No DBPoolService for database ID "
					+ dbID);
		return poolService;
	}

	public synchronized static Set<String> getAvailablePoolServices() {
		return new HashSet<String>(instanceMap.keySet());
	}

	public synchronized static IDBPoolService getPoolService(String dbID) {
		return (IDBPoolService) instanceMap.get(dbID);
	}

	/**
	 * 
	 * @param dbConfig
	 * @return
	 */
	public synchronized static IDBPoolService addDatabase(DBConfig dbConfig) {
		if (instanceMap == null)
			throw new RuntimeException("DBPoolService not properly initiated!");
		DBPoolService poolService = instanceMap.get(dbConfig.getId());
		if (poolService != null) {
			return poolService;
		}
		String driverClassName = driverClassMap.get(dbConfig.getDbType());
		poolService = new DBPoolService(driverClassName, dbConfig.getDbURL(),
				dbConfig.getUserMap());

		// test if the database for this particular DBPoolService is
		// reachable
		User adminUser = dbConfig.getUser(Constants.ADMIN_USER);
		Connection con = null;
		try {
			poolService.startup();
			con = poolService.getConnection(adminUser.getName());
		} catch (Throwable t) {
			log.error("No connection to database " + dbConfig.getId(), t);
			return null;
		} finally {
			if (con != null)
				try {
					poolService.releaseConnection(adminUser.getName(), con);
				} catch (DBPoolServiceException e1) {
				}
		}
		poolService.dbID = dbConfig.getId();
		log.info("adding poolService " + poolService.dbID + " ("
				+ dbConfig.getDbType() + ") " + dbConfig.getDbURL());
		instanceMap.put(dbConfig.getId(), poolService);

		return poolService;
	}

	public synchronized static void removeDatabase(String dbID) {
		if (instanceMap == null)
			throw new RuntimeException("DBPoolService not properly initiated!");
		DBPoolService poolService = instanceMap.get(dbID);
		if (poolService != null) {
			try {
				poolService.shutdown();
			} catch (DBPoolServiceException e) {
				log.error("", e);
			}
		}
	}

	public void setShowStats(boolean newShowStats) {
		this.showStats = newShowStats;
		if (pool != null) {
			pool.setShowStats(this.showStats);
		}
	}

	public boolean getShowStats() {
		return this.showStats;
	}

	public synchronized void addUser(User webUser) {
		if (!userMap.containsKey(webUser.getName())) {
			log.info("adding user " + webUser);
			userMap.put(webUser.getName(), webUser);
		}
	}

	public synchronized void removeUser(User webUser) {
		if (userMap.containsKey(webUser.getName())) {
			log.info("removing user " + webUser);
			userMap.remove(webUser.getName());
		}
	}

	public synchronized Connection getConnection(String userName)
			throws DBPoolServiceException {
		Connection con = null;
		User user = userMap.get(userName);
		if (user == null)
			throw new DBPoolServiceException(
					"Cannot connect:Not a valid user!:" + userName);
		try {
			con = pool.getConnection(user.getDbUser().getName(), user
					.getDbUser().getPwd());
			if (log.isDebugEnabled())
				log.debug("returning connection for user "
						+ user.getDbUser().getName() + " con=" + con);
			return con;
		} catch (Exception x) {
			log.error("getConnection", x);
			throw new DBPoolServiceException(x);
		}
	}

	public synchronized void releaseConnection(String userName, Connection con)
			throws DBPoolServiceException {
		if (con != null) {
			User user = userMap.get(userName);
			if (user == null)
				throw new DBPoolServiceException(
						"Cannot return connection:Not a valid user!");
			try {
				pool.releaseConnection(user.getDbUser().getName(), con);
				log.debug("released connection for user "
						+ user.getDbUser().getName());
			} catch (Exception x) {
				log.error("releaseConnection", x);
				throw new DBPoolServiceException(x);
			}
		}
	}

	public void startup() throws DBPoolServiceException {
		try {
			pool = NamedUserPool.getInstance(driverClass, dbURL);
			if (log.isDebugEnabled())
				log.debug("pool is created using driverClass=" + driverClass
						+ ", dbURL=" + dbURL);

		} catch (Exception x) {
			log.fatal("startup", x);
			throw new DBPoolServiceException(x);
		}
	}

	public synchronized void shutdown() throws DBPoolServiceException {
		if (pool != null) {
			pool.shutdown();
		}
		if (log.isDebugEnabled()) {
			log.debug("Connection pool is shutdown.");
		}
		instanceMap.remove(instance.dbID);
		if (instanceMap.isEmpty()) {
			instanceMap = null;
		}
		instance = null;
	}
	
	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("DBPoolService::[");
		buf.append("dbID=").append(dbID);
		buf.append(",dbURL=").append(dbURL);
		buf.append(']');
		return buf.toString();
	}
}
