package clinical.web.services;

import imagelib.util.SuffixFilenameFilter;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Rawdata;
import clinical.utils.Assertion;
import clinical.utils.BIRNURIException;
import clinical.utils.BIRNURIUtils;
import clinical.utils.FileUtils;
import clinical.utils.JargonUtils;
import clinical.utils.SRBFileSystemAdapter;
import clinical.web.Constants;
import clinical.web.IAppConfigService;
import clinical.web.IImageDataService;
import clinical.web.IRemoteDBServices;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.download.DataNode;
import clinical.web.download.FileInfo;
import clinical.web.download.ImageSeriesDataInfo;
import clinical.web.download.ObjectInfo;
import clinical.web.download.PathWrapper;
import clinical.web.download.SubjectImageDataBundle;
import clinical.web.download.Unpacker;
import clinical.web.helpers.ShoppingCartHelper;
import clinical.web.image.viewer.IPlugin;
import clinical.web.image.viewer.PluginManager;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.RawDataObject;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DownloadJobServiceImpl.java,v 1.2 2008/08/26 00:08:00 bozyurt
 *          Exp $
 */

public class DownloadJobServiceImpl implements IDownloadJobService {
	private Log log = LogFactory.getLog(DownloadJobServiceImpl.class);

	public void prepImagePathsForSubject(String siteID, String dbID,
			UserInfo ui, String[] subjectIDArr, String expID,
			SubjectImageDataBundle[] sidbArr, String rootPath,
			IAppConfigService configService, String[] specifiedObjectTypes)
			throws Exception {

		log.info("prepImagePathsForSubject : dbID:" + dbID);
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
		IImageDataService ids = ServiceFactory.getImageDataService(dbID);
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);

		String[] objectTypes = null;
		if (specifiedObjectTypes == null) {
			objectTypes = configService.getParamValues("download.object.types");
		} else {
			objectTypes = specifiedObjectTypes;
		}
		System.out.println("Object Types");
		for (String ot : objectTypes) {
			System.out.println(ot);
		}
		System.out.println("----------");

		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		Map<String, Experiment> experimentMap = new HashMap<String, Experiment>(7);
		for (Iterator<Experiment> iter = experiments.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			experimentMap.put(exp.getUniqueid().toString(), exp);
		}
		if (siteID != null) {
			Map<String, Experiment> remoteSiteExpMap = rds
					.getAllExperiments(siteID);
			for (Experiment exp : remoteSiteExpMap.values()) {
				experimentMap.put(exp.getUniqueid().toString(), exp);
			}
		}

		List<String> subjectIDList = new ArrayList<String>(1);
		for (int i = 0; i < subjectIDArr.length; i++) {
			subjectIDList.add(subjectIDArr[i]);
		}

		// String[] objectTypes = {"3_LOCAL COL DICOM", "3_LOCAL COL NIFTI"};

		int[] expIDs = { Integer.parseInt(expID) };
		List<RawDataObject> rdoDataList = null;
		boolean verbose = false;

		if (siteID != null) {
			rdoDataList = ids.getRawData(siteID, ui, expIDs, subjectIDList,
					objectTypes, verbose);
		} else {
			rdoDataList = ids.getRawData(ui, expIDs, subjectIDList, objectTypes,
					verbose);
		}
		rdoDataList = ShoppingCartHelper.removeIntermediatePaths(rdoDataList);

		List<String> expNames = new ArrayList<String>(1);
		expNames.add(experimentMap.get(expID).getName());

		Map<String, Map<String, Expsegment>> esMap = ids.getSegments(siteID, ui,
				expNames, subjectIDList);

		List<AssessmentInfo> asiList = null;

		if (siteID == null) {
			asiList = ids.getAssessmentInfos(ui, null, subjectIDList);
		} else {
			// remote site
			asiList = ids.getAssessmentInfos(siteID, null, subjectIDList);
		}

		Map<ObjectInfo, ObjectInfo> expMap = new HashMap<ObjectInfo, ObjectInfo>(
				7);
		Map<ObjectInfo, ObjectInfo> visitMap = new HashMap<ObjectInfo, ObjectInfo>(
				7);
		Map<ObjectInfo, ObjectInfo> segmentMap = new HashMap<ObjectInfo, ObjectInfo>(
				31);
		DataNode expNode = null;
		DataNode visitNode = null;
		DataNode segmentNode = null;

		ObjectInfo scratch = new ObjectInfo();
		for (int i = 0; i < sidbArr.length; i++) {
			SubjectImageDataBundle sidb = sidbArr[i];
			expMap.clear();
			visitMap.clear();
			segmentMap.clear();

			for (Iterator<RawDataObject> iter = rdoDataList.iterator(); iter
					.hasNext();) {
				RawDataObject rdo = iter.next();
				Rawdata rd = rdo.getRawData();
				if (!rd.getSubjectid().equals(sidb.getSubjectID())) {
					continue;
				}
				scratch.setId(rd.getNcExperimentUniqueid().toString());
				scratch.setType("experiment");

				if (expMap.get(scratch) == null) {
					String id = rd.getNcExperimentUniqueid().toString();
					Experiment experiment = experimentMap.get(id);

					sidb.addExpID(experiment.getUniqueid().toString());

					ObjectInfo exp = new ObjectInfo(id, experiment.getName(),
							"experiment");
					expNode = sidb.addObject(exp, sidb.getRoot());
					expMap.put(exp, exp);

					id = rd.getComponentid().toString();
					ObjectInfo visit = new ObjectInfo(id, "Visit " + id, "visit");
					visitNode = sidb.addObject(visit, expNode);
					visitMap.clear();
					visitMap.put(visit, visit);
					prepareAssessmentBranch(sidb, asiList, visitNode);

					Map<String, Expsegment> segListMap = esMap.get(exp.getLabel());
					Assertion.assertNotNull(segListMap);

					Expsegment es = segListMap.get(prepKey(sidb.getSubjectID(), rd));
					Assertion.assertNotNull(es);
					id = rd.getSegmentid().toString();
					ObjectInfo segment = prepareSegmentOI(id, es);
					segmentNode = sidb.addObject(segment, visitNode);
					segmentMap.clear();
					segmentMap.put(segment, segment);
				} else {

					ObjectInfo expOI = expMap.get(scratch);
					Map<String, Expsegment> segListMap = esMap.get(expOI.getLabel());
					Assertion.assertNotNull(segListMap);

					scratch.setId(rd.getComponentid().toString());
					scratch.setType("visit");
					if (visitMap.get(scratch) == null) {
						String id = rd.getComponentid().toString();
						ObjectInfo visit = new ObjectInfo(id, "Visit " + id, "visit");
						visitNode = sidb.addObject(visit, expNode);
						visitMap.put(visit, visit);
						prepareAssessmentBranch(sidb, asiList, visitNode);

						Expsegment es = segListMap.get(prepKey(sidb.getSubjectID(),
								rd));
						Assertion.assertNotNull(es);

						id = rd.getSegmentid().toString();
						ObjectInfo segment = prepareSegmentOI(id, es);
						segmentNode = sidb.addObject(segment, visitNode);
						segmentMap.clear();
						segmentMap.put(segment, segment);
					} else {
						scratch.setId(rd.getSegmentid().toString());
						scratch.setType("segment");
						if (segmentMap.get(scratch) == null) {
							Expsegment es = segListMap.get(prepKey(
									sidb.getSubjectID(), rd));
							Assertion.assertNotNull(es);

							String id = rd.getSegmentid().toString();
							ObjectInfo segment = prepareSegmentOI(id, es);
							segmentNode = sidb.addObject(segment, visitNode);
							segmentMap.put(segment, segment);
						}
					}

				}// expMap
				boolean isDir = ShoppingCartHelper
						.isCollection(rdo.getObjectType());
				int fileType = ShoppingCartHelper.extractFileType(rdo
						.getObjectType());
				if (fileType <= 0) {
					fileType = FileInfo.UNKNOWN;
				}

				int storageType = getStorageType(configService, experimentMap, rdo,
						rd);
            // TODO no longer used remove? 
				if (Constants.USE_BIRN_URIS) {
					String id = rd.getNcExperimentUniqueid().toString();
					Experiment experiment = experimentMap.get(id);
					String baseURI = experiment.getBaseuri();
					Assertion.assertNotNull(baseURI);

					if (BIRNURIUtils.isSRBScheme(baseURI)) {
						storageType = FileInfo.SRB;
					} else if (BIRNURIUtils.isLocalScheme(baseURI)) {
						storageType = FileInfo.LOCAL;
					} else {
						throw new Exception("Unknown URI scheme:" + baseURI);
					}
				}

				FileInfo fi = new FileInfo(rd.getDatauri(), fileType, storageType,
						rootPath, isDir);

				fi.setFileSize(rdo.getObjectSize());

				sidb.addFile(fi, segmentNode, false);
			}
		}
	}

	protected int getStorageType(IAppConfigService configService,
			Map<String, Experiment> experimentMap, RawDataObject rdo, Rawdata rd) {
		String id = rd.getNcExperimentUniqueid().toString();
		Experiment experiment = experimentMap.get(id);

		String storageTypeStr = experiment.getStoragetype();

		int storageType = ShoppingCartHelper.extractStorageType(rdo
				.getObjectType());
		if (storageTypeStr == null) {
			storageTypeStr = configService.getParamValue("storage.type");
		}
		if (storageTypeStr != null) {
			if (storageTypeStr.equalsIgnoreCase("srb"))
				storageType = FileInfo.SRB;
			else if (storageTypeStr.equalsIgnoreCase("gridftp"))
				storageType = FileInfo.GRIDFTP;
			else if (storageTypeStr.equals("local"))
				storageType = FileInfo.LOCAL;
		}
		return storageType;
	}

	protected ObjectInfo prepareSegmentOI(String id, Expsegment es) {
		ObjectInfo segment = null;
		if (es.getName() != null && es.getName().trim().length() > 0) {
			segment = new ObjectInfo(id, es.getName().trim(), "segment");
		} else {
			segment = new ObjectInfo(id, "Segment " + id, "segment");
		}
		return segment;
	}

	protected String prepKey(String subjectID, Rawdata rd) {
		StringBuilder sb = new StringBuilder(100);
		sb.append(subjectID).append('_');
		sb.append(rd.getComponentid()).append('_');
		sb.append(rd.getSegmentid());
		return sb.toString();
	}

	private void prepareAssessmentBranch(SubjectImageDataBundle sidb,
			List<AssessmentInfo> asiList, DataNode visitNode) {
		DataNode asListNode;
		if (!asiList.isEmpty()) {
			ObjectInfo asLabel = new ObjectInfo("Assessment", "Assessment",
					"label");
			asListNode = sidb.addObject(asLabel, visitNode);
			for (Iterator<AssessmentInfo> it = asiList.iterator(); it.hasNext();) {
				AssessmentInfo asi = it.next();
				ObjectInfo oi = new ObjectInfo(asi.getAssessmentIDAsString(), asi
						.getName(), "assessment");
				sidb.addObject(oi, asListNode);
			}
		}
	}

	public static boolean hasFiles(File dir) {
		if (!dir.isDirectory())
			return false;
		return dir.list().length > 0;
	}

	public File[] downloadAndPrepareImageSeries(ImageSeriesDataInfo isdi,
			String cacheRoot, UserInfo ui) throws Exception {
		String cachePath = buildCachePath(cacheRoot, isdi);
		File f = new File(cachePath);
		FileInfo fi = isdi.getFileInfo();

		PluginManager pm = PluginManager.getInstance();
		IPlugin plugin = pm.getPlugin(fi.getFileType());
		if (plugin == null) {
			log.info("No Image conversion plugin support for this file type!");
			return null;
		}
		boolean local = false;
		if (!f.exists() || !hasFiles(f)) {
			List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
			List<PathWrapper> pfList = new ArrayList<PathWrapper>();

			loadFromStorage(pfList, fi, cachePath, ui);
			allFiles.addAll(pfList);

			List<File> imageFiles = prepareImageFiles(allFiles, local, cachePath);
			if (!imageFiles.isEmpty()) {
				File srcDir = new File(imageFiles.get(0).getParent());

				plugin.convertToJpeg(srcDir, new File(cachePath));
			}
			// do cleanup
			File tmpDir = new File(cachePath, "tmp");
			FileUtils.deleteRecursively(tmpDir);
		}

		File[] jpegFileArr = f.listFiles(new SuffixFilenameFilter(".jpeg"));

		return jpegFileArr;
	}

	protected List<File> prepareImageFiles(List<PathWrapper> files,
			boolean local, String cacheDir) throws Exception {
		File tmpDir = new File(cacheDir, "tmp");
		tmpDir.mkdirs();

		if (local) {
			for (PathWrapper pw : files) {
				File f = new File(pw.getSourceRootDir(), pw.getRelPath());
				if (f.isDirectory()) {
					throw new RuntimeException("Directory is not supported:" + f);
				}
				String baseName = f.getName();
				if (baseName.endsWith(".tar") || baseName.endsWith(".tar.gz")
						|| baseName.endsWith(".tgz") || baseName.endsWith("tar.bz2")) {
					Unpacker unpacker = new Unpacker(f.getAbsolutePath(), tmpDir
							.getAbsolutePath());
					unpacker.unpack();
				} else {
					File destFile = new File(tmpDir, f.getName());
					FileUtils.copyFile(f.getAbsolutePath(), destFile
							.getAbsolutePath());
				}
			}

		} else {
			for (PathWrapper pw : files) {
				File f = new File(pw.getSourceRootDir(), pw.getRelPath());
				if (f.isDirectory()) {
					throw new RuntimeException("Directory is not supported:" + f);
				}
				String baseName = f.getName();
				if (baseName.endsWith(".tar") || baseName.endsWith(".tar.gz")
						|| baseName.endsWith(".tgz") || baseName.endsWith("tar.bz2")) {
					Unpacker unpacker = new Unpacker(f.getAbsolutePath(), tmpDir
							.getAbsolutePath());
					unpacker.unpack();
					f.delete();
				} else {
					File destFile = new File(tmpDir, f.getName());
					if (!f.renameTo(destFile)) {
						throw new Exception("Cannot move file to:" + destFile);
					}
				}
			}
		}
		// get a list of files in temp directory
		List<File> allFilesUnderDir = FileUtils.getAllFilesUnderDir(tmpDir
				.getAbsolutePath());
		return allFilesUnderDir;
	}

	public static String buildCachePath(String cacheRoot,
			ImageSeriesDataInfo isdi) {
		StringBuilder sb = new StringBuilder();
		sb.append(cacheRoot).append('/');
		sb.append(isdi.getSubjectID()).append('_');
		sb.append(isdi.getSiteID()).append('_');
		sb.append(isdi.getExpID()).append('_');
		sb.append(isdi.getVisitID()).append('_');
		sb.append(isdi.getSegmentID());

		return sb.toString();
	}

	protected void loadFromStorage(List<PathWrapper> pfList, FileInfo fi,
			String cacheDir, UserInfo ui) throws Exception {
		Map<String, Object> paramMap = new HashMap<String, Object>();
		new File(cacheDir).mkdirs();
		paramMap.put("cacheDir", cacheDir);
		clinical.storage.PluginManager spm = clinical.storage.PluginManager
				.getInstance();
		clinical.storage.IPlugin plugin = null;
		if (fi.getStorageType() == FileInfo.SRB) {
			plugin = spm.getPlugin(FileInfo.SRB_STR, paramMap);
		} else if (fi.getStorageType() == FileInfo.LOCAL) {
			plugin = spm.getPlugin(FileInfo.LOCAL_STR, paramMap);
		} else if (fi.getStorageType() == FileInfo.GRIDFTP) {
			IAppConfigService acs = ServiceFactory.getAppConfigService();
			paramMap.put("myproxy.host", acs
					.getParamValue(Constants.MYPROXY_HOST_KEY));
			paramMap.put("rls.url", acs.getParamValue(Constants.RLS_URL_KEY));
			paramMap.put("gridftp.port", acs
					.getParamValue(Constants.GRIDFTP_PORT_KEY));
			paramMap.put("myproxy.port", acs
					.getParamValue(Constants.MYPROXY_PORT_KEY));
			paramMap.put("master.site.gridftp.host", acs
					.getParamValue(Constants.MASTER_GRIDFTP_HOST_KEY));

			paramMap.put("credential", ui.getCredential());
			plugin = spm.getPlugin(FileInfo.GRIDFTP_STR, paramMap);
		}
		if (plugin != null) {
			try {
				plugin.startup();
				List<PathWrapper> list = plugin.load(fi);
				pfList.addAll(list);
			} finally {
				plugin.shutdown();
			}
		}

	}

	protected void loadFromSRB(SRBFileSystemAdapter sfsAdapter,
			List<PathWrapper> srbFiles, FileInfo fi, String srbPath,
			String cacheDir) throws Exception {
		if (fi.isDir()) {
			File dir = FileInfo.getFullPathFile(fi, fi.getSourceRootDir());
			String dirPath = dir.getAbsolutePath();

			FileInfo nfi = new FileInfo(dirPath, fi.getFileType(), fi
					.getStorageType(), fi.getSourceRootDir(), false);

			String relDirPath = nfi.getRelativePath();
			String localPath = FileUtils.createFullPath(cacheDir, relDirPath);

			System.out.println("**** from SRB '" + srbPath + "' \nto '"
					+ localPath + "'");

			JargonUtils.copyFromSRB(dir.getAbsolutePath(), localPath, sfsAdapter);

			File localDir = new File(FileUtils
					.createFullPath(cacheDir, relDirPath));
			File[] localFilePaths = localDir.listFiles();
			for (int i = 0; i < localFilePaths.length; i++) {
				if (localFilePaths[i].isFile()) {
					String filePath = localFilePaths[i].getAbsolutePath();
					if (filePath.startsWith(cacheDir)) {
						filePath = filePath.substring(cacheDir.length());
						filePath = filePath.replaceFirst("^\\/+", "");
					}
					srbFiles.add(new PathWrapper(filePath, nfi.getSourceRootDir()));
				} else {
					throw new Exception("Unexpected dir: " + localFilePaths[i]);
				}
			}

		} else {
			String relPath = fi.getRelativePath();
			if (FileUtils.buildLocalPath(cacheDir, relPath)) {
				String localPath = FileUtils.createFullPath(cacheDir, relPath);
				JargonUtils.copyFromSRB(srbPath, localPath, sfsAdapter);
				srbFiles.add(new PathWrapper(relPath, cacheDir));
			}
		}
	}

	protected void handleLocal(FileInfo afi, List<PathWrapper> files)
			throws BIRNURIException {
		String rootDir = ""; // FileInfo.findCommonPath(fbc.getFileInfoList());

		if (afi.isDir()) {
			File dir = FileInfo.getFullPathFile(afi, afi.getSourceRootDir());
			List<File> fileList = FileUtils.getAllFilesUnderDir(dir
					.getAbsolutePath());
			for (File file : fileList) {
				FileInfo nfi = new FileInfo(file.getAbsolutePath(), afi
						.getFileType(), afi.getStorageType(), rootDir, false);

				files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
			}

		} else {
			FileInfo nfi = new FileInfo(afi.getPath(), afi.getFileType(), afi
					.getStorageType(), rootDir, false);
			files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
		}

	}
}
