package edu.ucsd.fmri.codegen;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import edu.ucsd.fmri.dbutils.DBTable;
import edu.ucsd.fmri.dbutils.ITypeConversion;
import edu.ucsd.fmri.dbutils.SchemaExtractor;
import edu.ucsd.fmri.dbutils.TypeConversionFactory;
import edu.ucsd.fmri.dbutils.DBTable.DBColumn;

/**
 * @author: I. Burak Ozyurt
 * @version: $Id: BeanGenerator.java,v 1.14 2007/12/11 02:14:21 bozyurt Exp $
 */

public class BeanGenerator {
	protected List<BeanGenHelper> beans = new LinkedList<BeanGenHelper>();
	protected String templateFile;
	protected String loaderPath;
	protected String outRootDir;
	protected String dbSchema;
	protected int codeType;
	protected boolean useViewsAlso;
	protected SchemaExtractor schemaExtractor;
	protected String message;
	protected ITypeConversion typeConverter;
	/** mainly for label security columns which are not hidden */
	protected Map<String, String> excludedColumnsMap = new HashMap<String, String>(
			3);
	protected PropertyChangeSupport pcs = new PropertyChangeSupport(this);
	protected Map<String, TypeExceptionInfo> excepMap;

	public final static int VO = 1;
	public final static int DAO = 2;
	public final static int PK = 3;

	public BeanGenerator(String dbType, String templateFile, String loaderPath,
			String outRootDir, boolean useViewsAlso) throws Exception {

		this.templateFile = templateFile;
		this.loaderPath = loaderPath;

		typeConverter = TypeConversionFactory.getTypeConversionObject(dbType);

		java.net.URL url = BeanGenerator.class.getResource("/" + templateFile);
		if (url != null) {
			String path = url.getPath();
			int idx = path.indexOf(templateFile);
			if (idx != -1) {
				path = path.substring(0, idx);
			}
			this.loaderPath = path;
			// System.out.println("path="+ path);
		}

		java.net.URL excpFileURL = BeanGenerator.class
				.getResource("/exceptions.xml");
		ExceptionsManager excpMan = new ExceptionsManager(excpFileURL);
		this.excepMap = excpMan.getExceptionsMap(dbType);

		this.outRootDir = outRootDir;
		this.useViewsAlso = useViewsAlso;
		excludedColumnsMap.put("sec_label", "sec_label");
	}

	public void addPropertyChangeListener(PropertyChangeListener pcl) {
		pcs.addPropertyChangeListener(pcl);
	}

	public void removePropertyChangeListener(PropertyChangeListener pcl) {
		pcs.removePropertyChangeListener(pcl);
	}

	public void setCodeType(int codeType) {
		this.codeType = codeType;
	}

	public void setDbSchema(String newDbSchema) {
		this.dbSchema = newDbSchema;
	}

	public String getDbSchema() {
		return this.dbSchema;
	}

	public void setMessage(String msg) {
		String oldMsg = message;
		message = msg;
		pcs.firePropertyChange("message", oldMsg, msg);
	}

	public void generate() throws Exception {
		Properties props = new Properties();
		props.setProperty("file.resource.loader.path", loaderPath);
		Velocity.init(props);
		VelocityContext context = null;
		Template template = null;
		Iterator<BeanGenHelper> iter = beans.iterator();
		BufferedWriter out = null;
		while (iter.hasNext()) {
			BeanGenHelper bgh = iter.next();
			if (codeType == DAO) {
				// set exceptions map to handle exceptions to the rule kind type
				// conversions
				if (bgh instanceof DBBeanGenHelper) {
					((DBBeanGenHelper) bgh).setExcpMap(excepMap);
				}
			}

			context = new VelocityContext();
			context.put("bean", bgh);

			template = Velocity.getTemplate(templateFile);

			StringWriter sw = new StringWriter(4096);
			template.merge(context, sw);
			System.out.println(sw);
			try {
				String path = outRootDir + File.separator
						+ packagetoPath(bgh.getPackageName());
				File f = new File(path);
				if (!f.exists()) {
					f.mkdirs();
				}
				String fullPath = path + File.separator + bgh.getBeanName();
				switch (codeType) {
				case VO:
					fullPath += ".java";
					break;
				case DAO:
					fullPath += "DAO.java";
					break;
				case PK:
					fullPath += "PK.java";
					break;
				default:
					throw new RuntimeException("CodeType not supported:"
							+ codeType);
				}
				File file = new File(fullPath);
				if (file.exists()) {
					preserveUserCode(sw, fullPath);
				} else {
					out = new BufferedWriter(new FileWriter(fullPath));
					out.write(sw.toString());
				}
			} catch (IOException iox) {
				iox.printStackTrace();
			} finally {
				if (out != null)
					try {
						out.close();
					} catch (Exception x) { /* ignore */
					}
			}
		}
	}

	protected void writeToFile(StringWriter writer, String path)
			throws IOException {
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(path), 4096);
			out.write(writer.toString());
		} finally {
			if (out != null)
				try {
					out.close();
				} catch (Exception x) { /* ignore */
				}
		}
	}

	protected void preserveUserCode(StringWriter writer, String path)
			throws IOException {
		File file = new File(path);
		File origFile;
		BufferedReader in = null;
		BufferedReader newIn = null;
		BufferedWriter out = null;

		writeToFile(writer, path + ".tmp");

		if (file.renameTo(origFile = new File(path + ".orig"))) {
			try {
				out = new BufferedWriter(new FileWriter(path), 4096);
				in = new BufferedReader(new FileReader(origFile), 4096);
				newIn = new BufferedReader(new FileReader(path + ".tmp"), 4096);
				String line, line2;
				while ((line = newIn.readLine()) != null) {
					if (line.indexOf("/*+++") == -1) {
						out.write(line);
						out.newLine();
					} else {
						out.write(line);
						out.newLine();
						if (findUserCodeDelim(in)) {
							// copy all the user code within delimiters to the
							// new generated code
							while ((line2 = in.readLine()) != null) {
								out.write(line2);
								out.newLine();
								if (line2.indexOf("/*+++") != -1) {
									break;
								}
							}
							// go to the end of user section delimiter
							findUserCodeDelim(newIn);
						}
					}

				}
			} finally {
				if (in != null)
					try {
						in.close();
					} catch (Exception x) {
					}
				if (newIn != null)
					try {
						newIn.close();
					} catch (Exception x) {
					}
				if (out != null)
					try {
						out.close();
					} catch (Exception x) {
					}
			}
		}

		origFile.delete();
		new File(path + ".tmp").delete();
	}

	protected boolean findUserCodeDelim(BufferedReader in) throws IOException {
		String line;
		while ((line = in.readLine()) != null) {
			line = line.trim();
			if (line.length() == 0)
				continue;
			if (line.indexOf("/*+++") != -1)
				return true;
		}
		return false;
	}

	public static void makePath(String root, String relPath) {
		StringTokenizer stok = new StringTokenizer(relPath, File.separator);
		String path = root;
		while (stok.hasMoreTokens()) {
			path += File.separator + stok.nextToken();
			File f = new File(path);
			if (!f.exists()) {
				f.mkdir();
			}
		}
	}

	public static String packagetoPath(String packageName) {
		return packageName.replace('.', File.separatorChar);
	}

	// unit test
	public void testData() {
		// create some canned BeanGenHelpers
		BeanGenHelper bgh = new BeanGenHelper("Subject", "edu.ucsd.fmri.beans");
		bgh.addProperty("name", "String");
		bgh.addProperty("age", "int");
		bgh.addProperty("visits", "java.util.List");

		beans.add(bgh);

	}

	public static String toJavaClassName(String s) {
		char[] carr = s.toCharArray();
		StringBuffer buf = new StringBuffer();
		buf.append(Character.toUpperCase(carr[0]));
		int i = 1;
		while (i < carr.length) {
			if (carr[i] == '_') {
				buf.append(Character.toUpperCase(carr[++i]));
				++i;
				continue;
			}
			buf.append(carr[i]);
			++i;
		}
		return buf.toString();
	}

	public static String toJavaVarName(String s) {
		char[] carr = s.toCharArray();
		StringBuffer buf = new StringBuffer();
		int i = 0;
		while (i < carr.length) {
			if (carr[i] == '_') {
				buf.append(Character.toUpperCase(carr[++i]));
				++i;
				continue;
			}
			buf.append(carr[i]);
			++i;
		}
		String rs = buf.toString();
		if (rs.equalsIgnoreCase("package"))
			rs += "_";
		return rs;
	}

	public void extractDAOInterfaceInfoFromDB(SchemaExtractor se,
			String interfacePackageName, String voPackageName, String[] includes) {
		Map<String, String> includeMap = new HashMap<String, String>();
		if (includes != null) {
			for (int i = 0; i < includes.length; ++i)
				includeMap.put(includes[i].toLowerCase(), includes[i]);
		}
		try {
			Iterator<DBTable> iter = se.getTables();
			while (iter.hasNext()) {
				DBTable table = iter.next();
				if (!includeMap.isEmpty()
						&& includeMap.get(table.getName().toLowerCase()) == null) {
					continue;
				}
				// skip non morph-birn tables
				if (!table.getName().toLowerCase().startsWith("nc_")) {
					continue;
				}

				System.out.println("processing table " + table.getName());
				setMessage("processing table " + table.getName());
				String beanName = table.getName().toLowerCase().substring(3);

				DAOInterfaceGenHelper dig = new DAOInterfaceGenHelper(
						toJavaClassName(beanName), interfacePackageName,
						voPackageName);
				beans.add(dig);
				setMessage("finished generation for " + table.getName());

			}
		} catch (Exception ex) {
			ex.printStackTrace();
			setMessage(ex.getMessage());
		}
	}

	public void extractBeanInfoFromDB(SchemaExtractor se, String packageName,
			String interfacePackageName, String voPackageName,
			String expPackageName, String utilsPackageName, String[] includes) {
		Map<String, String> includeMap = new HashMap<String, String>();
		if (includes != null) {
			for (int i = 0; i < includes.length; ++i)
				includeMap.put(includes[i].toLowerCase(), includes[i]);
		}
		try {
			Iterator<DBTable> iter = se.getTables();
			while (iter.hasNext()) {
				DBTable table = iter.next();
				if (!includeMap.isEmpty()
						&& includeMap.get(table.getName().toLowerCase()) == null) {
					continue;
				}
				// skip non morph-birn tables
				if (!table.getName().toLowerCase().startsWith("nc_")) {
					continue;
				}

				System.out.println("processing table " + table.getName());
				setMessage("processing table " + table.getName());
				String beanName = table.getName().toLowerCase().substring(3);
				BeanGenHelper bgh = new DBBeanGenHelper(
						toJavaClassName(beanName), packageName, table.getName());
				if (voPackageName != null) {
					bgh = new DBBeanGenHelper(toJavaClassName(beanName),
							packageName, table.getName());
					((DBBeanGenHelper) bgh).setVoPackageName(voPackageName);
				} else {
					bgh = new BeanGenHelper(toJavaClassName(beanName),
							packageName);
				}

				if (expPackageName != null) {
					((DBBeanGenHelper) bgh).setExpPackageName(expPackageName);
				}

				if (utilsPackageName != null) {
					((DBBeanGenHelper) bgh)
							.setUtilsPackageName(utilsPackageName);
				}
				if (interfacePackageName != null) {
					((DBBeanGenHelper) bgh)
							.setInterfacePackageName(interfacePackageName);
				}

				if (codeType == PK) {
					for (DBTable.DBColumn column : table.getPrimaryKeyColumns()) {
						String colName = column.getName().toLowerCase();
						bgh.addProperty(toJavaVarName(colName), typeConverter
								.toJavaType(column));
						if (voPackageName != null) {
							((DBBeanGenHelper) bgh).addColMap(
									toJavaVarName(colName), column.getName());
							((DBBeanGenHelper) bgh).addColMapType(
									toJavaVarName(colName), typeConverter
											.getConversionType(column));
						}
					}

				} else {
					for (Object element : table.getColumns()) {
						List<DBColumn> pkCols = table.getPrimaryKeyColumns();
						DBColumn column = (DBColumn) element;
						String colName = column.getName().toLowerCase();
						// skip columns
						if (excludedColumnsMap.get(colName) != null) {
							continue;
						}

						bgh.addProperty(toJavaVarName(colName), typeConverter
								.toJavaType(column));

						if (voPackageName != null) {

							((DBBeanGenHelper) bgh).addColMap(
									toJavaVarName(colName), column.getName());
							((DBBeanGenHelper) bgh).addColMapType(
									toJavaVarName(colName), typeConverter
											.getConversionType(column));
							((DBBeanGenHelper) bgh).addSQLTypeType(
									toJavaVarName(colName), typeConverter
											.getSQLType(column));
							if (pkCols == null) {
								System.err
										.println("There is no primary key for table "
												+ table.getName());
							} else if (pkCols.contains(column)) {
								((DBBeanGenHelper) bgh)
										.addPkColumn(toJavaVarName(colName));
							}

						}
					}
				}
				beans.add(bgh);
				setMessage("finished generation for " + table.getName());
			}
		} catch (Exception ex) {
			ex.printStackTrace();
			setMessage(ex.getMessage());
		}

	}

	public static SchemaExtractor createSchemaExtractor(String driverName,
			String dbURL, String schemaName, String usr, String pwd,
			boolean useViewsAlso) throws Exception {
		SchemaExtractor se = new SchemaExtractor(driverName, dbURL, usr, pwd);
		se.loadDBTables(schemaName, useViewsAlso, true); // only load BIRN DB
		// tables
		return se;
	}

	public static void generate(String dbType, SchemaExtractor se,
			ConfigurationInfo ci, String codeType, String[] tableList,
			PropertyChangeListener listener) throws Exception {
		BeanGenerator bg = null;
		if (codeType.equals("vo")) {

			bg = new BeanGenerator(dbType, "JavaBean.vm", ci.getLoaderPath(),
					ci.getOutRootDir(), true);
			bg.addPropertyChangeListener(listener);
			bg.setDbSchema(ci.getDbSchema());
			bg.setCodeType(BeanGenerator.VO);
			bg.extractBeanInfoFromDB(se, ci.getPackage("vo"), // "clinical.server.vo",
					null, null, null, null, tableList);
		} else if (codeType.equals("dao")) {
			String velocityTemplate = dbType.equals("postgres") ? "PostgresDAO.vm"
					: "DAO.vm";

			bg = new BeanGenerator(dbType, velocityTemplate,
					ci.getLoaderPath(), ci.getOutRootDir(), true);
			bg.addPropertyChangeListener(listener);
			bg.setDbSchema(ci.getDbSchema());
			bg.setCodeType(BeanGenerator.DAO);

			String daoImplPackage = null;
			if (ci.getDbType().equals("oracle")) {
				daoImplPackage = ci.getPackage("dao_oracle");
			} else {
				daoImplPackage = ci.getPackage("dao_postgres");
			}

			bg.extractBeanInfoFromDB(se, daoImplPackage, // "clinical.server.dao.oracle|postgres",
					ci.getPackage("interface"), ci.getPackage("vo"), // "clinical.server.vo",
					ci.getPackage("exception"), // "clinical.exception",
					ci.getPackage("utils"), // "clinical.server.utils",
					tableList);

		} else if (codeType.equals("pk")) {
			bg = new BeanGenerator(dbType, "PK.vm", ci.getLoaderPath(), ci
					.getOutRootDir(), true);
			bg.addPropertyChangeListener(listener);
			bg.setDbSchema(ci.getDbSchema());
			bg.setCodeType(BeanGenerator.PK);
			bg.extractBeanInfoFromDB(se,
					ci.getPackage("pk") /* "clinical.server.ejb" */, null,
					null, null, null, tableList);
		} else
			throw new RuntimeException("Not a supported code type!");

		bg.generate();

		if (codeType.equals("dao")) {

			String velocityTemplate = "DAOInterface.vm";

			bg = new BeanGenerator(dbType, velocityTemplate,
					ci.getLoaderPath(), ci.getOutRootDir(), true);
			bg.addPropertyChangeListener(listener);
			bg.setDbSchema(ci.getDbSchema());
			bg.setCodeType(BeanGenerator.DAO);
			bg.extractDAOInterfaceInfoFromDB(se, ci.getPackage("interface"), ci
					.getPackage("vo"), tableList);

			bg.generate();
		}
	}

}
