package edu.ucsd.fmri.codegen;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import edu.ucsd.fmri.dbutils.DBTable;
import edu.ucsd.fmri.dbutils.SchemaExtractor;
import edu.ucsd.fmri.utils.CLIUtils;
import edu.ucsd.fmri.utils.ConfigurationUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CodeGenCLI {
	BeanGenerator beanGenerator;
	SchemaExtractor schemaExtractor;
	ConfigurationInfo ci = new ConfigurationInfo();

	public CodeGenCLI(String pwd, Properties props) {
		ci = new ConfigurationInfo();
		ci.setPwd(pwd);
		ci.setDbType("postgres");
		ci.setDbURL(props.getProperty("db.url"));
		ci.setUser(props.getProperty("db.user"));
		ci.setOutRootDir(props.getProperty("output.root.dir"));
		System.out.println("outputDir:" + ci.getOutRootDir());
		System.out.println("dbURL:" + ci.getDbURL());
		System.out.println("======================");
	}

	public void init() throws Exception {
		String schemaName = ci.getDbSchema();
		if (ci.getDBType().equals("postgres")) {
			schemaName = null;
		}

		schemaExtractor = BeanGenerator.createSchemaExtractor(ci.getDriverName(),
				ci.getDbURL(), schemaName, ci.getUser(), ci.getPwd(), true);

	}

	public void generateCode(String srcOutputDir) throws Exception {
		List<String> schemaObjNames = new ArrayList<String>(100);
		for (Iterator<DBTable> iter = schemaExtractor.getTables(); iter.hasNext();) {
			DBTable table = iter.next();
			schemaObjNames.add(table.getName());
		}
		Collections.sort(schemaObjNames);
		String[] table = CLIUtils.prepTable(schemaObjNames, 3);
		CLIUtils.showTable(table);
		Set<Integer> selections = CLIUtils.selectFromTable(schemaObjNames.size(),
				"Please select schema objects for code generation.");

		// code type
		List<String> codeTypes = new ArrayList<String>(3);
		codeTypes.add("vo");
		codeTypes.add("dao");
		CLIUtils.showTable(CLIUtils.prepTable(codeTypes, 1));
		Set<Integer> codeTypeSelections = CLIUtils.selectFromTable(codeTypes.size(),
				"Please select code type.");
		String codeType = codeTypes.get(codeTypeSelections.iterator().next());
		String[] tableList = new String[selections.size()];
		int i = 0;
		for (Integer selIdx : selections) {
			tableList[i++] = schemaObjNames.get(selIdx);
		}
		if (srcOutputDir != null) {
			ci.setOutRootDir(srcOutputDir);
		}
		if (ci.getOutRootDir() == null) {
			throw new Exception("No output source root directory is set!");
		}
		BeanGenerator.generate(ci.getDbType(), schemaExtractor, ci, codeType,
				tableList, null);

		// also fake Oracle code generation by using Postgres type code generation
		// for Oracle DAOs
		if (codeType.equals("dao")) {
			ci.setPackage("dao_postgres", "clinical.server.dao.oracle");
			BeanGenerator.generate(ci.getDbType(), schemaExtractor, ci, codeType,
					tableList, null);
		}
	}

	public static void usage() {
		System.err
				.println("Usage: CodeGenCLI -props <properties-file> -p <db-pwd> [-o <output-src-root-dir>]");
		System.exit(1);
	}

	public static String getArgValue(int idx, String[] args) {
		if (args.length <= idx) {
			System.out.println("Missing argument value!");
			usage();
		}
		return args[idx];
	}

	public static void main(String[] args) throws Exception {
		if (args.length < 4) {
			usage();
		}
		String propsFile = null;
		String pwd = null;
		String outSrcRootDir = null;
		int i = 0;
		while (i < args.length) {
			if (args[i].equals("-props")) {
				propsFile = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-p")) {
				pwd = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-o")) {
				outSrcRootDir = getArgValue(i + 1, args);
				i += 2;
			} else {
				i++;
			}
		}
		if (propsFile == null || pwd == null) {
			usage();
		}
		if (outSrcRootDir != null) {
			if (!new File(outSrcRootDir).isDirectory()) {
				System.err.println("Not a valid output source root dir:"
						+ outSrcRootDir);
				usage();
			}
		}
		Properties props = ConfigurationUtils.loadProperties(propsFile);
		CodeGenCLI cli = new CodeGenCLI(pwd, props);
		cli.init();
		cli.generateCode(outSrcRootDir);
	}

}
