#!/usr/bin/perl

##############################################################################
# cold_restore.pl
# Script to restore an FBIRN Oracle database instance from a cold backup.
#
#
#
##############################################################################

my ($bfile) = @ARGV;
my $cur_dir = `pwd`;
chomp($cur_dir);
my $dir;

if($#ARGV < 0)
{
    print "\nUSAGE: ./cold_restore cold_backup.tar.gz \n   OR: ./cold_restore cold_backup_dir \n\n";
    exit;
}
elsif(! -e $bfile)
{
    print "\nERROR: $bfile does not exist.\n\n";
    exit;
}

########################################################
# Check if database files already exist.
# If they do, prompt user if they want to overwrite.
# If so, check if the database being overwritten is still
# running, and, if so, inform the user and exit.
# Otherwise, check for the existence of database directories
# and copy the files into them, or prompt the user about
# creating them.
########################################################

&checkDatabase();

&checkDirs();

if( $bfile =~ m/\.tar\.gz/ )  ## Need to unzip and untar the backup file into a temp directory.
{
    ## Determine amount of space needed;
    $req_space = `gzip -cd $bfile | wc -c`;
    chomp($req_space);

    ## Check for space.
    @df = `df ~`;

    @space = split(/\s+/,$df[1]);
    chomp($space[3]);

    print "\n   Space needed: $req_space \nSpace available: " . $space[3] * 1000 . " \n";

    if($space[3] * 1000 <= $req_space)
    {
	print "\nThere is not enough space in your home directory to unzip this file.\n\n";
	&myExit();
    }
    
    $date = `date +%s`;
    chomp($date);

    ## Set the temp directory
    $home = `echo ~`;
    chomp($home);
    $dir = $home . "/db_backup/$date";

    if(system("mkdir $dir") != 0)
    {
	print "\nmkdir failed with exit status $?\n\n";
	&myExit();
    }

    chdir("$dir") or die "\nCan't chdir to $dir: $!\n\n";

    if(system("tar -xvzf $cur_dir/$bfile") != 0)
    {
	print "\ntar failed with exit status $?\n\n";
	&myExit();
    }

    chdir("nas/nas0/oracle_backup/hidprd/cold/temp/") || die "\nCan't chdir to nas/nas0/oracle_backup/hidprd/cold/temp\n\n";
}
else
{
    chdir($bfile) || die "\nERROR: Can't chdir to $bfile: $!\n\n";
}


&copyFiles();

if( -e $dir)
{
    if(system("rm -rf $dir") != 0)
    {
	print "\nUnable to remove $dir, rm failed with exit code $?\n\n";
	&myExit();
    }
}
sub checkDatabase
{
    $lsn = `lsnrctl status`;

    if ($lsn =~ m/hidprd/)
    {
	print "\nThere is an instance of hidprd currently running.  Please stop the database and rerun cold_restore.pl.\n\n";
	&myExit();
    }
    elsif ( -e "/dbms/oracle/v02/oradata/hidprd/control01.ctl"
	    || -e "/dbms/oracle/v02/oradata/hidprd/hidprd_sysaux01.dbf"
	    || -e "/dbms/oracle/v02/oradata/hidprd/hidprd_system01.dbf"
	    || -e "/dbms/oracle/v03/oradata/hidprd/control02.ctl"
	    || -e "/dbms/oracle/v03/oradata/hidprd/hidprd_redo01b.log"
	    || -e "/dbms/oracle/v03/oradata/hidprd/hidprd_redo02a.log"
	    || -e "/dbms/oracle/v03/oradata/hidprd/hidprd_redo03b.log"
	    || -e "/dbms/oracle/v03/oradata/hidprd/hidprd_unotbs01.dbf"
	    || -e "/dbms/oracle/v04/oradata/hidprd/control03.ctl"
	    || -e "/dbms/oracle/v04/oradata/hidprd/hidprd_data_01.dbf"
	    || -e "/dbms/oracle/v04/oradata/hidprd/tools01.dbf"
	    || -e "/dbms/oracle/v05/oradata/hidprd/hidprd_indx_01.dbf"
	    || -e "/dbms/oracle/v06/oradata/hidprd/hidprd_temp01.dbf"
	    || -e "/dbms/oracle/v09/oradata/hidprd/hidprd_redo01a.log"
	    || -e "/dbms/oracle/v09/oradata/hidprd/hidprd_redo02b.log"
	    || -e "/dbms/oracle/v09/oradata/hidprd/hidprd_redo03a.log"
	    || -e "/dbms/oracle/v01/app/oracle/product/10.2.0/dbs/inithidprd.ora"
	    || -e "/dbms/oracle/v01/app/oracle/product/10.2.0/dbs/spfilehidprd.ora"
	    || -e "/dbms/oracle/v01/app/oracle/admin/hidprd/admin/hidprd")
    {
	#print "\nRESTORE FAILED: /dbms/oracle/v02/oradata/hidprd/control01.ctl already exists.\n\n";
	#&myExit();
	$p_done = 0;
	while(! $p_done)
	{
	    print "\nAt least one of the hidprd files already exist.  BEWARE!  If you choose to overwrite, it's permanent.  Whatever database that currently exists will be replaced with the database in the cold backup.  Use at your own risk!.\n\n";
	    $res = &promptUser("\nAt least one of the database files already exists,\ndo you want to overwrite it/them? (Y/N) ","N");
	    if( $res =~ m/[n|N]/)
	    {
		print "\n Restore terminated due to existence of datafiles that can't be overwritten.\n\n";
		&myExit();
	    }
	    elsif( $res =~ m/[y|Y]/)
	    {
		print "\nOverwriting.\n";
		$p_done = 1;
	    }
	    else
	    {
		print "\nIncorrect response\n";
	    }
	}
    }
    
}

sub checkDirs
{
    &checkDirSub("/dbms/oracle/v02/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v03/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v04/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v05/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v06/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v07/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v08/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v09/oradata/hidprd/");
    &checkDirSub("/dbms/oracle/v01/app/oracle/admin/hidprd/admin/hidprd");

    if (! -e "/dbms/oracle/v01/app/oracle/product/10.2.0/dbs/")
    {
	print "\nRESTORE FAILED: /dbms/oracle/v01/app/oracle/product/10.2.0/dbs/ does not exist or is not readable.\n\n";
	print "Is Oracle installed?\n\n";
	&myExit();
    }

}

sub checkDirSub
{
    local($new_dir) = @_;

    if (! -e $new_dir)
    {
	print "\nRESTORE ERROR:$new_dir does not exist or is not readable\n";

	print "Please create the hidprd directories by running mkdir_hidprd.sh.\n";
	print "Available from cvs.\n\n";
	&myExit();
    }
}

sub copyFiles
{
    
    if(system("cp control01.ctl /dbms/oracle/v02/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp arch/* /dbms/oracle/v02/oradata/hidprd/arch/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_sysaux01.dbf /dbms/oracle/v02/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_system01.dbf /dbms/oracle/v02/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp control02.ctl /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo01b.log /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo02a.log /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo03b.log /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_undotbs01.dbf /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp control03.ctl /dbms/oracle/v04/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_data_01.dbf /dbms/oracle/v04/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp tools01.dbf /dbms/oracle/v04/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_indx_01.dbf /dbms/oracle/v05/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_temp01.dbf /dbms/oracle/v06/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo01a.log /dbms/oracle/v09/oradata/hidprd") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo02b.log /dbms/oracle/v09/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp hidprd_redo03a.log /dbms/oracle/v09/oradata/hidprd/") != 0)
    {
	&cpError();
    }
    if(system("cp inithidprd.ora /dbms/oracle/v01/app/oracle/product/10.2.0/dbs/") != 0)
    {
	&cpError();
    }
    if(system("cp spfilehidprd.ora /dbms/oracle/v01/app/oracle/product/10.2.0/dbs/") != 0)
    {
	&cpError();
    }

    if(system("cp -r hidprd/ /dbms/oracle/v01/app/oracle/admin/hidprd/admin/") != 0)
    {
	&cpError();
    }
    if(system("chown oracle:oinstall /dbms/oracle/v01/app/oracle/product/10.2.0/dbs/*hidprd*") != 0)
    {
    	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v02/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v03/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v04/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v05/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v06/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v07/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v08/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v09/oradata/hidprd/") != 0)
    {
	&chError();
    }
    if(system("chown -R oracle:oinstall /dbms/oracle/v01/app/oracle/admin/hidprd/") != 0)
    {
	&chError();
    }

}

sub cpError
{
    print "\nCopying failed, exit status = $?\n\n";
    &myExit();
}

sub chError
{
    print "\nChown failed, exit status = $?\n\n";
    &myExit();
}

sub myExit
{
    chdir("$cur_dir") or die "\nCan't chdir to $cur_dir: $!\n\n";
    exit;
}

sub promptUser 
{

    local($promptString,$defaultValue) = @_;

    if ($defaultValue) 
    {
	print $promptString, "[",$defaultValue,"]: ";
    }
    else
    {
	print $promptString, ": ";
    }

    $| = 1;
    $_ = <STDIN>;

    chomp;

    if("$defaultValue") 
    {
	return $_ ? $_ : $defaultValue; 
    }
    else
    {
	return $_;
    }
}
