#!/usr/bin/perl

#############################################################################
# HISTORY
#
# Date      Author         Change description
# ---------------------------------------------------------------------------
# 05/14/03  Bobby Higbie   In the copyFile procedure, sleep for 5 sec. to ensure
#                          subsequent calls won't interfere with the copy.
# 12/12/03  Bobby Higbie   Replaced File copy with opening a file handler and 
#                          piping a copy command in the copyFile procedure.
#                          Commented out code that purged the backup archive
#                          logs.
#                          Added to uppercase to allow case-insensitive tablespace
#                          specifications
#                          Added sleep 15 to doLogSwitch
#
# PSEUDOCODE
#
#  1. Setup environment for OS specific details 
#  2. Get arguments from dbbackup.sh:  SID
#  3. Get parameter values from config file:  dbbackup_<SID>.cfg
#     - USER           - ARCH_BU_DIR
#     - BACKUP_DIR     - EXCL_DF
#     - EXCL_TS
#  4. Connect to db
#  5. Backup the following in order.  Add an entry in the restore
#     script per file backed up.
#     - init.ora
#     - listener.ora
#     - sqlnet.ora
#     - tnsnames.ora
#     - controlfile (binary & trace)
#     - system ts
#     - other ts (excluding ts/df specified in config dbbackup_<SID>.cfg
#     - Archive Logs
#
# Methods:
# backupOraConfig();
# backupCTL();
# beginBuTs("tsname");
# endBuTs("tsname");
# copyFile(srcFile, destFile);
# doLogSwitch();
# getConfigParams();
# getOSEnv();
# purgeArch();
# writeLog("Log entry");
# writeRestore(srcFile, destPath);
# writeRestore(srcFile, destFile);
################################################################

use DBI;
use File::Path;
use File::Copy;
use File::Find;
use File::Basename;

my $version = "1.1";

# Arguments Setup
my ($sid, $cfg) = @ARGV;
# File Setup
my ($fpath, $fname, $tns_admin, $init_ora, $prog_dir);
my $ctl_b_name="";
my $ctl_t_name="";
# DB Specific
my ($p2, @df_excluded, @ts_excluded, $bu_dir, $arch_bu_dir, $arch_retention_days);
my ($dbh, $tssql, $dfsql, $sql, $p1);
my ($ts, $df);
my $skip_ts="F";
my $skip_df="F";
# OS Specific
my ($dsn, $copy_cmd);

###############################################################################
# Start
###############################################################################
getOSEnv();
if (! getConfigParams()) { exit(1) };
writeLog("DBBACKUP version: $version");

$ENV{'ORACLE_HOME'}='/dbms/oracle/v01/app/oracle/product/10.2.0';
$ENV{'ORACLE_SID'}='hidprd';
#$ENV{'NLS_LANG'}='AMERICAN_AMERICA.WE8ISO8859P1';
$ENV{'NLS_LANG'}='AMERICAN_AMERICA.US7ASCII';

#$dbh = DBI->connect($dsn, "system", $p2) || die writeLog("ERROR:  Unable to connect to $sid.  $DBI::errstr"); 
$dbh = DBI->connect("DBI:Oracle:", "/", "",{ora_session_mode=>2}) || die writeLog("ERROR:  Unable to connect to $sid.  $DBI::errstr"); 
writeLog("Successfully connected to $sid");

backupOraConfig();
doLogSwitch();
backupCTL();

# Backup tablespaces
# Get list of tablespaces to backup
$tssql = $dbh->prepare("select tablespace_name
                          from dba_tablespaces
                         where contents != 'TEMPORARY'");

if ($tssql != undef) {
  $tssql->execute();
  while ($ts = $tssql->fetchrow_array) {
    chomp $ts;

    # Determine if this tablespace is excluded from backup
    foreach $ts_excl(@ts_excluded) {
      chomp $ts_excl;
      #12/12/03 Bobby Higbie
      #Added to uppercase to allow case-insensitive tablespace specifications
      if (uc($ts) eq uc($ts_excl)) {
        $skip_ts = "T";
      }
    }

    # Begin backup procedure
    if ($skip_ts eq "T") {
      writeLog("Skipping tablespace $ts");
    }
    else {
      beginBuTs($ts);
      #Get list of datafiles in tablespace $ts
      $dfsql = $dbh->prepare("select file_name 
                                from dba_data_files
                               where tablespace_name = ?");
      if ($dfsql != undef) {
        $dfsql->bind_param(1,$ts);
        $dfsql->execute();
        while ($df = $dfsql->fetchrow_array) {
          #Determine if this file is excluded
          foreach $df_excl(@df_excluded) {
            if ($df eq $df_excl) {
              $skip_df = "T";
            }
          } #end of for

          if ($skip_df eq "T") {
            writeLog("Skipping $df in tablespace $ts");
          } 
          else {
            $fpath = dirname($df);
            $fname = basename($df);
            
            # Copy the datafile to the backup dir
            copyFile($df,$bu_dir . "\/" . $fname);

            # Add entry to restore script
            writeRestore($bu_dir . "\/" . $fname, $fpath);
            $fname = "";
            $fpath = "";
          } #end if $skip_df eq "T"
          $skip_df = "F";
        } #end while
        $dfsql->finish();
      } #end if $dfsql != undef
      endBuTs($ts);
    } #end if $skip_ts eq "T"
    $skip_ts = "F";
  }
  $tssql->finish();
}

#Must do log switch before backing up the archive logs
#so we get the last archive log to recover.
doLogSwitch();

backupArch();

$dbh->disconnect || die writeLog("ERROR: Exception occurred while disconnecting from database.  $!");
writeLog("Successfully disconnected from $sid");
writeLog("Successfully completed backup for $sid");

##############################################################################
# Subroutines
##############################################################################
sub backupArch {
  my ($arch_dir,$arch_name,$arch_full_name, $arch_bu_full_name);

  #Get archive dir
  $sql = $dbh->prepare("select destination
                          from v\$archive_dest
                         where destination is not null
                           and status = 'VALID'
                           and rownum = 1");
  $sql->execute();
  $arch_dir = $sql->fetchrow_array();
  chomp $arch_dir;
  $sql->finish();

  #Loop through all the files in the arch dir and check if it's in
  #the arch backup dir.  If not, then copy it.
  opendir(ARCHDIR, $arch_dir) || writeLog("ERROR: Unable to open arch dir $arch_dir.  $!");
  while (defined($arch_name = readdir(ARCHDIR))) {
    next if $arch_name =~ /\.\.?$/;
    chomp $arch_name;
    $arch_full_name = $arch_dir . "\/" . $arch_name;
    $arch_bu_full_name = $arch_bu_dir . "\/" . $arch_name;

    #Purge archive files as specified by $arch_retention_days
    if ( !(-M $arch_full_name <= $arch_retention_days) ) {
      purgeArch($arch_full_name);
    }

    #Backup the arch and purge unnecessary backup archives
    if ( ! -e $arch_bu_full_name) {
      copyFile($arch_full_name,$arch_bu_full_name);
      writeRestore($arch_bu_full_name,$arch_full_name);
    }
    #No longer purging backup archive logs:  12/10/03 Bobby Higbie
    #else {
       #Purge backup archive files as specified by $arch_retention_days
    #  if ( !(-M $arch_bu_full_name <= $arch_retention_days+1) ) {
    #    purgeArch($arch_bu_full_name);
    #  }
    #}
  }

  closedir(ARCHDIR);
}

sub backupCTL {
  my $bu_ctl="";
  my ($ctl_name, $trc_dir, $trc_name, $trc_full_name);
  my $one_min = 1/1440 ; #1hr = 1/24; 1min = 1/60 * 1hr;
  my $line = "";

  #Backup the binary controlfile
  if ($ctl_b_name eq "") {
    $ctl_b_name = "control.ctl";
  }
  $bu_ctl = $bu_dir . "\/" . $ctl_b_name;

  #Need to delete backup binary controlfile if it exists.  The following do 
  #command would fail otherwise.
  if (-e $bu_ctl) { unlink($bu_ctl) || warn writeLog("Unable to delete the backup controlfile $bu_ctl.  $!"); }

  writeLog("Backing up binary controlfile $ctl_b_name to $bu_dir");
  if ($dbh->do("alter database backup controlfile to '$bu_ctl'")) {
    writeLog("Successfully backed up $ctl_b_name");
    $sql = $dbh->prepare("select name from v\$controlfile");
    $sql->execute();
    while ($ctl_name = $sql->fetchrow_array()) {
      chomp $ctl_name;
      writeRestore($bu_ctl, $ctl_name);
    }
    $sql->finish();
  }
  else {
    writeLog("ERROR: error occurred while backing up $bu_ctl.  Exiting due to critical error.  $!");
    exit(1);
  }

  #Backup the create controlfile script
  if ($ctl_t_name eq "") {
    $ctl_t_name = "createCTL.sql";
  }
  $bu_ctl = $bu_dir . "\/" . $ctl_t_name;
  writeLog("Backing up create controlfile script $ctl_t_name to $bu_dir");
  if ($dbh->do("alter database backup controlfile to trace")) {
    writeLog("Successfully backed up create controlfile script $ctl_t_name");

    #Find the trace file created and rename it to $bu_dir/$ctl_t_name
    $sql = $dbh->prepare("select value
                            from v\$parameter
                           where name = 'user_dump_dest'");
    $sql->execute();
    $trc_dir = $sql->fetchrow_array();
    $sql->finish();
    opendir(TRCDIR, $trc_dir) || writeLog("ERROR: Unable to open trace dir $trc_dir.  $!");
    while (defined($trc_name = readdir(TRCDIR))) {
      $trc_full_name = $trc_dir . "\/" . $trc_name;
      #Determine if the file is only 2 min old and it has
      #CREATE CONTROLFILE in it.
      if (-M $trc_full_name <= (2 * $one_min)) {
        open(TRC, $trc_full_name) || writeLog("ERROR: Unable to open trace file $trc_full_name. $!");
        while ($line = <TRC>) {
          if ($line =~ /CREATE CONTROLFILE/) {
            copyFile($trc_full_name,$bu_dir . "\/" . $ctl_t_name);
            unlink($trc_full_name);
            last;
          }
        }
        close(TRC);
      }
    }
    closedir(TRCDIR);
  }
}

sub backupOraConfig {
  writeLog("Backing up initialization file $init_ora");
  $fname = basename($init_ora);
  $fpath = dirname($init_ora);
  copyFile($init_ora,$bu_dir . "\/" . $fname);
  writeRestore($bu_dir . "\/" . $fname, $fpath);
  $fname = "";
  $fpath = "";

  writeLog("Backing up files in TNS_ADMIN");
  copyFile($tns_admin . "\/sqlnet.ora",$bu_dir . "\/sqlnet.ora");
  copyFile($tns_admin . "\/listener.ora",$bu_dir . "\/listener.ora");
  copyFile($tns_admin . "\/tnsnames.ora",$bu_dir . "\/tnsnames.ora");
  writeRestore($bu_dir . "\/sqlnet.ora", $tns_admin);
  writeRestore($bu_dir . "\/listener.ora", $tns_admin);
  writeRestore($bu_dir . "\/tnsnames.ora", $tns_admin);
}

sub beginBuTs {
  my $tsname = shift;
  if ($dbh->do("alter tablespace $tsname begin backup")) {
    writeLog("Starting backup of $tsname");
  }
  else {
    writeLog("ERROR:  Error occurred while starting backup of $tsname. $!");
    exit(1);
  }
}

sub endBuTs {
  my $tsname = shift;
  if ($dbh->do("alter tablespace $tsname end backup")) {
    writeLog("Finished backup of $tsname");
  }
  else {
    writeLog("ERROR:  Error occurred while ending backup of $tsname. $!");
    exit(1);
  }
}

sub copyFile {
  my ($src,$dest) = @_;
  my $destpath;

  $dest_path = dirname($dest);
  if (! -e $dest_path) {
    eval { mkpath($dest_path) };
    if ($@) { writeLog("ERROR:  Unable to create dir $dest_path:  $@"); }
  }

  if (-e $src) {
    #Added 12/12/03 Bobby Higbie
    #Use file handle rather than the File copy method.  The File copy
    #method has a bug where the procedure will exit before the file
    #has finished copying.  In other words, the destination file
    #is corrupt.
    open(COPY, "|cp $src $dest" );
    close(COPY);
    writeLog("Copied $src to $dest");
    #if (copy($src,$dest) == 1) {
    #  writeLog("Copied $src to $dest");
    #}
    #else {
    #  writeLog("ERROR: $!");
    #}
  }
  else {
    writeLog("ERROR:  Unable to find $src:  $!");
  }

  #Sleep for 5 sec. to ensure subsequent calls won't interfere with the copy.
  #Added 5/14/2003 when we discovered the last archive log copied was corrupt.
  #The copied version was always 512 bytes regardless what size the source file
  #was.
  sleep 5;
}

sub doLogSwitch {

  writeLog("Starting Log Switch.");
  if ($dbh->do("alter system switch logfile")) {
    writeLog("Successfully completed Log Switch.");

    #Sleep for 15 sec to allow proper time for logswitch
    sleep 15;
  }
  else {
    writeLog("ERROR: Unsuccessful log switch.");
    exit(1);
  }

}

sub getConfigParams {
  my $line;

  if (! -e $cfg) { 
    writeLog("ERROR:  Unable to find $cfg:  $!");
    return;
  }
  else {
    open( CFG, "< $cfg" ) || die writeLog("ERROR:  Unable to open file $cfg:  $!");

    while ($line = <CFG>) {
      $line =~ s/\s+//g;
      #Process the line if it's not a comment
      if (substr($line,0,1) ne "#") { 
        chomp $line;
        #if ($line =~ m/USER/) { 
        #  $line =~ s/USER(\s*)=(\s*)//g; 
        #  $line =~ s/\"//g; 
        #  $p2 = $line; 
        #}
        if ($line =~ m/PROG_DIR/) { 
          $line =~ s/PROG_DIR(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $prog_dir = $line; 
        }
        if ($line =~ m/TNS_ADMIN/) { 
          $line =~ s/TNS_ADMIN(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $tns_admin = $line; 
        }
        if ($line =~ m/INIT_ORA/) { 
          $line =~ s/INIT_ORA(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $init_ora = $line; 
        }
        if ($line =~ m/CTL_B_NAME/) { 
          $line =~ s/CTL_B_NAME(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $ctl_b_name = $line;
        }
        if ($line =~ m/CTL_T_NAME/) {  
          $line =~ s/CTL_T_NAME(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $ctl_t_name = $line; 
        }
        if ($line =~ m/BACKUP_DIR/) { 
          $line =~ s/BACKUP_DIR(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $bu_dir = $line;
        }
        if ($line =~ m/ARCH_BU_DIR/) { 
          $line =~ s/ARCH_BU_DIR(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $arch_bu_dir = $line; 
          chomp $arch_bu_dir;
        }
        if ($line =~ m/ARCH_RETENTION_DAYS/) { 
          $line =~ s/ARCH_RETENTION_DAYS(\s*)=(\s*)//g; 
          $line =~ s/\"//g; 
          $arch_retention_days = $line; 
        }
        if ($line =~ m/EXCL_TS/) { 
          $line =~ s/EXCL_TS(\s*)=(\s*)//g; 
          $line =~ s/[\"\(\)\s*]//g; 
          @ts_excluded = split(/,\s*/, $line); 
        }
        if ($line =~ m/EXCL_DF/) { 
          $line =~ s/EXCL_DF(\s*)=(\s*)//g; 
          $line =~ s/[\"\(\)\s*]//g; 
          @df_excluded = split(/,\s*/, $line); 
        }
      }
    }
    writeLog("Retrieved backup parameters from $cfg");
    close(CFG);
    return 1; #returns true
  }
}

sub getOSEnv {
  if ($^O eq "linux") {
    $copy_cmd = "cp ";
    $dsn = "dbi:Oracle:$sid";
    #$dsn = "dbi:Oracle:";
  }
  elsif ($^O eq "MSWin32") {
    $copy_cmd = "copy ";
    $dsn = "dbi:Oracle:$sid";
  }
}

sub purgeArch {
  my @arch = @_;

  foreach $arch(@arch) {
    if (-e $arch) {
      writeLog("Purging archive log $arch");
      unlink($arch) || writeLog("ERROR: Unable to purge archive log $arch.  You will need to remove it manually.");
    }
    else {
      writeLog("ERROR: Unable to find $arch.  $!");
    }
  }
}

sub writeLog {
  my($line) = @_;
  my $log = $prog_dir . "\/dbbackup_$sid.log";
  my($ss,$mi,$hr,$dd,$mm,$yy) = (localtime)[0,1,2,3,4,5,6];
  $mm += 1;
  $yy = substr($yy+1900,2);

  open(LOG, ">> $log");
  if (-e $log) {
    printf LOG ("%02d\/%02d\/%02d %02d:%02d:%02d\t%s\n", $mm,$dd,$yy,$hr,$mi,$ss,$line);
    close(LOG);
  }
  else {
    close(LOG);
    printf STDERR ("%02d\/%02d\/%02d %02d:%02d:%02d\t%s\n", $mm,$dd,$yy,$hr,$mi,$ss,"ERROR:  Unable to find $log for writing.");
  }
}

sub writeRestore {
  my($src, $dest) = @_;
  my $restore_file = $bu_dir . "\/restore_$sid.sh";

  open(RESTORE, ">> $restore_file") || writeLog("WARN: Unable to open restore script.  $!");
  if (-e $restore_file) {
    print RESTORE ("#$copy_cmd $src $dest\n"); 
    close(RESTORE);
  }
  else {
    close(RESTORE);
    writeLog("ERROR: Unable to find restore script $restore_file.  $!");
  }
}
