package dbutils;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: DBExporter.java,v 1.4 2008/10/29 22:44:25 bozyurt Exp $
 */

public class DBExporter {
   protected Connection con;
   private String usr, pwd, dbURL;
   protected Map birnIDMap = new HashMap();
   protected Map validSAIDMap = new HashMap();
   protected boolean doFilter = true;
   protected Map siteBidsMap;
   protected String driverClass;
   protected boolean postgresSrc = false;
   protected final boolean SKIP_SEC_LABEL_COLUMN = true;

   public DBExporter(String propertiesFile) throws Exception {
      Properties props = GenUtils.loadProperties(propertiesFile);

      this.usr = props.getProperty("user");
      this.pwd = props.getProperty("pwd");
      this.dbURL = props.getProperty("dburl");
      this.driverClass = props.getProperty("jdbc.driver");
      if (this.driverClass == null)
         this.driverClass = "oracle.jdbc.driver.OracleDriver";
      con = connect(dbURL, usr, pwd);
      System.err.println("connected to " + dbURL);
      if (dbURL.indexOf("postgres") != -1)
         postgresSrc = true;

      String oldNewMapFile = props.getProperty("old_new.map.file"); // "/home/bozyurt/mbirn/public_demo/db/old2new_birnid.txt";
      String siteBidsFile = props.getProperty("site.birnids.file"); // "/home/bozyurt/mbirn/public_demo/srb_data/in/site2_subjects.txt";
      if (oldNewMapFile != null && siteBidsFile != null) {
         Map oldNewMap = loadMap(oldNewMapFile);
         siteBidsMap = loadSiteBirnIDs(oldNewMap, siteBidsFile);

         for (Iterator iter = siteBidsMap.keySet().iterator(); iter.hasNext();) {
            String oldBid = (String) iter.next();
            birnIDMap.put(oldBid, oldBid);
         }

         // populateBirnIDs();

         List l = getValidStoredAssessmentIDs();
         for (Iterator iter = l.iterator(); iter.hasNext();) {
            Integer item = (Integer) iter.next();
            validSAIDMap.put(item, item);

         }
      } else {
         doFilter = false;
      }
   }

   protected void populateBirnIDs() {
      List birndIDs = new ArrayList(20);
      birndIDs.add("000824635528");
      birndIDs.add("000800458596");
      birndIDs.add("000815282938");
      birndIDs.add("000811586953");
      birndIDs.add("000828867468");
      birndIDs.add("000801229536");
      birndIDs.add("000804638825");
      birndIDs.add("000847081634");
      birndIDs.add("000818958791");
      birndIDs.add("000800156805");
      birndIDs.add("000814066654");
      birndIDs.add("000803514716");
      birndIDs.add("000830520046");
      birndIDs.add("000894469074");
      birndIDs.add("000822610505");
      birndIDs.add("000811064132");
      birndIDs.add("000800267855");
      birndIDs.add("000846737005");
      birndIDs.add("000847825956");
      birndIDs.add("000836352007");
      for (Iterator iter = birndIDs.iterator(); iter.hasNext();) {
         String s = (String) iter.next();
         birnIDMap.put(s, s);
      }
   }

   public static Map loadMap(String filename) throws IOException {
      Map map = new HashMap();
      BufferedReader in = null;
      try {
         in = new BufferedReader(new FileReader(filename));

         String line = null;
         while ((line = in.readLine()) != null) {
            StringTokenizer stok = new StringTokenizer(line, ",");
            if (stok.countTokens() == 2) {
               map.put(stok.nextToken(), stok.nextToken());
            }
         }
      } finally {
         if (in != null)
            try {
               in.close();
            } catch (Exception x) {}
      }
      return map;
   }

   protected Map loadSiteBirnIDs(Map oldNewMap, String filename)
         throws IOException {
      // site old BIRN ID - site new Birn ID
      Map siteBidMap = new HashMap();
      Map newOldMap = new HashMap();

      for (Iterator iter = oldNewMap.entrySet().iterator(); iter.hasNext();) {
         Map.Entry entry = (Map.Entry) iter.next();
         newOldMap.put(entry.getValue(), entry.getKey());

      }
      BufferedReader in = null;
      try {
         in = new BufferedReader(new FileReader(filename));

         String line = null;
         while ((line = in.readLine()) != null) {
            String newSiteBid = line.trim();
            String oldSiteBid = (String) newOldMap.get(newSiteBid);
            siteBidMap.put(oldSiteBid, newSiteBid);
         }
      } finally {
         if (in != null)
            try {
               in.close();
            } catch (Exception x) {}
      }

      return siteBidMap;
   }

   protected List getValidStoredAssessmentIDs() throws SQLException {
      List saIds = new LinkedList();
      Statement st = null;
      try {
         StringBuffer buf = new StringBuffer(200);
         buf
               .append("select uniqueid from nc_storedassessment where subjectid in (");
         for (Iterator iter = birnIDMap.values().iterator(); iter.hasNext();) {
            String item = (String) iter.next();
            buf.append("'").append(item).append("'");
            if (iter.hasNext())
               buf.append(",");

         }
         buf.append(")");
         st = con.createStatement();
         ResultSet rs = st.executeQuery(buf.toString());
         while (rs.next()) {
            saIds.add(new Integer(rs.getInt(1)));
         }
         rs.close();
         return saIds;

      } finally {
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }
   }

   public static String readCLOB(oracle.sql.CLOB clob) throws SQLException {
      if (clob == null)
         return null;
      StringBuffer buf = new StringBuffer(128);
      Reader r = clob.characterStreamValue();
      int c;
      try {
         while ((c = r.read()) != -1) {
            buf.append((char) c);
         }
      } catch (IOException iox) {
         throw new SQLException(iox.getMessage());
      }
      return buf.toString();
   }

   public void extractTables(String dbType) throws Exception {
      SchemaExtractor se = null;
      try {
         se = new SchemaExtractor(con);
         if (postgresSrc) {
            se.loadDBTables(null);
         } else {
            se.loadDBTables(usr.toUpperCase());
         }
         Iterator iter = se.getTables();
         while (iter.hasNext()) {
            DBTable table = (DBTable) iter.next();
            // skip non morph-birn tables
            if (!table.getName().toLowerCase().startsWith("nc_")) {
               continue;
            }
            System.out.println("-- processing table " + table.getName());
            System.out.println("-- ");
            // if ( table.getName().equalsIgnoreCase("nc_storedassessment") ) {

            // for Oracle
            if (dbType.equals("oracle")) {
               extractData(table);
            } else if (dbType.equals("postgres")) {
               extractDataForPostgres(table);
            } else {
               throw new RuntimeException("Not a supported database type:"
                     + dbType);
            }

            // }
         }
      } finally {}
   }

   protected Connection connect(String dbURL, String usr, String pwd)
         throws Exception {
      Class.forName(driverClass);
      return DriverManager.getConnection(dbURL, usr, pwd);
   }

   public boolean acceptRow(ResultSet rs, String tableName,
         ResultSetMetaData rsmd) throws SQLException {
      int colCount = rsmd.getColumnCount();
      for (int i = 1; i <= colCount; i++) {
         if (rsmd.getColumnLabel(i).equalsIgnoreCase("subjectid")) {
            return birnIDMap.get(rs.getString(i)) != null;
         }
      }

      if (tableName.equalsIgnoreCase("nc_assessmentdata")) {
         int saID = rs.getInt(14);
         return (validSAIDMap.get(new Integer(saID)) != null);

      } else if (tableName.equalsIgnoreCase("nc_assessmentinteger")) {
         int saID = rs.getInt("storedassessmentid");
         return (validSAIDMap.get(new Integer(saID)) != null);

      } else if (tableName.equalsIgnoreCase("nc_assessmentfloat")) {
         int saID = rs.getInt("storedassessmentid");
         return (validSAIDMap.get(new Integer(saID)) != null);

      } else if (tableName.equalsIgnoreCase("nc_assessmentvarchar")) {
         int saID = rs.getInt("storedassessmentid");
         return (validSAIDMap.get(new Integer(saID)) != null);

      } else if (tableName.equalsIgnoreCase("nc_rawdata")) {
         String dataURI = rs.getString("datauri");
         for (Iterator iter = validSAIDMap.values().iterator(); iter.hasNext();) {
            String item = (String) iter.next();
            if (dataURI.indexOf(item) != -1)
               return true;
         }
         return false;
      }

      return true;

   }

   protected void extractDataForPostgres(DBTable table) throws SQLException {
      StringBuffer buf = new StringBuffer(512);
      buf.append("select * from ").append(table.getName());
      Statement st = null;
      try {

         st = con.createStatement();
         System.err.println(buf.toString());
         ResultSet rs = st.executeQuery(buf.toString());
         ResultSetMetaData rsmd = rs.getMetaData();
         int colCount = rsmd.getColumnCount();
         while (rs.next()) {

            StringBuffer ibuf = new StringBuffer(512);
            ibuf.append("insert into ").append(table.getName()).append(" (");
            for (int i = 1; i <= colCount; ++i) {
               ibuf.append(rsmd.getColumnLabel(i));
               if (i != colCount)
                  ibuf.append(',');
            }
            ibuf.append(") values(");

            for (int i = 1; i <= colCount; ++i) {
               Object colValue = rs.getObject(i);
               if (colValue == null) {
                  if (rsmd.getColumnType(i) == Types.CLOB) {
                     // ibuf.append("EMPTY_CLOB()");
                     ibuf.append("NULL");
                  } else {
                     ibuf.append("NULL");
                  }
               } else if (colValue instanceof oracle.sql.CLOB) {
                  String clob = readCLOB((oracle.sql.CLOB) colValue);
                  ibuf.append(prepPostgresStringLiteral(clob));
               } else {
                  switch (rsmd.getColumnType(i)) {
                  case Types.CHAR:
                  case Types.VARCHAR:
                  case Types.DATE:
                  case Types.TIMESTAMP:
                     ibuf
                           .append(prepPostgresStringLiteral(colValue
                                 .toString()));
                     // ibuf.append("'").append(colValue).append("'");
                     break;
                  case Types.NUMERIC:
                     // int scale = rsmd.getScale(i);
                     // int precision = rsmd.getPrecision(i);
                     // String columnLabel = rsmd.getColumnLabel(i);
                     if (rsmd.getScale(i) == 0 && rsmd.getPrecision(i) == 1) {
                        int colValueInt = Integer.parseInt(colValue.toString());
                        ibuf.append((colValueInt == 1) ? "TRUE" : "FALSE");
                     } else {
                        ibuf.append(colValue);
                     }
                     break;
                  default:
                     if (colValue instanceof oracle.sql.TIMESTAMP) {
                        oracle.sql.TIMESTAMP tstamp = (oracle.sql.TIMESTAMP) colValue;
                        Timestamp tsv = getTimeStampValue(tstamp);
                        // System.err.println("tstamp=" + tsv);
                        if (tsv == null) {
                           ibuf.append("NULL");
                        } else {
                           ibuf.append("'").append(tsv).append("'");
                        }

                     } else {
                        ibuf.append(colValue);
                     }
                  }
               }
               if (i != colCount)
                  ibuf.append(',');
            }

            ibuf.append(");");

            System.out.println(ibuf.toString());
         }
      } finally {
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }

   }

   public static String prepPostgresStringLiteral(String value) {
      StringBuffer buf = new StringBuffer();
      if (value.indexOf(';') != -1 || value.indexOf(',') != -1 || value.indexOf('\'') != -1) {
         buf.append("E'");
         value = value.replaceAll(";", "\\\\;");
         value = value.replaceAll(",", "\\\\,");
         value = value.replaceAll("'", "''");

         buf.append(value);
         buf.append("'");
      } else {
         buf.append('\'').append(value).append('\'');
      }
      return buf.toString();
   }

   protected void extractData(DBTable table) throws SQLException {
      StringBuffer buf = new StringBuffer(512);
      buf.append("select * from ").append(table.getName());
      Statement st = null;
      try {

         st = con.createStatement();
         System.err.println(buf.toString());
         ResultSet rs = st.executeQuery(buf.toString());
         ResultSetMetaData rsmd = rs.getMetaData();
         int colCount = rsmd.getColumnCount();
         while (rs.next()) {

            // in case you do filtering of the input source
            if (doFilter && !acceptRow(rs, table.getName(), rsmd)) {
               continue;
            }
            StringBuffer ibuf = new StringBuffer(512);
            ibuf.append("insert into ").append(table.getName()).append(" (");
            boolean first = true;

            for (int i = 1; i <= colCount; ++i) {
               if (SKIP_SEC_LABEL_COLUMN
                     && rsmd.getColumnLabel(i).equalsIgnoreCase("sec_label")) {
                  continue;
               }
               if (!first)
                  ibuf.append(',');
               if (first)
                  first = false;
               ibuf.append(rsmd.getColumnLabel(i));
            }

            ibuf.append(") values(");

            first = true;
            for (int i = 1; i <= colCount; ++i) {
               if (SKIP_SEC_LABEL_COLUMN
                     && rsmd.getColumnLabel(i).equalsIgnoreCase("sec_label")) {
                  continue;
               }
               if (!first) {
                  ibuf.append(',');
               }
               if (first)
                  first = false;

               Object colValue = rs.getObject(i);
               if (colValue == null) {
                  if (rsmd.getColumnType(i) == Types.CLOB) {
                     ibuf.append("EMPTY_CLOB()");
                  } else {
                     ibuf.append("NULL");
                  }
               } else if (colValue instanceof oracle.sql.CLOB) {
                  ibuf.append("'").append(readCLOB((oracle.sql.CLOB) colValue))
                        .append("'");
               } else {
                  switch (rsmd.getColumnType(i)) {
                  case Types.CHAR:
                  case Types.VARCHAR:
                  case Types.DATE:
                  case Types.TIMESTAMP:
                     if (doFilter
                           && rsmd.getColumnLabel(i).equalsIgnoreCase(
                                 "subjectid")) {
                        System.err.println("mapping " + colValue + " to "
                              + siteBidsMap.get(colValue));
                        colValue = this.siteBidsMap.get(colValue);
                     }

                     // oracle does not like date values with millisecs
                     if (rsmd.getColumnType(i) == Types.DATE
                           || rsmd.getColumnType(i) == Types.TIMESTAMP) {
                        ibuf.append("TIMESTAMP ");
                     }
                     ibuf.append("'").append(colValue).append("'");
                     break;
                  default:
                     if (colValue instanceof oracle.sql.TIMESTAMP) {
                        oracle.sql.TIMESTAMP tstamp = (oracle.sql.TIMESTAMP) colValue;
                        Timestamp tsv = getTimeStampValue(tstamp);
                        // System.err.println("tstamp(2)=" + tsv);
                        if (tsv == null) {
                           ibuf.append("NULL");
                        } else {
                           ibuf.append(" TIMESTAMP '").append(tsv).append("'");
                        }
                     } else {
                        ibuf.append(colValue);
                     }
                  }
               }
            }

            ibuf.append(");");

            System.out.println(ibuf.toString());
         }
      } finally {
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }
   }

   public static Timestamp getTimeStampValue(oracle.sql.TIMESTAMP tstamp)
         throws SQLException {
      if (tstamp == null)
         return null;
      try {
         return tstamp.timestampValue();
      } catch (NullPointerException npe) {
         return null;
      }
   }

   public void shutdown() {
      if (con != null)
         try {
            con.close();
         } catch (Exception x) {}
   }

   public static void main(String[] args) {
      String dbType = "oracle";
      if (args.length == 1 && args[0].equalsIgnoreCase("postgres")) {
         dbType = "postgres";
      }

      DBExporter dbe = null;
      try {
         dbe = new DBExporter("dbexporter.properties");
         dbe.extractTables(dbType);
      } catch (Exception ex) {
         ex.printStackTrace();
      } finally {
         if (dbe != null)
            dbe.shutdown();
      }

   }
}
