package dbutils;

import java.io.Writer;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DBTable.java,v 1.2 2008/10/29 22:44:25 bozyurt Exp $
 */

public class DBTable {
   private String name;
   private List columns;
   private boolean sorted = false;
   private final static SimpleDateFormat df = new SimpleDateFormat("dd-MMM-yy");

   public DBTable(Connection con, String tableName, String schema)
         throws SQLException {
      columns = new LinkedList();
      extractTableInfo(con, tableName, schema);
      name = tableName;

   }

   public String getName() {
      return name;
   }

   public static class DBColumn {
      String name;
      String type;
      int width;
      int fracDigit;
      boolean notNullable;
      int order;
      boolean primaryKey = false;
      int primaryKeyOrder;

      public DBColumn(String name, String type, int width, int fracDigit,
            int order, boolean notNullable) {
         this.name = name;
         this.type = type;
         this.width = width;
         this.fracDigit = fracDigit;
         this.order = order;
         this.notNullable = notNullable;
      }

      // ---------------------- getters --------------
      public String getName() {
         return this.name;
      }

      public String getType() {
         return this.type;
      }

      public int getWidth() {
         return this.width;
      }

      public int getFracDigit() {
         return this.fracDigit;
      }

      public boolean getNotNullable() {
         return this.notNullable;
      }

      public int getOrder() {
         return this.order;
      }

      public void setPrimaryKey(boolean value) {
         this.primaryKey = value;
      }

      public void setPrimaryKeyOrder(int value) {
         this.primaryKeyOrder = value;
      }

      public String toString() {
         StringBuffer buf = new StringBuffer(128);
         buf.append("[name=").append(name).append(",type=").append(type);
         buf.append(",width=").append(width).append(",frac-dig=").append(
               this.fracDigit).append(",order=").append(order);
         buf.append(",notNullable=").append(this.notNullable).append("]");
         return buf.toString();
      }

      public String toDDL() {
         StringBuffer buf = new StringBuffer(128);
         buf.append(name).append("   ").append(type);
         // geared towards Oracle
         if (width > 0 && !type.equals("DATE")) {
            buf.append("(").append(width);
            if (this.fracDigit > 0)
               buf.append(",").append(fracDigit);
            buf.append(")  ");
         } else
            buf.append("  ");
         // if (primaryKey) buf.append(" PRIMARY KEY ");
         if (notNullable)
            buf.append("NOT NULL");

         return buf.toString();
      }
   }

   public void extractTableInfo(Connection con, String tableName, String schema)
         throws SQLException {
      DatabaseMetaData md = con.getMetaData();
      ResultSet rs = md.getColumns("", schema, tableName, "%");
      while (rs.next()) {
         String nullYN = rs.getString(18);

         DBColumn col = new DBColumn(rs.getString(4),/* col name */
         rs.getString(6), /* type name */
         rs.getInt(7), /* col size */
         rs.getInt(9), /* frac digits (if any) */
         rs.getInt(17), /* ordinal position of column */
         nullYN.equals("NO"));
         this.columns.add(col);
      }
      rs.close();
      getPrimaryKeys(con, tableName, schema);
   }

   public String[] getColumnNames() {
      String[] colNames = new String[columns.size()];
      Iterator it = columns.iterator();
      int count = 0;
      while (it.hasNext()) {
         DBColumn col = (DBColumn) it.next();
         colNames[count++] = col.name;
      }
      return colNames;
   }

   private void getPrimaryKeys(Connection con, String tableName, String schema)
         throws SQLException {
      DatabaseMetaData md = con.getMetaData();
      ResultSet rs = md.getPrimaryKeys("", schema, tableName);
      while (rs.next()) {
         String pkName = rs.getString(4);
         // System.out.println(">> "+ rs.getString(4)/* col name */+","+
         // rs.getInt(5)/* sequence no within primary key*/);
         Iterator it = columns.iterator();
         while (it.hasNext()) {
            DBColumn col = (DBColumn) it.next();
            if (col.name.equals(pkName)) {
               col.primaryKey = true;
               col.primaryKeyOrder = rs.getInt(5);
               break;
            }
         }
      }
      rs.close();
   }

   public String getPrimaryKeyDDL() {
      StringBuffer buf = new StringBuffer(128);
      buf.append("ALTER TABLE ").append(name).append(" ADD (PRIMARY KEY (");
      int[] idxs = getPrimaryKeyIndices();
      if (idxs == null)
         return null;
      for (int i = 0; i < idxs.length; ++i) {
         DBColumn col = (DBColumn) columns.get(idxs[i]);
         buf.append(col.name);
         if (i < (idxs.length - 1))
            buf.append(",");
      }
      buf.append(") );\n");
      return buf.toString();

   }

   public List getPrimaryKeyColumns() {
      int[] idxs = getPrimaryKeyIndices();
      if (idxs == null)
         return null;
      List pkCols = new LinkedList();
      for (int i = 0; i < idxs.length; ++i) {
         DBColumn col = (DBColumn) columns.get(idxs[i]);
         pkCols.add(col);
      }
      return pkCols;
   }

   private int[] getPrimaryKeyIndices() {
      List idxs = new ArrayList(10);
      Iterator it = columns.iterator();
      while (it.hasNext()) {
         DBColumn col = (DBColumn) it.next();
         if (col.primaryKey) {
            idxs.add(new Integer(col.primaryKeyOrder - 1));
            // System.out.println(">> "+ (col.primaryKeyOrder-1) );
         }
      }
      if (idxs.isEmpty())
         return null;
      int[] is = new int[idxs.size()];
      it = idxs.iterator();
      int i = 0;
      while (it.hasNext()) {
         is[i++] = ((Integer) it.next()).intValue();
      }
      return is;
   }

   public List getColumns() {
      return columns;
   }

   public void extractData(Connection con, int noRows, Writer w) {
      ResultSet rs = null;
      Statement st = null;
      try {
         st = con.createStatement();
         rs = st.executeQuery("SELECT * from " + name);
         int count = 0;
         while (rs.next()) {
            ++count;
            if (noRows > 0 && count > noRows)
               break;
            ResultSetMetaData rsmd = rs.getMetaData();
            int colCount = rsmd.getColumnCount();
            StringBuffer buf = new StringBuffer(256);
            buf.append("INSERT INTO ").append(name).append(" VALUES(");
            for (int i = 0; i < colCount; ++i) {
               Object o = rs.getObject(i + 1);
               if (o == null) {
                  buf.append("NULL");
               } else {
                  if (o instanceof java.lang.String)
                     buf.append("'").append(escapeQuotes(o.toString())).append(
                           "'");
                  else if (o instanceof java.sql.Timestamp) {
                     Timestamp ts = (Timestamp) o;
                     buf.append('\'').append(
                           df.format(new java.util.Date(ts.getTime()))).append(
                           '\'');
                  } else {
                     // System.out.println(">> type="+ o.getClass().getName() );
                     buf.append('\'').append(o.toString()).append('\'');
                  }
               }
               if (i < (colCount - 1))
                  buf.append(",");
            }
            buf.append(");\n");
            w.write(buf.toString());
            w.flush();
         }

      } catch (Exception e) {
         e.printStackTrace();
      } finally {
         if (rs != null)
            try {
               rs.close();
            } catch (Exception x) {}
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }
   }

   public int getNoRows(Connection con) {
      ResultSet rs = null;
      Statement st = null;
      int rowCount = 0;
      try {
         st = con.createStatement();
         rs = st.executeQuery("SELECT COUNT(*) from " + name);
         if (rs.next())
            rowCount = rs.getInt(1);
      } catch (Exception e) {
         e.printStackTrace();
      } finally {
         if (rs != null)
            try {
               rs.close();
            } catch (Exception x) {}
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }
      return rowCount;
   }

   public static String escapeQuotes(String s) {
      StringBuffer buf = new StringBuffer(s.length() + 1);
      char[] c = s.toCharArray();
      for (int i = 0; i < c.length; ++i) {
         if (c[i] == '\'')
            buf.append("''");
         else
            buf.append(c[i]);
      }
      return buf.toString();
   }

   public void show() {
      StringBuffer buf = new StringBuffer(512);
      buf.append("TABLE=").append(name).append("\n\t");
      Iterator it = columns.iterator();
      while (it.hasNext()) {
         DBColumn col = (DBColumn) it.next();
         // buf.append( col.toString() );
         buf.append(col.toDDL());
         buf.append("\n\t");
      }
      buf.append("\n");
      System.out.println(buf.toString());
   }

   public void prepare() {
      sortColumns();
   }

   public void showDDL() {
      StringBuffer buf = new StringBuffer(512);
      buf.append("CREATE TABLE ").append(name).append(" (\n\t");
      sortColumns();
      /*
       * // flatten columns list and sort by the database column order Object[]
       * colArr = new Object[ columns.size() ]; Iterator it =
       * columns.iterator(); int i=0; while(it.hasNext()) { colArr[i++] =
       * it.next(); } Arrays.sort(colArr, new ColumnComparator()); for(i=0; i <
       * colArr.length; ++i) { buf.append( ((DBColumn)colArr[i]).toDDL() ); if (
       * i < (colArr.length-1)) buf.append(",\n\t"); }
       */
      Iterator it = columns.iterator();
      while (it.hasNext()) {
         DBColumn col = (DBColumn) it.next();
         buf.append(col.toDDL());
         if (it.hasNext())
            buf.append(",\n\t");
      }
      buf.append(");\n");
      System.out.println(buf.toString());
   }

   // sorts table column to be the same order as defined in DB
   private void sortColumns() {
      if (this.sorted)
         return;
      // flatten columns list and sort by the database column order
      Object[] colArr = new Object[columns.size()];
      Iterator it = columns.iterator();
      int i = 0;
      while (it.hasNext()) {
         colArr[i++] = it.next();
      }
      Arrays.sort(colArr, new ColumnComparator());
      columns.clear();
      for (i = 0; i < colArr.length; ++i) {
         columns.add(colArr[i]);
      }
      sorted = true;
   }

   private class ColumnComparator implements Comparator {
      public int compare(Object o1, Object o2) {
         return ((DBColumn) o1).order - ((DBColumn) o2).order;
      }
   }

}// ;
