package dbutils;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: SchemaDiff.java,v 1.2 2008/10/29 22:44:25 bozyurt Exp $
 */
public class SchemaDiff {
   protected SchemaExtractor se;
   protected SchemaParser parser;
   protected Properties props;

   public SchemaDiff() throws IOException {
      props = GenUtils.loadProperties("schemadiff.properties");
   }

   public void findDiff() throws Exception {
      String dbURL = props.getProperty("dburl");
      String user = props.getProperty("user");
      String pwd = props.getProperty("pwd");
      se = new SchemaExtractor(dbURL, user, pwd);

      se.loadDBTables(user.toUpperCase());
      System.out.println("loaded db tables for database " + dbURL);
      se.loadConstraints();
      Map constraintMap = se.getConstraintMap();
      System.out.println("constraintMap.size=" + constraintMap.size());

      String tableCreateFile = props.getProperty("schema2.table_create_file");
      parser = new SchemaParser(tableCreateFile);
      parser.parse();
      Map tableMap = parser.getTableMap();

      String schema2Name = props.getProperty("schema2.name");
      System.out.println("\n\nDIFFERENCES\n-------------\n");
      for (Iterator iter = se.getTables(); iter.hasNext();) {
         DBTable dbTable = (DBTable) iter.next();
         TableInfo ti = (TableInfo) tableMap.get(dbTable.getName());
         // System.out.println("table "+ dbTable.getName());
         // System.out.println("");
         if (ti == null) {
            System.out.println("Table " + dbTable.getName()
                  + " is not in schema " + schema2Name);
         } else {
            checkTableConstraints(constraintMap, ti, schema2Name);
            findColumnDiffs(dbTable, ti, schema2Name);
         }
      }
   }

   protected void checkTableConstraints(Map constraintMap, TableInfo ti,
         String schema2Name) throws Exception {
      for (Iterator iter = ti.getConstraints().iterator(); iter.hasNext();) {
         // System.out.println(">> "+ ti.name + " has constraints");
         ConstraintInfo ci = (ConstraintInfo) iter.next();
         if (ci instanceof UniqueConstraintInfo) {
            DBConstraint dbCon = (DBConstraint) constraintMap.get(ci.getName());
            if (dbCon == null) {
               System.out.println("Unique constraint " + ci.getName()
                     + "for table " + ti.name + " for schema " + schema2Name
                     + " was not in the other schema");
            } else {
               checkConstraintColumnDiffs(dbCon, (UniqueConstraintInfo) ci,
                     schema2Name);

            }
         }

      }
   }

   protected void checkConstraintColumnDiffs(DBConstraint dbCon,
         UniqueConstraintInfo uci, String schema2Name) throws Exception {
      Iterator iter2 = uci.getColNames().iterator();
      for (Iterator iter = dbCon.columnNames.iterator(); iter.hasNext();) {
         String colName = (String) iter.next();
         if (iter2.hasNext()) {
            String schema2ColName = (String) iter2.next();
            if (!schema2ColName.equalsIgnoreCase(colName)) {
               System.out.println("Column name " + schema2ColName
                     + " from the constraint " + dbCon.getName()
                     + " for table " + dbCon.getTableName() + " for schema "
                     + schema2Name + " does not match the column name "
                     + colName);
            }
         } else {
            System.out.println("Column name " + colName
                  + " is missing from the constraint " + dbCon.getName()
                  + " for table " + dbCon.getTableName() + " for schema "
                  + schema2Name);
         }
      }
      if (iter2.hasNext()) {
         System.out.println("Constraint " + dbCon.getName() + " for table "
               + dbCon.getTableName() + " for schema " + schema2Name
               + " has more columns than the other schema");
      }
   }

   /**
    * diffing two live database schemas
    *
    * @param dbCon1
    * @param dbCon2
    * @param schema2Name
    * @throws java.lang.Exception
    */
   protected void checkConstraintColumnDiffs(DBConstraint dbCon1,
         DBConstraint dbCon2, String schema2Name) throws Exception {

      Iterator iter2 = dbCon2.columnNames.iterator();
      for (Iterator iter = dbCon1.columnNames.iterator(); iter.hasNext();) {
         String colName = (String) iter.next();
         if (iter2.hasNext()) {
            String schema2ColName = (String) iter2.next();
            if (!schema2ColName.equalsIgnoreCase(colName)) {
               System.out.println("Column name " + schema2ColName
                     + " from the constraint " + dbCon1.getName()
                     + " for table " + dbCon1.getTableName() + " for schema "
                     + schema2Name + " does not match the column name "
                     + colName);
            }
         } else {
            System.out.println("Column name " + colName
                  + " is missing from the constraint " + dbCon1.getName()
                  + " for table " + dbCon1.getTableName() + " for schema "
                  + schema2Name);
         }
      }
      if (iter2.hasNext()) {
         System.out.println("Constraint " + dbCon1.getName() + " for table "
               + dbCon1.getTableName() + " for schema " + schema2Name
               + " has more columns than the other schema");
      }
   }

   protected void findColumnDiffs(DBTable dbTable, TableInfo ti,
         String schema2Name) throws Exception {
      List dbColumns = dbTable.getColumns();
      List columnInfos = ti.getColumns();
      Map dbColumnMap = new LinkedHashMap();
      for (Iterator iter = dbColumns.iterator(); iter.hasNext();) {
         DBTable.DBColumn dbColumn = (DBTable.DBColumn) iter.next();
         dbColumnMap.put(dbColumn.getName(), dbColumn);
      }

      for (Iterator iter = columnInfos.iterator(); iter.hasNext();) {
         ColumnInfo ci = (ColumnInfo) iter.next();
         DBTable.DBColumn dbCol = (DBTable.DBColumn) dbColumnMap.get(ci
               .getName());
         if (dbCol == null) {
            System.out.println("Schema " + schema2Name + "has the column "
                  + ci.getName() + " in table " + dbTable.getName()
                  + " which is missing from the other schema");
         } else {
            if (!dbCol.getType().equalsIgnoreCase(ci.getDataType())) {
               System.out.println("column type for " + dbTable.getName() + "."
                     + ci.getName() + " is " + ci.getDataType()
                     + " for schema " + schema2Name + " was " + dbCol.getType()
                     + " for the other schema");
            }
         }
      }

   }

   public static void main(String[] args) {
      try {
         SchemaDiff sd = new SchemaDiff();
         sd.findDiff();
      } catch (Exception x) {
         x.printStackTrace();
      }
   }
}