package dbutils;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: SchemaExtractor.java,v 1.3 2008/10/29 22:44:25 bozyurt Exp $
 */
public class SchemaExtractor {
   private Connection con = null;
   private Map tableMap = new HashMap();
   private Map consMap;
   private static Map sql2javaMap = new HashMap();

   static {
      sql2javaMap.put("varchar2", "String");
      sql2javaMap.put("date", "java.util.Date");
      sql2javaMap.put("clob", "String");
      sql2javaMap.put("float", "Double");

   }

   public SchemaExtractor(String dbURL, String usr, String pwd)
         throws Exception {
      this.con = connect(dbURL, usr, pwd);
   }

   public SchemaExtractor(Connection con) throws Exception {
      this.con = con;

   }

   protected Connection connect(String dbURL, String usr, String pwd)
         throws Exception {
      Class.forName("oracle.jdbc.driver.OracleDriver");
      return DriverManager.getConnection(dbURL, usr, pwd);
   }

   public void shutdown() {
      if (con != null)
         try {
            con.close();
         } catch (Exception x) { /* ignore */}
   }

   public void loadDBTables(String schema) throws SQLException {
      loadDBTables(schema, false);
   }

   public void loadDBTables(String schema, boolean loadViewsAlso)
         throws SQLException {
      DatabaseMetaData md = con.getMetaData();
      ResultSet rs = null;
      String[] tableTypes = { "TABLE" };
      if (loadViewsAlso) {
         tableTypes = new String[2];
         tableTypes[0] = "TABLE";
         tableTypes[1] = "VIEW";
      }
      // String[] tableTypes={"VIEW"}; // do also for views similar to table
      rs = md.getTables("", schema, "%", tableTypes);
      // System.out.println("rs="+rs);

      while (rs.next()) {
         String tableName = rs.getString(3);
         DBTable table = new DBTable(con, tableName, schema);
         tableMap.put(tableName, table);
         table.prepare();
      }
   }

   public Map getConstraintMap() {
      return consMap;
   }

   public void loadConstraints() throws SQLException {
      consMap = loadConstraintNames("unique");
      for (Iterator iter = consMap.values().iterator(); iter.hasNext();) {
         DBConstraint dbCon = (DBConstraint) iter.next();
         setConstraintColumns(dbCon);
      }
   }

   protected void setConstraintColumns(DBConstraint dbCon) throws SQLException {
      Statement st = null;
      try {
         st = con.createStatement();
         ResultSet rs = st
               .executeQuery("select column_name from user_cons_columns where constraint_name='"
                     + dbCon.getName()
                     + "' and table_name='"
                     + dbCon.getTableName() + "' order by position");
         while (rs.next()) {
            String colName = rs.getString(1);
            dbCon.addColumnName(colName);
         }
         rs.close();
      } finally {
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }

   }

   protected Map loadConstraintNames(String constraintType) throws SQLException {
      Map consMap = new LinkedHashMap();
      Statement st = null;
      String dbConsType = "U";
      try {
         st = con.createStatement();
         ResultSet rs = st
               .executeQuery("select constraint_name, table_name from user_constraints where constraint_type = '"
                     + dbConsType + "'");
         while (rs.next()) {
            String constraintName = rs.getString(1);
            String tableName = rs.getString(2);
            if (tableMap.get(tableName) != null) {
               DBConstraint cons = new DBConstraint(constraintName, tableName);
               consMap.put(constraintName, cons);
            }
         }
      } finally {
         if (st != null)
            try {
               st.close();
            } catch (Exception x) {}
      }
      return consMap;
   }

   public Map getForeignKeys(Connection con, String schema, String tableName)
         throws SQLException {
      DatabaseMetaData md = con.getMetaData();
      ResultSet rs = md.getExportedKeys(null, schema, tableName);
      Map fkMap = new LinkedHashMap();
      while (rs.next()) {
         // System.out.println("pk column="+ rs.getObject(4));
         // System.out.println("fk table="+ rs.getObject(7));
         // System.out.println("fk column="+ rs.getObject(7));

         String pkColumn = rs.getObject(4).toString();
         String pkTable = rs.getObject(3).toString();
         String fkTable = rs.getObject(7).toString();
         String fkColumn = rs.getObject(8).toString();
         int deleteRule = rs.getShort(11);
         String pkName = rs.getString(13);
         String fkName = rs.getString(12);
         int keySeq = rs.getShort(9);

         ForeignKey fk = (ForeignKey) fkMap.get(pkTable + fkTable);
         if (fk == null) {
            fk = new ForeignKey(fkName, pkName, pkTable, fkTable, 10,
                  deleteRule);
            fkMap.put(pkTable + fkTable, fk);
         }
         fk.addFkColumn(fkColumn, keySeq - 1);
         fk.addPkColumn(pkColumn, keySeq - 1);

      }
      System.out.println("imported keys");
      rs = md.getImportedKeys(null, schema, tableName);

      while (rs.next()) {
         String pkColumn = rs.getObject(4).toString();
         String pkTable = rs.getObject(3).toString();
         String fkTable = rs.getObject(7).toString();
         String fkColumn = rs.getObject(8).toString();
         int deleteRule = rs.getShort(11);
         String pkName = rs.getString(13);
         String fkName = rs.getString(12);
         int keySeq = rs.getShort(9);

         ForeignKey fk = (ForeignKey) fkMap.get(pkTable + fkTable);
         if (fk == null) {
            fk = new ForeignKey(fkName, pkName, pkTable, fkTable, 10,
                  deleteRule);
            fkMap.put(pkTable + fkTable, fk);
         }
         fk.addFkColumn(fkColumn, keySeq - 1);
         fk.addPkColumn(pkColumn, keySeq - 1);

      }
      for (Iterator iter = fkMap.values().iterator(); iter.hasNext();) {
         ForeignKey fk = (ForeignKey) iter.next();
         System.out.println(fk.toForeignKey());

      }
      return fkMap;
   }

   public DBTable getTable(String tableName) {
      return (DBTable) tableMap.get(tableName);
   }

   public Iterator getTables() {
      return tableMap.values().iterator();
   }

   public static String getConversionType(DBTable.DBColumn column) {
      if (column.getType().equalsIgnoreCase("number") && column.getWidth() == 1) {
         return "bigdecimal2bool";
      } else if (column.getType().equalsIgnoreCase("char")
            && column.getWidth() == 1) {
         return "string2bool";
      } else if (column.getType().equalsIgnoreCase("clob")) {
         return "clob2string";
      }
      return "";
   }

   public static String toJavaType(DBTable.DBColumn column) {
      String javaType = (String) sql2javaMap
            .get(column.getType().toLowerCase());
      if (javaType != null)
         return javaType;

      if (column.getType().toLowerCase().startsWith("timestamp")) {
         return "java.sql.Timestamp";
      }

      if (column.getType().toLowerCase().startsWith("date")) {
         return "java.util.Date";
      }

      if (column.getType().equalsIgnoreCase("number")) {
         if (column.getFracDigit() > 0) {
            return "Double";
         }
         if (column.getWidth() == 1) {
            return "Boolean";
         }
         return "java.math.BigDecimal";

      } else if (column.getType().equalsIgnoreCase("char")) {
         return column.getWidth() == 1 ? "Boolean" : "String";
      } else
         throw new RuntimeException("Not a supported SQL type: "
               + column.getType() + ", " + column.getName());
   }

   public static void testTimeStamp() throws SQLException {
      oracle.sql.TIMESTAMP ots = new oracle.sql.TIMESTAMP(new java.sql.Date(
            System.currentTimeMillis()));
      Timestamp jts = ots.timestampValue();
      System.out.println("java timestamp " + jts);
   }

   public static void main(String[] args) {
      SchemaExtractor se = null;
      try {
         se = new SchemaExtractor("jdbc:oracle:thin:@fmri-gpop:1521:orcl1",
               "ucsd_fmri", "");
      } catch (Exception x) {
         x.printStackTrace();
      } finally {
         if (se != null)
            se.shutdown();
      }
   }
}
