package dbutils;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: SchemaParser.java,v 1.3 2008/11/06 00:38:05 bozyurt Exp $
 */
public class SchemaParser {
   protected DMLTokenizer tokenizer;
   protected Map tableMap = new LinkedHashMap();
   protected Map pkMap = new LinkedHashMap();
   protected Map fkMap = new LinkedHashMap();

   public SchemaParser(String filename) throws IOException {
      tokenizer = new DMLTokenizer(filename);
   }

   public Map getTableMap() {
      return tableMap;
   }

   public Map getPkMap() {
      return pkMap;
   }

   public Map getFkMap() {
      return fkMap;
   }

   public void parse() throws IOException, ParseException {
      int tc = -1;
      try {
         while ((tc = tokenizer.getNextToken()) != -1 && tc != DMLTokenizer.EOF) {
            switch (tc) {
            case DMLTokenizer.WORD:
               // System.out.println("word: " + tokenizer.getStringValue());
               break;
            case DMLTokenizer.STRING:
               // System.out.println("string: " + tokenizer.getStringValue());
               break;
            case DMLTokenizer.NUMBER:
               // System.out.println("number: " + tokenizer.getNumber());
               break;
            default:
               String keyword = tokenizer.getKeyword(tc);
               if (keyword != null) {
                  System.out.println("keyword: " + keyword);
                  switch (tc) {
                  case DMLTokenizer.ALTER:
                     parseConstraint();
                     break;
                  case DMLTokenizer.CREATE:
                     parseCreateStatement();
                  }
               }

            }
         }
      } finally {
         if (tokenizer != null)
            tokenizer.shutdown();
      }
   }

   protected String getLiteral() throws IOException, ParseException {
      int tc = -1;
      if ((tc = tokenizer.getNextToken()) != -1 && tc != DMLTokenizer.EOF) {
         if (tc == DMLTokenizer.WORD) {
            return tokenizer.getStringValue();
         } else
            throw new ParseException("Expected a literal!");
      } else
         throw new ParseException("Expected a literal!");
   }

   protected int getInteger() throws IOException, ParseException {
      int tc = -1;
      if ((tc = tokenizer.getNextToken()) != -1 && tc != DMLTokenizer.EOF) {
         if (tc == DMLTokenizer.NUMBER) {
            return tokenizer.getNumber().intValue();
         } else
            throw new ParseException("Expected an integer!");
      } else
         throw new ParseException("Expected an integer!");

   }

   protected void checkKeyword(int actualTok, int expectedTok,
         String expectedKeyword) throws ParseException {
      if (actualTok != expectedTok)
         throw new ParseException(expectedKeyword + " was expected!");
   }

   void parseConstraint() throws IOException, ParseException {
      int tc = tokenizer.getNextToken();
      tokenizer.getKeyword(tc);
      if (tc == DMLTokenizer.TABLE) {
         String tableName = getLiteral();
         tc = tokenizer.getNextToken();
         checkKeyword(tc, DMLTokenizer.ADD, "ADD");
         tc = tokenizer.getNextToken();
         checkKeyword(tc, DMLTokenizer.CONSTRAINT, "CONSTRAINT");
         String constraintName = getLiteral();
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.FOREIGN) {
            ForeignKey fk = new ForeignKey(constraintName, tableName);
            fkMap.put(constraintName, fk);
            parseForeignKey(fk);
         } else if (tc == DMLTokenizer.PRIMARY) {
            PrimaryKey pk = new PrimaryKey(constraintName, tableName);
            pkMap.put(constraintName, pk);
            parsePrimaryKey(pk);

         }
      }
   }

   void parsePrimaryKey(PrimaryKey pk) throws IOException, ParseException {
      System.out.println("parsing primary key " + pk.getName());
      int tc = tokenizer.getNextToken();
      checkKeyword(tc, DMLTokenizer.KEY, "KEY");
      tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.LEFT_PAR) {
         do {
            String colName = getLiteral();
            pk.addColumnName(colName);
            tc = tokenizer.getNextToken();
         } while (tc != DMLTokenizer.RIGHT_PAR);
      }
   }

   void parseForeignKey(ForeignKey fk) throws IOException, ParseException {
      int tc = tokenizer.getNextToken();
      checkKeyword(tc, DMLTokenizer.KEY, "KEY");
      tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.LEFT_PAR) {
         int idx = 0;
         do {
            String colName = getLiteral();
            fk.addFkColumn(colName, idx);
            tc = tokenizer.getNextToken();
            ++idx;
         } while (tc != DMLTokenizer.RIGHT_PAR);

         tc = tokenizer.getNextToken();
         checkKeyword(tc, DMLTokenizer.REFERENCES, "REFERENCES");
         String pkTable = getLiteral();
         fk.setPkTable(pkTable);

         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.LEFT_PAR) {
            idx = 0;
            do {
               String colName = getLiteral();
               fk.addPkColumn(colName, idx);
               tc = tokenizer.getNextToken();
               ++idx;
            } while (tc != DMLTokenizer.RIGHT_PAR);
         }
      }
      tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.ON) {
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.DELETE) {
            /** @todo */
            // fk.setDeleteRule();
         }
      }
   }

   void parseCreateStatement() throws IOException, ParseException {
      int tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.TABLE) {
         parseCreateTable();
      }
   }

   void parseCreateTable() throws IOException, ParseException {
      String tableName = getLiteral().toUpperCase();

      TableInfo tableInfo = new TableInfo(tableName);
      tableMap.put(tableName, tableInfo);

      int tc = tokenizer.getNextToken(); // (

      do {
         tc = parseTableColumn(tableInfo);
      } while (tc != DMLTokenizer.RIGHT_PAR);

   }

   int parseTableColumn(TableInfo tableInfo) throws IOException, ParseException {
      int tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.CONSTRAINT) {
         String consName = getLiteral().toUpperCase();
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.UNIQUE) {
            UniqueConstraintInfo uci = new UniqueConstraintInfo(consName);

            tc = parseUniqueConstraint(uci);
            tableInfo.addConstraint(uci);
            return tc;
         }
      }
      String columnName = tokenizer.getStringValue().toUpperCase();
      ColumnInfo ci = new ColumnInfo(columnName);
      tc = tokenizer.getNextToken();
      switch (tc) {
      case DMLTokenizer.NUMBER_TOK:
         ci.setDataType(tokenizer.getKeyword(tc));
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.LEFT_PAR) {
            ci.setPrecision(getInteger());
            tc = tokenizer.getNextToken();
            if (tc == DMLTokenizer.COMMA) {
               ci.setScale(getInteger());
               tc = tokenizer.getNextToken(); // )
            }
         }

         break;
      case DMLTokenizer.VARCHAR2:
         ci.setDataType(tokenizer.getKeyword(tc));
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.LEFT_PAR) {
            ci.setPrecision(getInteger());
            tc = tokenizer.getNextToken(); // )
         }
         break;
      case DMLTokenizer.DATE:
         ci.setDataType(tokenizer.getKeyword(tc));
         break;
      case DMLTokenizer.TIMESTAMP:
         ci.setDataType(tokenizer.getKeyword(tc));
         break;
      case DMLTokenizer.CLOB:
         ci.setDataType(tokenizer.getKeyword(tc));
         break;
      case DMLTokenizer.DOUBLE:
         ci.setDataType(tokenizer.getKeyword(tc));
         tc = tokenizer.getNextToken();
         checkKeyword(tc, DMLTokenizer.PRECISION, "PRECISION");
         break;
      default:
         throw new ParseException("Unknown data type: "
               + tokenizer.getStringValue());

      }
      ci.setNullable(true);
      do {

         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.NOT) {
            tc = tokenizer.getNextToken();
            if (tc == DMLTokenizer.NULL_TOK) {
               ci.setNullable(false);
            }
         } else if (tc == DMLTokenizer.CONSTRAINT) {
            String consName = getLiteral();
            tc = tokenizer.getNextToken();
            if (tc == DMLTokenizer.UNIQUE) {
               UniqueConstraintInfo uci = new UniqueConstraintInfo(consName);

               tc = parseUniqueConstraint(uci);
               tableInfo.addConstraint(uci);
            } else if (tc == DMLTokenizer.CHECK) {
               CheckConstaintInfo cci = new CheckConstaintInfo(consName);

               tc = parseCheckConstraint(cci);
               tableInfo.addConstraint(cci);
            } else {
               throw new ParseException("Unsupported constraint type:"
                     + tokenizer.getStringValue());
            }

         }
      } while (tc != DMLTokenizer.COMMA && tc != DMLTokenizer.RIGHT_PAR);

      tableInfo.addColumnInfo(ci);
      return tc;
   }

   int parseUniqueConstraint(UniqueConstraintInfo uci) throws IOException,
         ParseException {
      int tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.LEFT_PAR) {
         do {
            String colName = getLiteral();
            uci.addColName(colName);
            tc = tokenizer.getNextToken();

         } while (tc != DMLTokenizer.RIGHT_PAR);
         return tokenizer.getNextToken();
      }
      return tc;
   }

   int parseCheckConstraint(CheckConstaintInfo cci) throws IOException,
         ParseException {
      int tc = tokenizer.getNextToken();
      if (tc == DMLTokenizer.LEFT_PAR) {
         getLiteral();
         tc = tokenizer.getNextToken();
         checkKeyword(tc, DMLTokenizer.IN, "IN");
         tc = tokenizer.getNextToken();
         if (tc == DMLTokenizer.LEFT_PAR) {
            do {
               tc = tokenizer.getNextToken();
            } while (tc != DMLTokenizer.RIGHT_PAR);
            tc = tokenizer.getNextToken(); // for outer )
         }
         return tokenizer.getNextToken();
      }
      return tc;
   }

   public static void main(String[] args) {
      SchemaParser parser = null;
      try {
         parser = new SchemaParser("/home/bozyurt/tmp/morph_fk_create.sql");
         parser.parse();

         if (!parser.tableMap.isEmpty()) {
            System.out.println("\nNumber of tables=" + parser.tableMap.size());
            for (Iterator iter = parser.tableMap.values().iterator(); iter
                  .hasNext();) {
               TableInfo ti = (TableInfo) iter.next();
               System.out.println(ti.toString());

            }
         }

         if (!parser.pkMap.isEmpty()) {
            System.out.println("\nNumber of primary keys="
                  + parser.pkMap.size());
            for (Iterator iter = parser.pkMap.values().iterator(); iter
                  .hasNext();) {
               PrimaryKey pk = (PrimaryKey) iter.next();
               System.out.println(pk.toString());
            }
         }

         if (!parser.fkMap.isEmpty()) {
            System.out.println("\nNumber of foreign keys="
                  + parser.fkMap.size());
            for (Iterator iter = parser.fkMap.values().iterator(); iter
                  .hasNext();) {
               ForeignKey fk = (ForeignKey) iter.next();
               System.out.println(fk.toForeignKey());
            }
         }

      } catch (Exception x) {
         x.printStackTrace();
      }
   }

}
