package dbutils;

import java.io.*;
import java.util.*;
/**
 *
 *  @author I. Burak Ozyurt
 *  @version $Id: Tokenizer.java,v 1.1.1.1 2005/10/25 18:32:59 bozyurt Exp $
 */
public class Tokenizer {
  StreamTokenizer stok;
  Map keywordMap = new HashMap();
  BufferedReader in = null;
  public final static int STRING = 1;
  public final static int INSERT = 2;
  public final static int INTO = 3;
  public final static int VALUES = 4;
  public final static int NULL_TOK = 5;
  public final static int TIMESTAMP = 6;
  public final static int LEFT_PAR = 6;
  public final static int RIGHT_PAR = 7;
  public final static int SEMICOL = 8;
  public final static int COMMA = 9;
  public final static int WORD = 10;
  public final static int NUMBER = 11;
  public final static int EOF = 100;



  public Tokenizer(String filename) throws IOException {

    stok = new StreamTokenizer( this.in = new BufferedReader( new FileReader(filename) ) );
    stok.resetSyntax();
    //stok.commentChar('-');
    stok.parseNumbers();
    stok.eolIsSignificant(true);
    stok.quoteChar('\'');
    stok.wordChars('a','z');
    stok.wordChars('A','Z');
    stok.wordChars('_','_');
    //stok.wordChars('.','.');
    //stok.wordChars('0','9');
    stok.ordinaryChar('(');
    stok.ordinaryChar(')');
    stok.ordinaryChar(',');
    stok.ordinaryChar(';');
    stok.whitespaceChars(' ',' ');
    stok.whitespaceChars('\t','\t');


    keywordMap.put("insert", new Integer(INSERT) );
    keywordMap.put("into", new Integer(INTO) );
    keywordMap.put("values", new Integer(VALUES) );
    keywordMap.put("timestamp", new Integer(TIMESTAMP) );

    // keywordMap.put("null", new Integer(NULL_TOK) );
  }

  public void shutdown() {
    if (in != null)
      try { in.close(); } catch(Exception x) {}
  }

  protected int getKeywordCode(String word) {
    Integer code = (Integer) keywordMap.get(word.toLowerCase());
    return (code != null) ? code.intValue() : -1;
  }

  public String getStringValue() {
    return stok.sval;
  }

  protected int getDelimiter(int c) {
    switch(c) {
      case '(' : return LEFT_PAR;
      case ')' : return RIGHT_PAR;
      case ',' : return COMMA;
      case ';' : return SEMICOL;
      default:
        return -1;
    }
  }

  public Number getNumber() {
    Double num = new Double( stok.nval );
    double diff = stok.nval - num.intValue();
    if ( Math.abs(diff) < 1e-10) {
      return new Integer( num.intValue() );
    }
    return num;
  }

  public int getNextToken() throws IOException{
    int ttype = stok.nextToken();
    int tc = -1;
    switch(ttype) {
      case StreamTokenizer.TT_WORD:
        tc = getKeywordCode( stok.sval);
        return ( tc == -1) ? WORD : tc;
      case StreamTokenizer.TT_NUMBER:
        return NUMBER;
      case StreamTokenizer.TT_EOF:
        return EOF;
      default:
        if (ttype == '\'') {
          return STRING;
        }
        tc = getDelimiter(ttype);
        return tc;
    }
  }


  public static void main(String[] args) {
     String filename = "c:/cygwin/home/bozyurt/test.sql";
     Tokenizer tokenizer = null;
     try {
       tokenizer = new Tokenizer(filename);

       int tc = -1;
       while ( (tc = tokenizer.getNextToken() ) != -1) {
         switch(tc) {
           case WORD:
             System.out.println("word: " + tokenizer.getStringValue() );
             break;
           case STRING:
            System.out.println("string: " + tokenizer.getStringValue() );
            break;
           case NUMBER:
              System.out.println("number: " + tokenizer.getNumber() );
              break;

         }
       }
     } catch(Exception x) {
       x.printStackTrace();
     } finally {
       if (tokenizer != null)
         tokenizer.shutdown();
     }
  }
}