#!/bin/bash


#-----------------------------
help()
{
cat << HELP

This script takes an input image and generate a binary brain mask through AFNI's 3dSkullStrip skull-stripping program.

DEPENDENCIES:

	3dSkullStrip	: `which 3dSkullStrip`
	3dcalc		: `which 3dcalc`


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -in          <file>    input intensity image
	-out         <file>    output binary mask

Optional
	-shrink_fac  <float>   shrink_fac parameter (default: 0.7 as suggested by Wang 2014, not 0.6).
	-exp_frac    <float>   exp_frac parameter.  (default: 0.1 as suggested by Wang 2014, also the program's default)

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-in) 
				input=$2;
				shift 2;;			# input image

			-out) 
				output=$2;
				shift 2;;			# output image

			-shrink_fac) 
				shrink_fac=$2
				shift 2;;			# HWA floodheight

			-exp_frac)
				exp_frac=$2
				shift 2;;

			-*) 
				echo "ERROR: no such option $1";
				helpshort;;
			 *) 
				break;;
		esac
	done
}



#---------------------------------



# --------------------------------
checkdependencies()
{
        for program in 3dSkullStrip 3dcalc; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
                        echo -e "\nThe dependencies include: 3dSkullStrip (AFNI), and 3dcalc (AFNI)\n"
                        exit 1
                fi
        done
        # make sure dramms/lib is also in the path
        p=`which dramms`
        pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
}



######################
### main
######################
if [ $# -lt 1 ]
then
	help
fi


### Default parameters
shrink_fac=0.7
exp_frac=0.1


### Reading the arguments
parse $*


### check dependencies
checkdependencies


# brain extraction
tmpbrain=`mktemp /tmp/3dSkullStrip-brain-"XXXXXX".nii.gz`
tmpmask=`mktemp /tmp/3dSkullStrip-mask-"XXXXXX".nii.gz`
\rm ${tmpbrain}
cmd="3dSkullStrip -shrink_fac ${shrink_fac} -exp_frac ${exp_frac} -input ${input} -prefix ${tmpbrain}"
echo ${cmd}
timeout 10m ${cmd} # allow 10min to generate results

# convert the extracted brain into a binary mask
tmpmask2=${tmpmask%.nii.gz}2.nii.gz
if [ -s ${tmpbrain} ]; then
	3dcalc -a ${tmpbrain} -expr "notzero(a)" -prefix ${tmpmask2} -datum byte
else
	fslmaths ${input} -mul 0 ${tmpmask2}
	#echo -e "\nError (bk1): ${tmpbrain} was not generated correctly. Program exits.\n"
	#exit 1
	echo -e "\nError (bk1): the following command failed \n${cmd}\nTherefore, we generated a blank image as output.\n"
fi

if [ -s ${tmpmask2} ]; then
	\mv ${tmpmask2} ${output}
	\rm ${tmpbrain} ${tmpmask}
	echo "output the brain mask into ${output}. Job finished successfully."
else
	echo -e "\nError (bk2): ${tmpmask2} was not generated correctly. Program exits.\n"
	\rm ${tmpbrain} ${tmpmask}
	exit 1
fi




