#!/bin/bash


#-----------------------------
help()
{
cat << HELP

This script takes an input image and generate a binary brain mask through ROBEX.

DEPENDENCIES:

	runROBEX.sh	: `which runROBEX.sh`
	3dcalc		: `which 3dcalc`


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -in   <file>	input intensity image
	-out  <file>	output binary mask

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-in) 
				input=$2;
				shift 2;;			# input image

			-out) 
				output=$2;
				shift 2;;			# output image

			-*) 
				echo "ERROR: no such option $1";
				helpshort;;
			 *) 
				break;;
		esac
	done
}


#----------------------------------
checkdependencies()
{
        for program in runROBEX.sh 3dcalc; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
                        echo -e "\nThe dependencies include: runROBEX.sh (ROBX tool) and 3dcalc (AFNI tool)\n"
                        exit 1
                fi
        done
        # make sure dramms/lib is also in the path
        p=`which dramms`
        pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
}


if [ $# -lt 1 ]
then
	help
fi




### Reading the arguments
parse $*


### checkdependencies (should be pre-installed in the system)
checkdependencies


# brain extraction
tmpbrain=`mktemp /tmp/robex-brain-"XXXXXX".nii.gz`
tmpmask=`mktemp /tmp/robex-mask-"XXXXXX".nii.gz`
cmd="runROBEX.sh ${input} ${tmpbrain}"
echo ${cmd}
timeout 10m ${cmd} # allow 10min to generate results

# convert the extracted brain into a binary mask
tmpmask2=${tmpmask%.nii.gz}2.nii.gz
if [ -s ${tmpbrain} ]; then
	3dcalc -a ${tmpbrain} -expr "notzero(a)" -prefix ${tmpmask2} -datum byte
else
	#echo -e "\nError (bk1): ${tmpbrain} was not generated correctly. Program exits.\n"
	#exit 1
	fslmaths ${input} -mul 0 ${tmpmask2}
	echo -e "\nError (bk1): the following command failed \n${cmd}\nTherefore, we generated a blank image as output.\n"
fi

\mv ${tmpmask2} ${output}
\rm ${tmpbrain} ${tmpmask}
echo "output the brain mask into ${output}."
echo "Job finished successfully."



