#!/bin/bash




#-----------------------------
help()
{
cat << HELP

########################################
Summary: This script takes an input brain MR image (T1-weighted) and normalizes its FOV to the standard FOV defined in the MNI152 space. This script can be run in a single linux machine or in a parallel cluster (e.g., PBS).


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -in      <file>     : input T1-weighted MRI (image must be in ANALYZE or Nifti format).
        -out     <file>     : output FOV mask image (in ANALYZE or Nifti format).

Optional:
        -ref     <path>     : path for template directory (default: $0/../data/intermediateTemplates)
        -debug              : debug mode, keep all intermediate results
        -runmode <int>      : 1 mutiple registrations run in parallel in the PBS/Mosix cluster (default if PBS/Mosix cluster exists);
                              2 multiple registrations run in parallel but in the same linux machine (default if PBS/Mosix cluster does not exist);
                              3 multiple registrations run sequentially in the same linux machine;

	-regqueue <char>    : cluster queue for registration jobs
	                      (default: r_max200)	
	                      note: this argument is only effect when the '-runmode' is set to 1. The r_max200 queue is specific to MGH martinos cluster (priority 9400, max job number 200). Users may need to adapt this queue to their own cluster.

########################################		     
Dependencies:
	FSL      < http://fsl.fmrib.ox.ac.uk/fsldownloads >
	DRAMMS   < https://www.nitrc.org/projects/dramms >
	NiftiSeg < http://sourceforge.net/projects/niftyreg >


########################################
Example Use:
	$0 -in myimage.nii.gz -out myimage_FOVnormalized.nii.gz


########################################
Author:
        Yangming Ou (yangming.ou@childrens.harvard.edu; yangming.ou@mgh.harvard.edu), 
        Harvard Medical Schol
        (c) 2014-2018


########################################
Reference:
	Y Ou, L Zöllei, X Da, K Retzepi, SN Murphy, ER Gerstner, BR Rosen, PE Grant, J Kalpathy-Cramer, RL Gollub, "Field of View Normalization in Multi-Site Brain MRI", Neuroinformatics, doi: 10.1007/s12021-018-9359-z, (2018).


HELP
exit 1
}




#-----------------------------
parse()
{
        while [ -n "$1" ]; do
                case $1 in
                        -h)
                                help;
                                shift 1;;                       # help is called

                        -in)
                                inputfile=$2;
                                shift 2;;                       # input folder

                        -out)
                                outputfile=$2;
                                shift 2;;                       # output image

			-ref)
				inrefdir=$2;
				shift 2;;                       # user-input template directory

                        -debug)
                                debug=1;                        # debug mode
                                shift 1;;

                        -runmode)
                                runmode=$2;                     # default is 1, but can be 2 or 3
                                shift 2;;

			-regqueue)
				regqueue=$2
				shift 2;;

                        -*)
                                echo "ERROR: no such option $1";
                                helpshort;;
                         *)
                                break;;
                esac
        done
}


# load library or executables, they are dependencies of this pipeline
checkdependencies()
{
	for program in dramms flirt seg_LabFusion; do
		p=`which ${program} 2>/dev/null`
		if [ -z "${p}" ]; then
			echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
			exit 1
		fi
	done
	# make sure dramms/lib is also in the path
	p=`which dramms`
	pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
}

#-------------------------------
getFilePrefix()
{
        bName=`basename $1`
        prefix=${bName%.nii.gz}
        prefix=${prefix%.nii}
        prefix=${prefix%.hdr}
        prefix=${prefix%.img}  # right now only support analyze or nifti format
}


#-------------------------------
mywait()
{
	# usage:
	# mywait $1(filenames) $2(targetnumber)
	targetnumber=$2
	if [ -z "${targetnumber}" ]; then targetnumber=1; fi
	count=0 
	interval=2  # check every 2 seconds
	
        #echo "wait for file $1"
        while [ ${count} -lt 108000 ]; do  # maximum wait 60 hours
		number=`ls -l $1 2>/dev/null |wc -l`
		if [ -z "${number}" ]; then number=0; fi
                if [ ${number} -lt ${targetnumber} ]; then
                        if [ $(( $count % 30 )) == 0 ]; then
                                minutes=$(( ${count} / 30 ))
                                echo -e "\nwaiting for $1, ${minutes} minutes have passed, actual # = ${number}, target # = ${targetnumber}...\n"
                        fi
                        sleep ${interval}
                	count=$(($count + 1))
		else
			break;
		fi
        done
}



# ----------------------------------------------------------------------------
# regularization for linear registration
calculatepenalty()
{
        transform=$1
        a11=`cat $transform | sed -n 1p | awk '{ print $1 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a12=`cat $transform | sed -n 1p | awk '{ print $2 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a13=`cat $transform | sed -n 1p | awk '{ print $3 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a21=`cat $transform | sed -n 2p | awk '{ print $1 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a22=`cat $transform | sed -n 2p | awk '{ print $2 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a23=`cat $transform | sed -n 2p | awk '{ print $3 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a31=`cat $transform | sed -n 3p | awk '{ print $1 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a32=`cat $transform | sed -n 3p | awk '{ print $2 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`
        a33=`cat $transform | sed -n 3p | awk '{ print $3 }' | sed 's/e/*10^/g;s/ /*/' | bc -l | tr -d -`

        if [ `echo ${a11} '>' ${a12} | bc -l` == 1 ]; then a1=${a11}; else a1=${a12}; fi
        if [ `echo ${a1}  '>' ${a13} | bc -l` == 1 ]; then a1=${a1};  else a1=${a13}; fi
        if [ `echo ${a21} '>' ${a22} | bc -l` == 1 ]; then a2=${a21}; else a2=${a22}; fi
        if [ `echo ${a2}  '>' ${a23} | bc -l` == 1 ]; then a2=${a2};  else a2=${a23}; fi
        if [ `echo ${a31} '>' ${a32} | bc -l` == 1 ]; then a3=${a31}; else a3=${a32}; fi
        if [ `echo ${a3}  '>' ${a33} | bc -l` == 1 ]; then a3=${a3};  else a3=${a33}; fi

        penalty=`echo "(1 - ${a1})*(1 - ${a2})*(1 - ${a3})" | bc -l`

        echo ${penalty} | tr -d -
}


#-------------------------------
trap cleanup EXIT INT QUIT TERM
cleanup()
{
    if [ -n ${tmpdir} ] && [ "${debug}" == 0 ]; then
        \rm -rf ${tmpdir}
    fi
}


##############################
##############################
## Main Script
##############################
##############################

# help if needed
if [ $# -lt 1 ]
then
        help
fi


#--------------------------------------
### preparation 1: make sure all dependent programs are in the system path
checkdependencies


#--------------------------------------
### preparation 2: parse input argument
debug=0   # default
inrefdir=""
pbscluster=0
mosixcluster=0
if [ ! -z `which pbsubmit 2>/dev/null` ]; then
      pbscluster=1
elif [ ! -z `which mosbatch 2>/dev/null` ]; then
      mosixcluster=1
      mos_fnndsc="mosbatch -q -j10.36.131.42,10.36.131.45,10.36.131.46,10.36.131.48,10.36.133.204,10.36.133.203,10.36.133.202,10.36.133.205" #see http://durban.tch.harvard.edu/wiki/index.php/Cluster_Overview#Total for the names of the nodes in these ip addresses
fi
if [ ${pbscluster} == 1 ]||[ ${mosixcluster} == 1 ]; then
      runmode=1 # default when a cluter exists
else
      runmode=2 # default when a cluster does not exist
fi
regqueue=r_max200;  # default
parse $*

if [ ${runmode} != 1 ]&&[ ${runmode} != 2 ]&&[ ${runmode} != 3 ]; then
        echo -e "\nError: the -runmode option must be either 1, 2 or 3. Program exits. Please reset."
        exit 1
fi



#----------------------------------------
### preparation 3: get folder and prefix
outputfolder=`dirname ${outputfile}`
if [ ! -d ${outputfolder} ]; then
        mkdir -p ${outputfolder}
fi
outputfolder=`readlink -f ${outputfile}`
outputfolder=`dirname ${outputfolder}`
getFilePrefix ${outputfile}
outputprefix=${prefix}
outputfile_full=${outputfolder}/`basename ${outputfile}`
if [ -f ${outputfile_full} ]; then
	\rm ${outputfile_full}	
fi

inputfolder=`dirname ${inputfile}`
inputfolder=`readlink -f ${inputfile}`
inputfolder=`dirname ${inputfolder}`
getFilePrefix ${inputfile}
inputprefix=${prefix}
inputfile_full=${inputfolder}/`basename ${inputfile}`
echo -e "\ninputfile_full=${inputfile_full}\n"
echo -e "outputfile_full=${outputfile_full}\n"

######
# check if input image exists and if it is in the right format
    if [ ! -f ${inputfile_full} ]; then
          echo ""
          echo "error: the input image (${inputfile_full}) does not exist."
          echo "program terminated."
          echo ""
          exit 1
    fi
    extension4=${inputfile_full: -4}
    extension7=${inputfile_full: -7}
    if [ "${extension4}" != ".nii" ]&&[ "${extension4}" != ".hdr" ]&&[ "${extension4}" != ".img" ]&&[ "${extension7}" != ".nii.gz" ]; then
            echo ""
            echo "error: the input image must be in ANALYZE (.hdr, .img) or NIFTI (.nii, .nii.gz) format."
            echo "program terminated."
            echo ""
            exit 1
    fi
# check if the output image is in the right format
    extension4=${outputfile_full: -4}
    extension7=${outputfile_full: -7}
    if [ "${extension4}" != ".nii" ]&&[ "${extension4}" != ".hdr" ]&&[ "${extension4}" != ".img" ]&&[ "${extension7}" != ".nii.gz" ]; then
            echo ""
            echo "error: the output image must be in ANALYZE (.hdr, .img) or NIFTI (.nii, .nii.gz) format."
            echo "program terminated."
            echo ""
            exit 1
    fi



#---------------------------------------
### preparation 4: make temporary directory to save intermediate results
tmpdir=`ls -d ${outputfolder}/intermediateresults-${outputprefix}-* 2>/dev/null |sed -n 1p`
if [ -z "${tmpdir}" ]; then
	tmpdir=`mktemp -d ${outputfolder}/intermediateresults-${outputprefix}-XXXXXX`
fi



#---------------------------------------                         
### step 1, transform standardized FOVs from intermediate templates to the target image space
programdir=`readlink -f $0`
programdir=`dirname ${programdir}`
if [ -z "${inrefdir}" ]; then 
	templatesdir=`readlink -f ${programdir}/../atlases/adultT1w` # default
else
	templatesdir=${inrefdir}
fi
echo -e "\nprogram dir = ${programdir}\n"
echo -e "\ndirectory for templates = ${templatesdir}\n"
# check input template dir contents
nImages=`ls ${templatesdir}/Template*_padded.nii.gz 2>/dev/null |wc -l`
nLabels=`ls ${templatesdir}/Template*_padded_label.nii.gz 2>/dev/null |wc -l`
nFOVs=`ls ${templatesdir}/Template*_padded_normalizedFOVmask.nii.gz 2>/dev/null |wc -l`
if [ -z "${nImages}" ]; then nImages=0; fi
if [ -z "${nLabels}" ]; then nLabels=0; fi
if [ -z "${nFOVs}" ];   then nFOVs=0;   fi
if [ ${nImages} != ${nLabels} ]||[ ${nLabels} != ${nFOVs} ]||[ ${nImages} -lt 6 ]; then
	echo -e "\nError: the template directory (${templatesdir}) should have images named Template*_padded.nii.gz, labels (brain masks) named as Template*_padded_label.nii.gz and FOV normalization masks named Template*_padded_normalizedFOVmask.nii.gz. The numbers of images, labels and FOV masks should be the same and at least 6, that is, at least 6 templates (atlases) are needed. Program exits.\n"
	exit 1
fi

 
# register
holdqueue=""
target=${inputfile_full}
for template in ${templatesdir}/Template*_padded.nii.gz; do
	templatepre=`basename ${template}`
	templatepre=${templatepre%_padded.nii.gz}
	
	reg=${tmpdir}/reg_${templatepre}_to_${inputprefix}.nii.gz
	mat=${tmpdir}/mat_${templatepre}_to_${inputprefix}.mat

	cmd="dramms -S ${template} -T ${target} -O ${reg} -D ${mat} -a 2" # -v -v"
	if [ ! -f ${reg} ]||[ ! -f ${mat} ]; then 
		echo -e "\naffinely register ${template} to ${inputprefix}\n${cmd}"
		if [ ${runmode} == 1 ]; then
	              if [ ${pbscluster} == 1 ]; then
        	              pbsubmit -q ${regqueue} -c "${cmd}" # p20 queue, priority 10300, max 100 jobs
	              elif [ ${mosixcluster} == 1 ]; then
        	              ${mos_fnndsc} ${cmd} &
				sleep 5
	              fi
		elif [ ${runmode} == 2 ]; then
			${cmd} &
		else
			${cmd} 
		fi
	fi
done

# sync, wait until all files have been generated
numTemplates=`ls ${templatesdir}/Template*_padded.nii.gz 2>/dev/null | wc -l`
mywait "${tmpdir}/mat_*_to_${inputprefix}.mat" ${numTemplates}
sleep 5
mywait "${tmpdir}/reg_*_to_${inputprefix}.nii.gz" ${numTemplates}
sleep 2
echo -e "\nall ${numTemplates} affine registration done. now template selection and fusion.\n"



# warp FOV masks
for template in ${templatesdir}/Template*_padded.nii.gz; do
       	templatepre=`basename ${template}`
        templatepre=${templatepre%_padded.nii.gz}
	templateFOV=${templatesdir}/${templatepre}_padded_normalizedFOVmask.nii.gz
	mat=${tmpdir}/mat_${templatepre}_to_${inputprefix}.mat
	reg=${tmpdir}/reg_${templatepre}_to_${inputprefix}.nii.gz
	wrpedFOV=${tmpdir}/wrpedFOVmask_${templatepre}_to_${inputprefix}.nii.gz

	# apply the transformation to the mask
	if [ ! -f ${wrpedFOV} ]; then
		dramms-warp ${templateFOV} ${mat} ${wrpedFOV} -n -t ${target}	
	fi
done


# initial fused FOV mask
f4d=${tmpdir}/4D_iter0.nii.gz
fusedFOV=${tmpdir}/normalizedFOVmask_iter0.nii.gz
if [ ! -f ${f4d} ]; then
	cmd="fslmerge -t ${f4d} ${tmpdir}/wrpedFOVmask_Template*_to_${inputprefix}.nii.gz"
	echo -e "\n${cmd}\n"
	${cmd}
	fslmaths ${f4d} -bin ${f4d} -odt char
fi

cmd="seg_LabFusion -in ${f4d} -out ${fusedFOV} -STAPLE"
echo -e "\n${cmd}\n"
${cmd}
if [ ! -f ${fusedFOV} ]; then
	cmd="seg_LabFusion -in ${f4d} -out ${fusedFOV} -MV"
	echo -e "\n${cmd}\n"
	${cmd}
fi
\rm ${f4d}


#---------------------------------------
### step 2, 3 in iterations

#simcalculator=/usr/pubsw/packages/DRAMMS/1.4.3/lib/CalculateImageSimilarity
simcalculator=CalculateImageSimilarity
for iter in 1 2; do  # two iterations
	#---------------------------------------
	### step 2, template selection
	#---------------------------------------
	# template selection a: find the threshold
	txtsim=${tmpdir}/similarity_allTemplates_to_${inputprefix}_iter${iter}.txt
	\rm ${txtsim} 2>/dev/null
	txtsel=${tmpdir}/selectedTemplates_to_${inputprefix}_iter${iter}.txt
	targetbyte=${tmpdir}/byte_${inputprefix}_iter${iter}.nii.gz
	if [ ${iter} == 1 ]; then
		ConvertImage -e --reset-scaling -t uchar -m 0 -M 255 ${target} ${targetbyte}
	else
		ConvertImage -e --reset-scaling -t uchar -m 0 -M 255 ${tmpout} ${targetbyte}
	fi

	max=0
	for template in ${templatesdir}/Template*_padded.nii.gz; do
        	templatepre=`basename ${template}`
	        templatepre=${templatepre%_padded.nii.gz}
		templateFOV=${templatesdir}/${templatepre}_padded_normalizedFOVmask.nii.gz
		mat=${tmpdir}/mat_${templatepre}_to_${inputprefix}.mat
		reg=${tmpdir}/reg_${templatepre}_to_${inputprefix}.nii.gz
		wrpedFOV=${tmpdir}/wrpedFOVmask_${templatepre}_to_${inputprefix}.nii.gz

		# apply the transformation to the mask
		if [ ! -f ${wrpedFOV} ]; then
			dramms-warp ${templateFOV} ${mat} ${wrpedFOV} -n -t ${target}	
		fi

		# measure similarity
		preiter=$(( ${iter} - 1 ))
		fusedFOV_previousiter=${tmpdir}/normalizedFOVmask_iter${preiter}.nii.gz
		regbyte=${tmpdir}/byte_reg_${templatepre}_to_${inputprefix}_iter${iter}.nii.gz
		if [ ${iter} == 1 ]; then
			ConvertImage -e --reset-scaling -t uchar -m 0 -M 255 ${reg} ${regbyte}
		else
			reg2=${tmpdir}/truncated_reg_${templatepre}_to_${inputprefix}.nii.gz
			fslmaths ${fusedFOV_previousiter} -mul ${reg} ${reg2}
			ConvertImage -e --reset-scaling -t uchar -m 0 -M 255 ${reg2} ${regbyte}
			\rm ${reg2}
		fi

	        cc=`${simcalculator} ${targetbyte} ${regbyte} -C -I | grep CC | awk '{print $3}'`
		nmi=`${simcalculator} ${targetbyte} ${regbyte} -N -I | grep MI | awk '{print $3}'`
        	penalty=`calculatepenalty ${mat}`
		nmicc=`echo "${cc}*${nmi}-${penalty}" | bc -l`
		dice=`c3d -verbose ${wrpedFOV} ${fusedFOV_previousiter} -overlap 1 | sed -n 8p | awk '{ print $4 }'`
		if [ ${iter} == 1 ]; then
			dice=`echo "${dice} > 0.45" | bc -l`  # in iteration 1, because we can trust fused FOVmask in iter0 too much, we binarize dice by a threshold of 0.45 (was 0.6 until 6/18/2017)
		fi
		nmicc=`echo "${nmicc}*${dice}" | bc -l`
		echo "${templatepre} ${nmicc}" >> ${txtsim}

        	if [ `echo ${nmicc} '>=' ${max} |bc -l` == 1 ]; then
			max=${nmicc}
		fi
        	echo "for ${inputprefix}, iteration ${iter}, template ${templatepre}: cc=${cc}, nmi=${nmi}, penalty=${penalty}, nmicc=${nmicc}"
	done
	thre=`echo "0.9*${max}" | bc -l`
	msg="max=${max} threshold=${thre}" 
	echo $msg
	echo $msg >> ${txtsim}

	# template selection b: selection
	kept=""
	for template in ${templatesdir}/Template*_padded.nii.gz; do
		echo ${template}
		templatepre=`basename ${template}`
        	templatepre=${templatepre%_padded.nii.gz}
		nmicc=`grep "${templatepre} " ${txtsim} | sed -n 1p | awk '{ print $2 }'`
		wrpedFOV=${tmpdir}/wrpedFOVmask_${templatepre}_to_${inputprefix}.nii.gz

		if [ `echo ${nmicc} '>=' ${thre} |bc -l` == 1 ]; then
			echo "after threshoding, keep ${templatepre}"
			kept=`echo ${kept} ${wrpedFOV}`
		fi
	done
	echo ${kept} > ${txtsel}
	\rm ${tmpdir}/byte*.nii.gz



	#---------------------------------------
	### step 3. fusion

	# merge selected templates
	f4d=${tmpdir}/4D_iter${iter}.nii.gz
	fusedFOV=${tmpdir}/normalizedFOVmask_iter${iter}.nii.gz
	cmd="fslmerge -t ${f4d} ${kept}"
	echo -e "\n${cmd}\n"
	#if [ ${runmode} == 1 ]; then
	#	jid=$(pbsubmit -q p30 -c "${cmd}") # p30 queue, priority 10600, max 75 jobs
	#	jid=`echo ${jid} | awk '{ print $NF }'`
	#else
        	${cmd}
	#fi
	#mywait ${f4d}
	fslmaths ${f4d} -bin ${f4d} -odt char

	cmd="seg_LabFusion -in ${f4d} -out ${fusedFOV} -STAPLE"
	echo -e "\n${cmd}\n"	
	#if [ ${runmode} == 1 ]; then
        #	jid=$(pbsubmit -q p30 -o "-W depend=afterok:${jid}" -c "${cmd}") # p30 queue, priority 10600, max 75 jobs
	#        jid=`echo ${jid} | awk '{ print $NF }'`
	#else
        	${cmd}
	#fi
	#mywait ${fusedFOV} 1
	if [ ! -f ${fusedFOV} ]; then
		cmd="seg_LabFusion -in ${f4d} -out ${fusedFOV} -MV"
	        echo -e "\n${cmd}\n"
		${cmd}
	fi
	mywait "${fusedFOV}" 1
	\rm ${f4d}

	# FOV normalization
	tmpout=${tmpdir}/${inputprefix}_normalizedFOV_iter${iter}.nii.gz
	cmd="fslmaths ${target} -mul ${fusedFOV} ${tmpout}"
	echo -e "\n${cmd}\n"
	#if [ ${runmode} == 1 ]; then
        #	jid=$(pbsubmit -q p30 -o "-W depend=afterok:${jid}" -c "${cmd}") # p30 queue, priority 10600, max 75 jobs
	#        jid=`echo ${jid} | awk '{ print $NF }'`
	#else
        	${cmd}
	#fi
	#mywait ${tmpout} 1
done


# the results of the 2nd iteration is the final results
\cp ${tmpout} ${outputfile_full}
fusedFOVfinal=${outputfolder}/${outputprefix}_normalizedFOVmask.nii.gz
\cp ${fusedFOV} ${fusedFOVfinal}
echo -e "\nFOV normalization has successfully finished.\n"
echo -e "\nThe FOV normalized image is saved as ${outputfile_full}\n"
echo -e "\nThe associated FOV mask is saved as ${fusedFOVfinal}.\n"
echo -e "\nProgram now exits.\n"




# ----------------------------------------------------------------------------
# clean up and exit
cleanup



