#!/bin/bash


#-----------------------------
help()
{
cat << HELP

This script takes an input image and generate a binary brain mask through FreeSurfer's HWA skull-stripping program.

DEPENDENCIES:

	mri_watershed	: `which mri_watershed`
	3dcalc		: `which 3dcalc`


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -in   <file>         input intensity image
	-out  <file>         output binary mask

Optional
	-height  <integer>   HWA's floodheight parameter (default: 25).
	-atlas   <0/1>       Using (1) atlas or not using (0) atlas.  (default: 0)

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-in) 
				input=$2;
				shift 2;;			# input image

			-out) 
				output=$2;
				shift 2;;			# output image

			-height) 
				height=$2
				shift 2;;			# HWA floodheight

			-atlas)
				atlas=$2
				shift 2;;

			-*) 
				echo "ERROR: no such option $1";
				helpshort;;
			 *) 
				break;;
		esac
	done
}






# --------------------------------
checkdependencies()
{
        for program in mri_watershed 3dcalc; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
                        echo -e "\nThe dependencies include: mri_watershed (FreeSurfer), and 3dcalc (AFNI)\n"
                        exit 1
                fi
        done
        # make sure dramms/lib is also in the path
        p=`which dramms`
        pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
}




##########################
### main
##########################
if [ $# -lt 1 ]
then
	help
fi


### Default parameters
height=25
atlas=0


### Reading the arguments
parse $*

### check dependencies
checkdependencies


if [ ${atlas} -eq 0 ]; then
	useatlas=""
else
	useatlas="-atlas"
fi

######################
# brain extraction
tmpbrain=`mktemp /tmp/hwa-brain-"XXXXXX".nii.gz`
tmpmask=`mktemp /tmp/hwa-mask-"XXXXXX".nii.gz`
echo tmpbrain=$tmpbrain
cmd="mri_watershed -less ${useatlas} -h ${height} ${input} ${tmpbrain}"
echo ${cmd}
timeout 10m ${cmd}  # kill the process if  no results in 3min (to prevent indefinite dead-end loops).

# if the above fails, try another round using slightly different parameter
if [ ! -s ${tmpbrain} ]; then
	cmd="mri_watershed ${useatlas} -h ${height} -wat ${input} ${tmpbrain}"
	echo -e "\nthe previous command failed. Now try a different set of parameter\n\n${cmd}\n"
	timeout 10m ${cmd} 
fi

# if still fails, output a blank file
if [ ! -s ${tmpbrain} ]; then
	echo -e "\nThe following command failed \n${cmd}\nTherefore, we generated a blank image as output.\n"
	fslmaths ${input} -mul 0 ${tmpbrain}
fi

# convert the extracted brain into a binary mask. 
tmpmask2=${tmpmask%.nii.gz}2.nii.gz
3dcalc -a ${tmpbrain} -expr "notzero(a)" -prefix ${tmpmask2} -datum byte

\mv ${tmpmask2} ${output}
\rm ${tmpbrain} ${tmpmask} 
echo "output the brain mask into ${output}. Job finished successfully."



