#!/bin/bash


#-----------------------------
help()
{
cat << HELP

This script takes an input image and generate a binary brain mask through BrainSuite's BSE skull-stripping program (version: BrainSuite16a).

DEPENDENCIES:

	bse		: `which bse 2>/dev/null`
	3dcalc		: `which 3dcalc 2>/dev/null`


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -in   <file>  input intensity image
	-out  <file>  output binary mask

Optional
	-s  <float>   edge detection constant [default: 0.62 suggested by Hartley 2006]
	-d  <float>   diffusion constant [default: 35 suggested by Hartley 2006]

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-in) 
				input=$2;
				shift 2;;			# input image

			-out) 
				output=$2;
				shift 2;;			# output image

			-s) 
				s=$2
				shift 2;;			# HWA floodheight

			-d)
				d=$2
				shift 2;;

			-*) 
				echo "ERROR: no such option $1";
				helpshort;;
			 *) 
				break;;
		esac
	done
}



#---------------------------------
# check library or executables, they are dependencies of this pipeline
checkdependencies()
{
        for program in bse 3dcalc; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
                        echo -e "\nThe dependencies include: bse (BrainSuite16a), and 3dcalc (AFNI)\n"
                        exit 1
                fi
        done
        # make sure dramms/lib is also in the path
        p=`which dramms`
        pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
}



########################
## main
########################

if [ $# -lt 4 ]
then
	help
fi


### Default parameters
s=0.62
d=35   #25


### Reading the arguments
parse $*


# brain extraction
tmpmask255=`mktemp /tmp/bse-mask0-"XXXXXX".nii.gz`
tmpmask=`mktemp /tmp/bse-mask-"XXXXXX".nii.gz`
tmpmaskrightorientation=`mktemp /tmp/bse-mask-"XXXXXX".nii.gz`
# we need a header in the right orientation and byte datatype
orientation=""
tmpinput=`mktemp /tmp/bse-mask-"XXXXXX".nii.gz`
filetype=`fslhd ${input} |grep file_type | awk '{ print $2 }'`
filetype=`echo ${filetype} | tr '[:lower:]' '[:upper:]'`
if [[ ${filetype} == *NIFTI* ]]; then
    # if nifti, always reorient to lps orientation in order to run bse properly
    # remember to reorient back when bse is done
    orientation1=`fslhd ${input} |grep sform_xorient |awk '{ print $2 }' |cut -c1`
    if [ ${orientation1} == "U" ]; then
	orientation1=`fslhd ${input} |grep qform_xorient |awk '{ print $2 }' |cut -c1`
    fi
    orientation2=`fslhd ${input} |grep sform_yorient |awk '{ print $2 }' |cut -c1`
    if [ ${orientation2} == "U" ]; then
	orientation2=`fslhd ${input} |grep qform_yorient |awk '{ print $2 }' |cut -c1`
    fi
    orientation3=`fslhd ${input} |grep sform_zorient |awk '{ print $2 }' |cut -c1`
    if [ ${orientation3} == "U" ]; then
	orientation3=`fslhd ${input} |grep qform_zorient |awk '{ print $2 }' |cut -c1`
    fi
    orientation=${orientation1}${orientation2}${orientation3}
    \rm ${tmpinput}
    echo "the input is a nifti image with orientation ${orientation}"
    3dresample -inset ${input} -prefix ${tmpinput} -orient LPI
fi


# bse extraction of mask
cmd="bse -i ${tmpinput} --mask ${tmpmask255} -s ${s} -d ${d} -r 2 -n 4 -p"
echo ${cmd}
timeout 5m ${cmd} # allow 5min to generate the results
if [ -s ${tmpmask255} ]; then
	\rm ${tmpmask}
	3dcalc -a ${tmpmask255} -expr 'a/255' -prefix ${tmpmask} -datum byte
else
	#echo -e "\nError (bk1): ${tmpmask255} was not generated successfully. Program exits.\n"
	#exit 1
	fslmaths ${input} -mul 0 ${tmpmask}
	echo -e "\nError (bk1): the following command failed \n${cmd}\nTherefore, we generated a blank image as output.\n"
fi


if [[ ${filetype} == *NIFTI* ]]; then
    \rm ${tmpmaskrightorientation}
    qformcode=`fslorient -getqformcode $input`
    sformcode=`fslorient -getsformcode $input`
    qform=`fslorient -getqform $input`
    sform=`fslorient -getsform $input`
    3dresample -inset ${tmpmask} -prefix ${tmpmaskrightorientation} -orient ${orientation}
    fslorient -setsformcode ${sformcode} ${tmpmaskrightorientation}  
    fslorient -setqformcode ${qformcode} ${tmpmaskrightorientation}
    fslorient -setsform ${sform} ${tmpmaskrightorientation}
    fslorient -setqform ${qform} ${tmpmaskrightorientation}
fi

\mv ${tmpmaskrightorientation} ${output}
\rm ${tmpmask} ${tmpmask255} ${tmpinput}

echo "output the brain mask into ${output}. Job finished successfully."



