#!/bin/bash



##############################
##############################
## Defining Functions Starts
##############################
##############################



#-------------------------------
help()
{
cat <<HELP
##############################################

This script does automatic skull stripping of an input brain MR image.
The image is preferrably T1w or MPRAGE MRI sequences.
But the program can also apply to T2 or ADC sequences (see examples below).
	
##############################################
USAGE :	$0 [OPTIONS]
OPTIONS:

Requiredd:	-in	   < file  >	input image file (ANALYZE or NIFTI format)
		-out	   < file  >	output brain mask label file (ANALYZE or NIFTI format)

Optional:  
		-outimage  < file  >    output the skull stripped image (apply the computed mask, ANALYZE or NIFTI format)
		-pediatric              using pediatric T1 atlases, typically for 0-6 years of age (default: adult T1 atlases)
		-contrast		using adult T1-contrast-enhanced atlases (default: adult T1 atlases with no contrast)
		-nonT1			to skull strip structural sequences such as T2-weighted, MPRAGE (default: off, for T1-weighted)
		-bc			bias correction in the beginning (default: on, by N4)
		-t	   < 0 | 1 >	keep intermediate results (1, yea) or not (0, no: default)
		-d         < path  >    path to keep intermediate results (only effective when '-t 1'). if not specificed, then a randomly generated subfolder in the folder for the output file.
                -ref       < path >     instead of using the default path for atlases, users input a directory containing atlas images, brain mask labels and FOV masks;
					(default: `readlink -f $0 | rev | cut -d/ -f3- |rev`/atlases/adultT1w)
		-runmode		1 mutiple registrations run in parallel in the PBS/Mosix cluster (default if PBS/Mosix cluster exists);
	                                2 multiple registrations run in parallel but in the same linux machine (default if PBS/Mosix cluster does not exist);
        	                        3 multiple registrations run sequentially in the same linux machine;

		-regqueue  < char >     cluster queue for registration jobs 
		                        (default: r_max200)
		                        note: this argument is only effective when '-runmode' is set at 1; r_max200 is specific to MGH martinos launchpad cluster (priority 9400, max job number 200). Users may need to adapt to their clusters)

		-fusequeue < char >     cluster queue for label fusion jobs 
		                        (default: p50)
		                        note: this argument is only effective when '-runmode' is set at 1; r_max100 is specific to MGH martinos launchpad cluster (priority 11200, max job number 20). Users may need to adapt to their clusters)

		-ssqueue   < char >	cluster queue for running hundreds of very fast (<1 or <3 min) skull stripping algorithms (BET, BSE, 3dSkullStrip, HWA, ROBEX)  with varying parameters 
		                        (default: p30)
		                        note: this argument is only effective when '-runmode' is set at 1; r_max200 is specific to MGH martinos launchpad cluster (priority 10600, max job number 75). Users may need to adapt to their clusters)


########################################
Example Use:
	# to compute the brain mask for a T1 image
        $0 -in myT1image.nii.gz -out myT1image_brainmask.nii.gz
	
	# to compute the brain mask and skull striped image
	$0 -in myT1image.nii.gz -out myT1image_brainmask.nii.gz -outimage myT1image_skullstripped.nii.gz

	# to compute the brain mask for a T2 image
	$0 -in myT2image.nii.gz -out myT2image_brainmask.nii.gz -nonT1

	# to compute the brain mask for a pediatric/neonatal T1 image
	$0 -in infantT1image.nii.gz -out infantT1image_brainmask.nii.gz -pediatric

	# to compute the brain mask for a MPRAGE image with contrast (gadolinum)
	$0 -in myMPRAGEimage.nii.gz -out myMPRAGEimage_brainmask.nii.gz -contrast

	# to compute the brain mask for an ADC map (pediatric)
	$0 -in myADCmap.nii.gz -out myADCmap_brainmask.nii.gz -ref readlink -f `readlink -f $0 | rev | cut -d/ -f3- |rev`/atlases/pediatricADC
	

########################################
Author:
        Yangming Ou (yangming.ou@mgh.harvard.edu), Massaschusetts General Hospital, Harvard Medical School
	Yangming Ou (yangming.ou@childrens.harvard.edu), Boston Children's Hospital, Harvard Medical School
        (c) 2014-2020


########################################
Reference:
        [1] pending.
	[2] Y Ou, L Zöllei, X Da, K Retzepi, SN Murphy, ER Gerstner, BR Rosen, PE Grant, J Kalpathy-Cramer, RL Gollub, "Field of View Normalization in Multi-Site Brain MRI", Neuroinformatics, doi: 10.1007/s12021-018-9359-z, (2018).

##############################################

HELP
exit 1
}


#-------------------------------
FileAtt()
{
	IP=$1;
	
	ext=`echo ${IP##*.}`
	bName=`basename ${IP%.${ext}}`
	
	if [ "$ext" == "gz" ]
	then
		ext=`echo ${bName##*.}`.${ext}
		bName=`basename ${IP%.${ext}}`
	fi
	
	if [ "$ext" != "nii.gz" ] && [ "$ext" != "hdr" ] && [ "$ext" != "img" ] && [ "$ext" != "nii" ]
	then
		echo -e "\nERROR: Input file extension $ext not recognized! Please check ..."
		cleanup
	fi
	
	echo $ext $bName
}


#-------------------------------
executionTime()
{
	endTimeStamp=`date +%s`
	total=$[ (${endTimeStamp} - ${startTimeStamp})]
	if [ ${total} -gt 60 ]
	then
		if [ ${total} -gt 3600 ]
		then
			if [ ${total} -gt 86400 ]
			then
				echoV "\nExecution time:  $[ ${total} / 86400]d $[ ${total} % 86400 / 3600]h $[ ${total} % 86400 % 3600 / 60]m $[ ${total} % 86400 % 3600 % 60]s" 1
			else
				echoV "\nExecution time:  $[ ${total} / 3600]h $[ ${total} % 3600 / 60]m $[ ${total} % 3600 % 60]s" 1
			fi
		else
			echoV "\nExecution time:  $[ ${total} / 60]m $[ ${total} % 60]s" 1
		fi
	else
		echoV "\nExecution time:  $[ ${total} % 60]s" 1
	fi
}


#-------------------------------
getFilePrefix()
{
	bName=`basename $1`
	prefix=${bName%.nii.gz}
	prefix=${prefix%.nii}
	prefix=${prefix%.hdr}
	prefix=${prefix%.img}  # right now only support analyze or nifti format
}

#-------------------------------
calculateDICE()
{
	dice=`c3d -verbose $1 $2 -overlap 1 | grep Dice | awk '{ print $4 }'`
	echo ${dice}
}


#-------------------------------
mywait()
{
        # usage:
        # mywait $1(filenames) $2(targetnumber)
        targetnumber=$2
	if [ -z "${targetnumber}" ]; then targetnumber=1; fi # default is 1
        count=0
        interval=2  # check every 2 seconds

        #echo "wait for file $1"
         # while [ ${count} -lt 108000 ]; do  # maximum wait 60 hours
        while [ ${count} -lt 10800 ]; do  # maximum wait 6 hour
                number=`ls -l $1 2>/dev/null |wc -l`
		if [ -z "${number}" ]; then number=0; fi
                if [ ${number} -lt ${targetnumber} ]; then
                        if [ $(( $count % 30 )) == 0 ]; then
                                minutes=$(( ${count} / 30 ))
                                echo -e "\nwaiting for $1, ${minutes} minutes have passed, actual # = ${number}, target # = ${targetnumber}...\n"
                        fi
                        sleep ${interval}
                        count=$(($count + 1))
                else
                        break;
                fi
        done
}


#-------------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			-in) 
				input=$2;
				temp=`FileAtt $input`				
				InExt=`echo $temp | awk '{ print $1 }'`
				InbName=`echo $temp | awk '{ print $2 }'`
				shift 2;;			# input image

			-out) 
				ssoutput=$2;
				temp=`FileAtt $ssoutput`				
				OutExt=`echo $temp | awk '{ print $1 }'`
				OutbName=`echo $temp | awk '{ print $2 }'`
				shift 2;;			# output brain mask image

			-outimage)
				outimage=$2;
				temp=`FileAtt ${outimage}`
				OutImageExt=`echo ${temp} | awk '{ print $1 }'`
				OutImagebName=`echo ${temp} | awk '{ print $2 }'`
				shift 2;;			# optional output ss-ed image

			-pediatric)
				usingPediatricAtlas=1
				shift 1;;

			-contrast)
				usingT1GadAtlas=1
				shift 1;;

			-nonT1)
				othermodality=1
				shift 1;;

			-bc)	
				biascorrection=1
				shift 1;;

			-t)
				keepIntermediateResultsOrNot=$2
				if [ ${keepIntermediateResultsOrNot} != 0 ]; then
				    keepIntermediateResultsOrNot=1
				fi
				shift 2;;

			-d) 	
				intermediateResultsDir=$2
				shift 2;;

                        -ref)
                                inrefdir=$2;
                                shift 2;;                       # user-input template directory

			-runmode)   
				runmode=$2  
				shift 2;;

			-regqueue)
				regqueue=$2
				shift 2;;

			-fusequeue)
				fusequeue=$2
				shift 2;;
		
			-ssqueue)
				ssqueue=$2
				shift 2;;

			-*) 
				echo "ERROR: no such option $1";
				help;;

			 *) 
				break;;
		esac
	done

	if [ -z ${input} ]||[ -z ${ssoutput} ]; then
	    echo ""
	    echo "ERROR: input and output are required. See help ($0 -h)."
	    echo ""
	    exit 1
	fi
}


# --------------------------------
checkdependencies()
{
	echo -e "\nLoading software dependencies..."
        for program in dramms flirt bet2 fslinfo fslmerge fslsplit fslmaths seg_LabFusion nifti1_test 3dSkullStrip 3dcalc 3dmerge 3dmask_tool N4 bse mri_watershed c3d runROBEX.sh 3dZeropad 3dAutobox; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
			echo -e "\nThe dependencies include: dramms (1.4.3); fslinfo, flirt, fslmerge, fslsplit, bet2 and fslmaths (FSL 5.0.9); seg_LabFusion (NiftySeg, 0.9.4); nifti1_test, 3dSkullStrip, 3dmerge, 3dmask_tool and 3dcalc (AFNI 16.2.06); N4; bse (BrainSuite16a); mri_watershed (FreeSurfer); c3d (ITK-SNAP 0.8.2); 3dZeropad (afni program for padding image), 3dAutobox (afni program to detect the bounding box) and runROBEX.sh (ROBEX)\n"
                        exit 1
		else
			echo "    ${program}:   ${p}"
                fi
        done
        # make sure dramms/lib is also in the path
        p=`which dramms`
        pp=`readlink -f ${p}/../lib`
        export PATH=$PATH:${pp}
	p=`which CalculateImageSimilarity 2>/dev/null`
	if [ -z "${p}" ]; then
		echo -e "\nError: please check if CalculateImageSimilarity is in ${pp}\n"
		exit 1
	else
		echo "    CalculateImageSimilarity:   ${p}"
	fi

	# make sure picasso/lib is also in the path
	p=`dirname ${0}`
	pp=`readlink -f ${p}/../lib`
	export PATH=$PATH:${pp}
	p=`which normalizeFOV.sh 2>/dev/null`
	if [ -z "${p}" ]; then
		echo -e "\nError: please check if normalizeFOV.sh is in ${pp}\n"
		exit 1
	else
		echo "    normalizeFOV.sh:   ${pp}"
	fi
	p=`which LabelFusionWithAtlasSelection.sh 2>/dev/null`
        if [ -z "${p}" ]; then
                echo -e "\nError: please check if LabelFusionWithAtlasSelection.sh is in ${pp}\n"
                exit 1
        else
                echo "    LabelFusionWithAtlasSelection.sh:   ${pp}"
        fi

	# make sure freesurfer environment is set (in order to use mri_watershed)
	p=`which mri_watershed`
	FREESURFER_HOME=`readlink -f ${p} | rev | cut -d/ -f3- | rev`
	source ${FREESURFER_HOME}/FreeSurferEnv.sh
	

	echo -e "Depending software tools have been successfully loaded.\n"
}


#-------------------------------
trap cleanup EXIT INT QUIT TERM
cleanup()
{
    if [ -n ${tmpdir} ] && [ "${keepIntermediateResultsOrNot}" == 0 ]; then
        \rm -rf ${tmpdir}
    fi	
}




##############################
##############################
## Defining Functions Ends
##############################
##############################



##############################
##############################
## Main Script
##############################
##############################

# help if needed
if [ $# -lt 1 ]
then
	help
fi


### Timestamps
startTime=`date +%F-%H:%M:%S`
startTimeStamp=`date +%s`

### checkcluster
pbscluster=0
mosixcluster=0
if [ ! -z `which pbsubmit 2>/dev/null` ]; then
	pbscluster=1;
elif [ ! -z `which mosbatch 2>/dev/null` ]; then
	mosixcluster=1;
	mos_fnndsc="mosbatch -q -j10.36.131.42,10.36.131.45,10.36.131.46,10.36.131.48,10.36.133.204,10.36.133.203,10.36.133.205" #see http://durban.tch.harvard.edu/wiki/index.php/Cluster_Overview#Total for the names of the nodes in these ip addresses
elif [ ! -z `which sbatch 2>/dev/null` ]; then
	slurmcluster=1;
fi

### set the default runmode
if [ ${pbscluster} == 1 ]||[ ${mosixcluster} == 1 ]||[ ${slurmcluster} == 1 ]; then
        runmode=1 # default when cluster exists
else
        runmode=2 # default when cluster does not exist
fi

# other default parameters
usingPediatricAtlas=0          # using adult T1 atlases by default
usingT1GadAtlas=0	       # using adult T1 atlases by default
othermodality=0		       # to skull strip T1w MRI (0), otherwise other modalities such as T2w, MRPAGE (1)
biascorrection=1               # default: bias correction
keepIntermediateResultsOrNot=0 # 0: do not keep intermediate results; 1: keep
intermediateResultsDir=""
regqueue=r_max200
fusequeue=p50
ssqueue=p30
#topdir=`dirname $0`
#topdir=`readlink -f ${topdir}/../`
topdir=`readlink -f $0 | rev | cut -d/ -f2- | rev`
topdir=`readlink -f ${topdir}/../`  # modified on 9/4/2017 to deal with symbolic link
inrefdir=""
outimage=""

# executibles in the lib
normalizeFOV=${topdir}/lib/normalizeFOV.sh
labelfuser=${topdir}/lib/LabelFusionWithAtlasSelection.sh

### Reading the arguments
parse $*
getFilePrefix ${input}
input_prefix=${prefix}
if [ -z "${inrefdir}" ]; then
	if [ ${usingPediatricAtlas} -eq 1 ]&&[ ${usingT1GadAtlas} -eq 0 ]; then
        	atlasdir=${topdir}/atlases/pediatricT1w/
	        suffix=Ped_Atlases_bc${biascorrection}
	elif [ ${usingT1GadAtlas} -eq 1 ]&&[ ${usingPediatricAtlas} -eq 0 ]; then
        	atlasdir=${topdir}/atlases/adultT1w_withcontrast/
	        suffix=Adult_T1Gad_Atlases_bc${biascorrection}
	elif [ ${usingT1GadAtlas} -eq 1 ]&&[ ${usingPediatricAtlas} -eq 1 ]; then
        	echo -e "\nplease use '-ped' OR '-gad', not both. Program exits.\n"
	        exit 1
	else
        	atlasdir=${topdir}/atlases/adultT1w/
	        suffix=Adult_Atlases_bc${biascorrection}
	fi
else
        atlasdir=${inrefdir}
	suffix=UserDefinedAtlases_bc${biascorrection}
fi
# check atlasdir contents
nImages=`ls ${atlasdir}/Template*_padded.nii.gz 2>/dev/null |wc -l`
nLabels=`ls ${atlasdir}/Template*_padded_label.nii.gz 2>/dev/null |wc -l`
nFOVs=`ls ${atlasdir}/Template*_padded_normalizedFOVmask.nii.gz 2>/dev/null |wc -l`
if [ -z "${nImages}" ]; then nImages=0; fi
if [ -z "${nLabels}" ]; then nLabels=0; fi
if [ -z "${nFOVs}" ];   then nFOVs=0;   fi
if [ ${nImages} != ${nLabels} ]||[ ${nLabels} != ${nFOVs} ]||[ ${nImages} -lt 6 ]; then
        echo -e "\nError: the atlas directory (${atlasdir}) should have images named Template*_padded.nii.gz, labels (brain masks) named as Template*_padded_label.nii.gz and FOV normalization masks named Template*_padded_normalizedFOVmask.nii.gz. The numbers of images, labels and FOV masks should be the same and at least 6, that is, at least 6 templates (atlases) are needed. Program exits.\n"
        exit 1
fi


if [ ${runmode} != 1 ]&&[ ${runmode} != 2 ]&&[ ${runmode} != 3 ]; then
        echo -e "\nError: the -runmode option must be either 1, 2 or 3. Program exits. Please reset."
        exit 1
fi

echo "--------------------------------------------------------------------"
echo "--> input image  = ${input}"
echo ""
echo "--> input_prefix = ${input_prefix}"
echo ""
echo "--> output       = ${ssoutput}"
echo ""
echo "--> using ${suffix}"
echo "--> atlas dir = ${atlasdir}"
echo "--------------------------------------------------------------------"


input_fullpath=`readlink -f ${input}`
numTemplates=${nImages}
numAvailable=`ls ${atlasdir}/*label.nii.gz |wc -l`
if [ ${numAvailable} -lt ${numTemplates} ]; then
	numTemplates=${numAvailable}
fi


######
# check if input image exists and if it is in the right format
    if [ ! -f ${input_fullpath} ]; then
          echo ""
          echo "error: the input image (${input_fullpath}) does not exist."
          echo "program terminated."
          echo ""
          exit 1
    fi
    extension4=${input_fullpath: -4}
    extension7=${input_fullpath: -7}
    if [ "${extension4}" != ".nii" ]&&[ "${extension4}" != ".hdr" ]&&[ "${extension4}" != ".img" ]&&[ "${extension7}" != ".nii.gz" ]; then
            echo ""
            echo "error: the input image must be in ANALYZE (.hdr, .img) or NIFTI (.nii, .nii.gz) format."
            echo "program terminated."
            echo ""
            exit 1
    fi
# check if the output image is in the right format
    extension4=${ssoutput: -4}
    extension7=${ssoutput: -7}
    if [ "${extension4}" != ".nii" ]&&[ "${extension4}" != ".hdr" ]&&[ "${extension4}" != ".img" ]&&[ "${extension7}" != ".nii.gz" ]; then
            echo ""
            echo "error: the output image must be in ANALYZE (.hdr, .img) or NIFTI (.nii, .nii.gz) format."
            echo "program terminated."
            echo ""
            exit 1
    fi


#-------------------------------
### preparation 1: load dependencies
checkdependencies


#-------------------------------
### preparation 2: create temporary directories
currentDIR=`pwd`
destDIR=`dirname ${ssoutput}`
echo keepIntermediateResultsOrNot=${keepIntermediateResultsOrNot}
echo intermediateResultsDir=${intermediateResultsDir}
if [ ${keepIntermediateResultsOrNot} -eq 1 ]; then
    if [[ -n "${intermediateResultsDir}" ]]; then
        tmpdir="${intermediateResultsDir}"
        [[ -d "${tmpdir}" ]] || mkdir -p "${tmpdir}"
    else
	tmpdir=`mktemp -d "${destDIR}/SkullStripping-${input_prefix}-${suffix}-XXXXXX"`
	mkdir -p ${tmpdir}
    fi
elif [[ -n "${intermediateResultsDir}" ]]; then
    tmpdir=`mktemp -d "${intermediateResultsDir}/SkullStripping-${input_prefix}-${suffix}-XXXXXX"`
elif [[ -n "${TMPDIR}" ]]; then
    tmpdir=`mktemp -d "${TMPDIR}/SkullStripping-${input_prefix}-${suffix}-XXXXXX"`
else
    tmpdir=`mktemp -d ${destDIR}/SkullStripping-${input_prefix}-${suffix}-XXXXXX`
fi
sysflag=`echo $?`
if [ ${sysflag} -ne 0 ]; then
    echo
    error "Failed to make a temporary directory!"
    echo
    exit 1
fi
echo destDIR=${destDIR}
echo "intermediate results will be written into tmpdir=${tmpdir}"
chmod u+x ${destDIR}
chmod g+x ${destDIR}
# make subdirectories for different intermediate results
fromAtlasesDIR=${tmpdir}/fromAtlases
fromSkullStripperDIR=${tmpdir}/fromSkullStrippers
consensus1DIR=${tmpdir}/consensus1
consensus2DIR=${tmpdir}/consensus2
mkdir -p ${fromAtlasesDIR}
chmod u+x ${fromAtlasesDIR}
chmod g+x ${fromAtlasesDIR}
sysflag=`echo $?`
[ ${sysflag} -eq 0 ] || {
    echo
    error "Failed to create directory ${fromAtlasesDIR}!"
    echo
    exit 1
}
mkdir -p ${fromSkullStripperDIR}
chmod u+x ${fromSkullStripperDIR}
chmod g+x ${fromSkullStripperDIR}
sysflag=`echo $?`
[ ${sysflag} -eq 0 ] || {
    echo
    error "Failed to create directory ${fromSkullStripperDIR}!"
    echo
    exit 1
}
mkdir -p ${consensus1DIR}
chmod u+x ${consensus1DIR}
chmod g+x ${consensus1DIR}
sysflag=`echo $?`
[ ${sysflag} -eq 0 ] || {
    echo
    error "Failed to create directory ${consensus1DIR}!"
    echo
    exit 1
}
mkdir -p ${consensus2DIR}
chmod u+x ${consensus2DIR}
chmod g+x ${consensus2DIR}
sysflag=`echo $?`
[ ${sysflag} -eq 0 ] || {
    echo
    error "Failed to create directory ${consensus2DIR}!"
    echo
    exit 1
}
echo "created temporary directory here:"
echo "     ${tmpdir}"


#-------------------------------
### preparation 3. padding image if necessary
# added on 2017/8/29, because we found in Abbott studies that some images has a too small bounding box, so atlas-to-target registration often has bigger errors in boundary of the brain since it is so close to the image bounding box
xSize=`fslinfo ${input_fullpath} | grep dim1 | sed -n 1p | awk '{ print $2 }'`
ySize=`fslinfo ${input_fullpath} | grep dim2 | sed -n 1p | awk '{ print $2 }'`
zSize=`fslinfo ${input_fullpath} | grep dim3 | sed -n 1p | awk '{ print $2 }'`
inputimage_bbox=${tmpdir}/inputimage_autobbox.nii.gz
if [ -f ${inputimage_bbox} ]; then rm ${inputimage_bbox}; fi
msg=`3dAutobox -prefix ${inputimage_bbox} -input ${input_fullpath} 2>&1 | grep "Auto bbox"`
xmarginN=`echo $msg | cut -d= -f2 | cut -d. -f1`;                 xmarginN=$(( ${xmarginN} - 1 ))
xmarginP=`echo $msg | cut -d= -f2 | cut -d. -f3 | cut -d' ' -f1`; xmarginP=$(( ${xSize} - ${xmarginP} ))
ymarginN=`echo $msg | cut -d= -f3 | cut -d. -f1`;                 ymarginN=$(( ${ymarginN} - 1 ))
ymarginP=`echo $msg | cut -d= -f3 | cut -d. -f3 | cut -d' ' -f1`; ymarginP=$(( ${ySize} - ${ymarginP} ))
zmarginN=`echo $msg | cut -d= -f4 | cut -d. -f1`;                 zmarginN=$(( ${zmarginN} - 1 ))
zmarginP=`echo $msg | cut -d= -f4 | cut -d. -f3 | cut -d' ' -f1`; zmarginP=$(( ${zSize} - ${zmarginP} ))
xrange=`fslinfo ${inputimage_bbox} | grep dim1 | sed -n 1p | awk '{ print $2 }'`
yrange=`fslinfo ${inputimage_bbox} | grep dim2 | sed -n 1p | awk '{ print $2 }'`
zrange=`fslinfo ${inputimage_bbox} | grep dim3 | sed -n 1p | awk '{ print $2 }'`
requiredxmargin=$(( ${xrange} / 6 ))
requiredymargin=$(( ${yrange} / 6 ))
requiredzmargin=$(( ${zrange} / 6 ))
xpadN=$(( ${requiredxmargin} - ${xmarginN} )); if [ ${xpadN} -lt 0 ]; then xpadN=0; fi
xpadP=$(( ${requiredxmargin} - ${xmarginP} )); if [ ${xpadP} -lt 0 ]; then xpadP=0; fi
ypadN=$(( ${requiredymargin} - ${ymarginN} )); if [ ${ypadN} -lt 0 ]; then ypadN=0; fi
ypadP=$(( ${requiredymargin} - ${ymarginP} )); if [ ${ypadP} -lt 0 ]; then ypadP=0; fi
zpadN=$(( ${requiredzmargin} - ${zmarginN} )); if [ ${zpadN} -lt 0 ]; then zpadN=0; fi
zpadP=$(( ${requiredzmargin} - ${zmarginP} )); if [ ${zpadP} -lt 0 ]; then zpadP=0; fi
xrangepadded=$(( ${xSize} + ${xpadN} + ${xpadP} ))
yrangepadded=$(( ${ySize} + ${ypadN} + ${ypadP} ))
zrangepadded=$(( ${zSize} + ${zpadN} + ${zpadP} ))
inputimage_padded=${tmpdir}/inputimage_padded.nii.gz
xorient=`3dinfo ${input_fullpath} 2>&1 | grep "first  (x)" | awk '{ print $4 }'`
xorientN=`echo ${xorient} | cut -c1`
xorientP=`echo ${xorient} | cut -d'-' -f3 | cut -c1`
yorient=`3dinfo ${input_fullpath} 2>&1 | grep "second (y)" | awk '{ print $4 }'`
yorientN=`echo ${yorient} | cut -c1`
yorientP=`echo ${yorient} | cut -d'-' -f3 | cut -c1`
zorient=`3dinfo ${input_fullpath} 2>&1 | grep "third  (z)" | awk '{ print $4 }'`
zorientN=`echo ${zorient} | cut -c1`
zorientP=`echo ${zorient} | cut -d'-' -f3 | cut -c1`
if [ ${xpadN} != 0 ]||[ ${xpadP} != 0 ]||[ ${ypadN} != 0 ]||[ ${ypadP} != 0 ]||[ ${zpadN} != 0 ]||[ ${zpadP} != 0 ]; then
	cmd="3dZeropad -${xorientN} ${xpadN} -${xorientP} ${xpadP} -${yorientN} ${ypadN} -${yorientP} ${ypadP} -${zorientN} ${zpadN} -${zorientP} ${zpadP} -prefix ${inputimage_padded} ${input_fullpath}"
	echo -e "\n------------padding-----------\n${cmd}\n"
	if [ -f ${inputimage_padded} ]; then \rm ${inputimage_padded}; fi
	${cmd}
else
	\cp ${input_fullpath} ${inputimage_padded}
fi


#-------------------------------
### preparation 4. bias correction
# first (added on Aug 30, 2014), bias correction
if [ ${biascorrection} == 1 ]; then
	bcimage=${tmpdir}/inputimage_padded_N4bc.nii.gz
	bcimagebsname=`basename ${bcimage%.nii.gz}`
	if [ ! -s ${bcimage} ]; then
	    cmd="N4 -i ${inputimage_padded} -o ${bcimage}"
	    echo " ==> bias correction using N4"
	    echo ${cmd}
	    ${cmd}
	fi
	if [ ! -s ${bcimage} ]; then
		\cp ${input_padded} ${bcimage}
		echo -e "\nBias correction did not finished successfully. So we use the original image with potential padding (${input_padded})."
	else
		echo -e "\nBias correction finished successfully: ${bcimage}\n"
	fi
	# added on 2019/8/15 (Michaela found an image with ASR orientation, but N4 forced it to be RAI. so we need to change back)
	qori=`fslorient -getqform ${inputimage_padded}`
	sori=`fslorient -getsform ${inputimage_padded}`
	fslorient -setqform $qori ${bcimage}
	fslorient -setsform $sori ${bcimage}
	# end of added on 2019/8/15
	input_fullpath=${bcimage}  # from now on, using the N4-bc image as the input
	getFilePrefix ${input_fullpath}
	input_prefix=${prefix}
fi


#-------------------------------
### preparation 5. FOV normalization (forward normalization --- use FOV of templates to normalize FOV of input images)
FOVnormalizationDIR=${fromAtlasesDIR}/1FOVnormalization_${bcimagebsname}
mkdir -p ${FOVnormalizationDIR} 2>/dev/null
FOVimage=${FOVnormalizationDIR}/${bcimagebsname}_FOVnormalized.nii.gz
cmd="${normalizeFOV} -in ${input_fullpath} -out ${FOVimage} -debug -runmode ${runmode} -ref ${atlasdir}"
if [ ! -f ${FOVimage} ]; then
	echo -e "\nNormalize FOV:\n\n${cmd}\n"
	${cmd}
else
	echo -e "\nNormalize FOV:\nFOV normalized image already exists (${FOVimage})."
fi
mywait "${FOVimage}"
sleep 5
echo -e "\nForward FOV normalization done.\n\n"


#-------------------------------
### preparation 6. use FOV normalization information for two things (added 8/2/2016)
#                  6a. backward FOV normalization (use FOV of input image to normalize FOV of templates). In case the input image is only a strip of the brain, we need to use only a strip of the brain in the atlases for multi-atlas skull stripping.
#                  6b. initial selection of atlases (6-9 out of 15 atlases that are very similar to the input image after forward and backward FOV normalization).
#
FOVimagepre=`basename ${FOVimage}`
FOVimagepre=${FOVimagepre%.nii.gz}
FOV_int_dir=`ls -d ${FOVnormalizationDIR}/intermediateresults-${FOVimagepre}*/ -th | sed -n 1p 2>/dev/null`
bwdFOV_dir=`readlink -f ${fromAtlasesDIR}/atlases_backwardFOVnormalized`
if [ ! -d ${bwdFOV_dir} ]; then
	mkdir -p ${bwdFOV_dir}
fi	
FOVmask=${FOVnormalizationDIR}/${FOVimagepre}_normalizedFOVmask.nii.gz
similarityFile=${FOV_int_dir}/similarity_allTemplates_to_${bcimagebsname}_iter2.txt
if [ -z "${FOV_int_dir}" ]; then
	echo -e "\nError: FOV normalization failed. FOV intermediate directory (intermediateresults-${FOVimagepre}*) does not exist in ${FOVnormalizationDIR}. Program exits.\n"
	exit 1
fi
if [ ! -f ${FOVmask} ]; then
	echo -e "\nError: FOV normalization failed. FOV mask (${FOVmask}) does not exist. Program exits.\n"
	exit 1
fi
if [ ! -f ${similarityFile} ]; then
	echo -e "\nError: FOV normalization failed. FOV intermediate results (${similarityFile}) does not exist. Program exits.\n"
fi
simthreshold=`cat ${similarityFile} | grep "threshold=" | sed -n 1p | cut -d'=' -f3`
if [ -z "${simthreshold}" ]; then simthreshold=0; fi
similarityFileSorted=${similarityFile%.txt}_sorted.txt
sed -i '/^$/d' ${similarityFile} # remove empty rows
sort -n -r -k2 ${similarityFile} |uniq > ${similarityFileSorted}
readmefile=${bwdFOV_dir}/readme.txt
\rm ${readmefile} 2>/dev/null
counter=1;  # counter for selected atlases
threflag=0
while [ ${threflag} == 0 ]; do
	for linenum in {1..7}; do # use at least 5 and up to 7 atlases
		templateID=`cat ${similarityFileSorted} | sed -n ${linenum}p | awk '{ print $1 }'`
		thistemplatewrped=${FOV_int_dir}/reg_${templateID}_to_${bcimagebsname}.nii.gz
		thistemplateFOVwrped=${FOV_int_dir}/wrpedFOVmask_${templateID}_to_${bcimagebsname}.nii.gz # files already existed

		thistemplatelabelname=${atlasdir}/${templateID}_padded_label.nii.gz
		mat=${FOV_int_dir}/mat_${templateID}_to_${bcimagebsname}.mat
		thistemplatelabelwrped=${FOV_int_dir}/wrpedlabel_${templateID}_to_${bcimagebsname}.nii.gz
		dramms-warp ${thistemplatelabelname} ${mat} ${thistemplatelabelwrped} -t ${FOVimage} -n # warp template brain label into the input image space

		sim=`cat ${similarityFileSorted} | sed -n ${linenum}p | awk '{ print $2 }'`
		if [ `echo ${sim} '>=' ${simthreshold} | bc -l` == 1 ]; then
			echo "${templateID} sim=${sim}, simthreshold=${simthreshold} => Template${counter}"
			# use this template for subsequent atlas
			newtemplateimagename=${bwdFOV_dir}/Template${counter}.nii.gz
			newtemplatelabelname=${bwdFOV_dir}/Template${counter}_label.nii.gz
			fslmaths ${thistemplatewrped} -mul ${thistemplateFOVwrped} -mul ${FOVmask} ${newtemplateimagename}
			fslmaths ${thistemplatelabelwrped} -mul ${thistemplateFOVwrped} -mul ${FOVmask} ${newtemplatelabelname}	
			echo "${templateID} in ${atlasdir} => Template${counter} in ${FOV_int_dir}" >> ${readmefile}
			counter=$(( ${counter} + 1 ))
		fi
	done
	if [ ${counter} -lt 5 ]; then
		simthreshold=`echo "${simthreshold} * 0.95" | bc -l`
		\rm ${readmefile}
		counter=1
	else
		threflag=1
	fi
done
numAtlasesToBeUsed=$(( ${counter} - 1 ))
if [ ${numAtlasesToBeUsed} -lt 7 ]; then
	echo -e "\033[1;31mWarning: only ${numAtlasesToBeUsed} atlases are to be used for multi-atlas skull stripping (this number may be too few)\033[0m" 
fi
echo -e "\nBackward FOV normalization and initial atlas selection done. ${numAtlasesToBeUsed} templates will be used for multi-atlas skull stripping. They are in ${bwdFOV_dir}\n\n"
#\rm -rf ${FOVnormalizationDIR}/intermediateresults-${FOVimagepre}*

#--------------
# the following was to backward FOV normalize based on inversely transforming the FOV of the input image to the atlas space. I chose not to do this, but the above (on 8/3/2016), for two reasons: since we have already calculated affine transformations from each atlas to the input image, and have found the most accurate transformations among all atlases, (a) we should use such information to select atlas, and select atlases in the input image space; (b) in subsequent multi-atlas skull stripping, we don't need to re-do affine registration part, just run dramms with the deformable component.)
#-------------
#for mat in ${FOV_int_dir}/mat_Template*_to_inputimage_N4bc.mat; do
#	matpre=`basename ${mat}`
#	templateID=`echo ${matpre} |cut -d_ -f2`
#	thistemplate=${atlasdir}/${templateID}_padded.nii.gz
#	template_label_bsname=${templateID}_padded_label.nii.gz # files already existed
#	FOVforthistemplate=${atlasdir}/${templateID}_padded_normalizedFOVmask.nii.gz
#
#	imat=${FOV_int_dir}/inv${matpre}
#	wrpFOVforthistemplate=${FOV_int_dir}/backwardFOVnormalization_${templateID}_padded_normalizedFOVmask.nii.gz
#        template_bwdFOVnormalized=${bwdFOV_dir}/${templateID}_padded_normalizedFOV.nii.gz # files to be generated
#
#	if [ ! -f ${template_bwdFOVnormalized} ]||[ ! -f ${bwdFOV_dir}/${template_label_bsname} ]; then
#		echo -e "backward FOV normalization for ${templateID} in ${atlasdir}, saved into ${bwdFOV_dir}\n"
#		dramms-defop -i ${mat} ${imat} # invert affine transform
#		dramms-warp ${FOVmask} ${imat} ${wrpFOVforthistemplate} -t ${thistemplate} -n # using inverted affine transformation to map normalized FOV from input image to the template
#		fslmaths ${thistemplate} -mul ${wrpFOVforthistemplate} -mul ${FOVforthistemplate} ${template_bwdFOVnormalized} # bwd normalize template's FOV
#		fslmaths ${atlasdir}/${template_label_bsname} -mul ${wrpFOVforthistemplate} -mul ${FOVforthistemplate} ${bwdFOV_dir}/${template_label_bsname}
#	fi
#done
#echo -e "\nBackward FOV normalization done.\n\n"



#-------------------------------
### preparation 7. submit multi-atlas registrations
[ "$(ls -A ${consensus1DIR})" ] && \rm ${consensus1DIR}/*
echo -e "\nInitialization: multi-atlas skull stripping" # this is a modified version of MASS [Doshi, et al, Aca Rad, 2013]
#cmd="multiatlas_skullstripping_modified.sh -in ${input_fullpath} -out ${tmpdir}/brainmask_${prefix}_fromAtlases.nii.gz -ref ${atlasdir} -dest ${fromAtlasesDIR} -regs ${numTemplates}"
#echo ${cmd}
#if [ ! -f ${tmpdir}/brainmask_${prefix}_fromAtlases.nii.gz ]; then
#	${cmd}
#fi
multiatlasdir=${fromAtlasesDIR}/2brainmask_bydeformableregistration_${input_prefix}
if [ ! -d ${multiatlasdir} ]; then
	mkdir -p ${multiatlasdir}
fi
for (( idx=1; idx<=${numAtlasesToBeUsed}; idx++ )); do
	thistemplateimage=${bwdFOV_dir}/Template${idx}.nii.gz
	thistemplatelabel=${bwdFOV_dir}/Template${idx}_label.nii.gz # files that already existed
	
	regimage=${multiatlasdir}/reg_Template${idx}_to_${input_prefix}.nii.gz
	def=${multiatlasdir}/def_Template${idx}_to_${input_prefix}.nii.gz
	wrplabel=${multiatlasdir}/wrpedlabel_Template${idx}_to_${input_prefix}.nii.gz # files to be generated

	if [ ${othermodality} == 0 ]; then
		cmd="dramms -S ${thistemplateimage} -T ${FOVimage} -O ${regimage} -W ${wrplabel} -L ${thistemplatelabel} -r 0 -a 0 -v -v" # > ${multiatlasdir}/log_Template${idx}_to_${input_prefix}.txt" # command
	else
		cmd="dramms -S ${thistemplateimage} -T ${FOVimage} -O ${regimage} -W ${wrplabel} -L ${thistemplatelabel} -r 0 -a 0 -g 0.4 -w 1 -v -v" # use cc of attribute vector as the similarity metric
	fi
	echo -e "\n${cmd}\n"

	if [ ! -f ${regimage} ]||[ ! -f ${wrplabel} ]; then
		if [ ${runmode} == 1 ]; then  # run parallelly in the cluster
                        if [ ${pbscluster} == 1 ]; then
                                pbsubmit -n 2 -q ${regqueue} -c "${cmd}"
                        elif [ ${mosixcluster} == 1 ]; then
                                ${mos_fnndsc} -m20000 ${cmd} &				
				sleep 5
			elif [ ${slurmcluster} == 1 ]; then
				cmdfile=${multiatlasdir}/cmd_reg_Template${idx}_to_${input_prefix}.sh
				echo -e '#!/bin/bash\n' "srun ${cmd}" > ${cmdfile}
				sbatch --partition=bch-largemem --mem=17000MB ${cmdfile}				
                        fi
		elif [ ${runmode} == 2 ]; then # run locally but all registrations in parallel
			echo "run in the current machine, in parallel"
			${cmd} > ${multiatlasdir}/log_Template${idx}_to_${input_prefix}.txt &
		else # run locally but registrations one by one
			echo "run in the current machine, sequentially"
			${cmd} > ${multiatlasdir}/log_Template${idx}_to_${input_prefix}.txt
		fi	
	fi
done

#-------------------------------
### preparation 8. generate brain masks using skull-strippers with varying parameters

# now using skull strippers at different parameter values
skullstrippermasks=""
skullstrippermaskstxt=${fromSkullStripperDIR}/masksfromskullstrippers.txt
if [ -f ${skullstrippermaskstxt} ]; then
    \rm ${skullstrippermaskstxt}
fi

# get the number of slices, program 3dSkullStrip will not run if numslices<16
numslices=`fslinfo ${FOVimage} | grep dim3 | sed -n 1p | awk '{ print $2 }'`

if [ ${numslices} -gt 16 ]; then
# ----3dSkullStrip
# totally 45 parameter settings
echo ""
echo ---------------
echo ""
echo "running AFNI's 3dSkullStrip with varying parameters..."
mkdir -p ${fromSkullStripperDIR}/3dSkullStrip
chmod u+x ${fromSkullStripperDIR}/3dSkullStrip
chmod g+x ${fromSkullStripperDIR}/3dSkullStrip
for sf in $(seq 0.4 0.05 0.8); do
#for sf in $(seq 0.4 0.2 0.8)
    for ef in $(seq 0.05 0.025 0.15); do
    #for ef in $(seq 0.05 0.05 0.15)
	  echo ""
	  echo "3dSkullStrip with varying parameters: sf=${sf}, ef=${ef}"
	  afni3dSkullStripmask=${fromSkullStripperDIR}/3dSkullStrip/3dSkullStrip_${input_prefix}_sf${sf}_ef${ef}_mask.nii.gz
	  if [ ! -f ${afni3dSkullStripmask} ]; then
		#cmd="${topdir}/lib/generateBrainMask_3dSkullStrip.sh -shrink_fac ${sf} -exp_frac ${ef} -in ${input_fullpath} -out ${afni3dSkullStripmask}"
		cmd="${topdir}/lib/generateBrainMask_3dSkullStrip.sh -shrink_fac ${sf} -exp_frac ${ef} -in ${FOVimage} -out ${afni3dSkullStripmask}" # on 8/5/2016, we start to feed skull strippers with FOV normalized images instead of the raw N4bc image
		if [ "${runmode}" == "1" ]; then
                        if [ ${pbscluster} == 1 ]; then
                                pbsubmit -q ${ssqueue} -c "${cmd}"
                        elif [ ${mosixcluster} == 1 ]; then
                                ${mos_fnndsc} ${cmd} &
				sleep 5
			elif [ ${slurmcluster} == 1 ]; then
				cmdfile=${fromSkullStripperDIR}/3dSkullStrip/cmd_3dSkullStrip_${input_prefix}_sf${sf}_ef${ef}.sh
				echo -e '#!/bin/bash\n' "srun ${cmd}\n" > ${cmdfile}
				sbatch --partition=fnndsc-compute ${cmdfile}
                        fi
		else
			${cmd}
		fi
	  fi
	  echo ${afni3dSkullStripmask} >> ${skullstrippermaskstxt}
    done
done
fi

# ----robex
# totally 1 parameter settings
echo ""
echo ---------------
echo ""
echo "running ROBEX..."
mkdir -p ${fromSkullStripperDIR}/robex 2>/dev/null
chmod g+x ${fromSkullStripperDIR}/robex 2>/dev/null
chmod u+x ${fromSkullStripperDIR}/robex 2>/dev/null
robexmask=${fromSkullStripperDIR}/robex/robex_${input_prefix}_mask.nii.gz
if [ ! -f ${robexmask} ]; then
	#cmd="${topdir}/lib/generateBrainMask_ROBEX.sh -in ${input_fullpath} -out ${robexmask}"
	robexcmd="${topdir}/lib/generateBrainMask_ROBEX.sh -in ${FOVimage} -out ${robexmask}" # starting 8/5/2016, we feed the skull strippers with FOV normalized image instead of the raw image (N4 bc-ed)
	echo -e "\n${robexcmd}\n"
	if [ "${runmode}" == "1" ]; then
                if [ ${pbscluster} == 1 ]; then
                        pbsubmit -q ${ssqueue} -c "${robexcmd}"
                elif [ ${mosixcluster} == 1 ]; then
                        ${mos_fnndsc} ${robexcmd} &
			sleep 5
		elif [ ${slurmcluster} == 1 ]; then
                        cmdfile=${fromSkullStripperDIR}/robex/cmd_robex_${input_prefix}.sh
                        echo -e '#!/bin/bash\n' "srun ${robexcmd}\n" > ${cmdfile}
                        sbatch --partition=fnndsc-compute ${cmdfile}
                fi
	else
		${robexcmd}
	fi
fi
echo ${robexmask} >> ${skullstrippermaskstxt}


# ----hwa
# totally 32 parameter settings
echo ""
echo ---------------
echo ""
echo "running FreeSurfer's HWA with varying parameters..."
mkdir -p ${fromSkullStripperDIR}/hwa 2>/dev/null
chmod u+x ${fromSkullStripperDIR}/hwa 2>/dev/null
chmod g+x ${fromSkullStripperDIR}/hwa 2>/dev/null
for atlasornot in 0 1; do
    for h in $(seq 10 2 40); do
	  echo ""
	  echo "hwa with varying parameters: h=${h}, atlasornot=${atlasornot}"
	  hwamask=${fromSkullStripperDIR}/hwa/hwa_${input_prefix}_h${h}_atlas${atlasornot}_mask.nii.gz
	  if [ ! -f ${hwamask} ]; then
		#cmd="${topdir}/lib/generateBrainMask_HWA.sh -height ${h} -atlas ${atlasornot} -in ${input_fullpath} -out ${hwamask} 2> /dev/null"
		cmd="${topdir}/lib/generateBrainMask_HWA.sh -height ${h} -atlas ${atlasornot} -in ${FOVimage} -out ${hwamask} 2> /dev/null" # starting from 8/5/2016, we feed the skull strippers with FOV normalized image instead of raw image N4 (bc-ed).
		echo -e "\n${cmd}\n"
		if [ "${runmode}" == "1" ]; then
                        if [ ${pbscluster} == 1 ]; then
                                pbsubmit -q ${ssqueue} -c "${cmd}"
                        elif [ ${mosixcluster} == 1 ]; then
                                ${mos_fnndsc} ${cmd} &
				sleep 5
			elif [ ${slurmcluster} == 1 ]; then
                                cmdfile=${fromSkullStripperDIR}/hwa/cmd_hwa_${input_prefix}_h${h}_atlas${atlasornot}.sh
                                echo -e '#!/bin/bash\n' "srun ${cmd}\n" > ${cmdfile}
                                sbatch --partition=fnndsc-compute ${cmdfile}
                        fi
		else
			${cmd}
		fi	  
	  fi
	  echo ${hwamask} >> ${skullstrippermaskstxt}
  done
done


# ----bet2
# totally 169 parameter settings
echo ""
echo ------------
echo ""
echo "running FSL's bet skull stripper with varying parameters..."
mkdir -p ${fromSkullStripperDIR}/bet2 2>/dev/null
chmod u+x ${fromSkullStripperDIR}/bet2 2>/dev/null
chmod g+x ${fromSkullStripperDIR}/bet2 2>/dev/null
for f in $(seq 0.2 0.05 0.8); do
#for f in $(seq 0.25 0.25 0.75)
    for g in $(seq -0.3 0.05 0.3); do
    #for g in $(seq -0.2 0.15 0.4)
	  echo ""
	  echo "bet2 with varying parameters: f=${f}, g=${g}"
	  bet2maskpre=${fromSkullStripperDIR}/bet2/bet2_${input_prefix}_f${f}_g${g}
	  bet2mask=${fromSkullStripperDIR}/bet2/bet2_${input_prefix}_f${f}_g${g}_mask.nii.gz
	  if [ ! -f ${bet2mask} ]; then
		#cmd="bet2 ${input_fullpath} ${bet2maskpre} -f ${f} -g ${g} -m -n"
		cmd="bet2 ${FOVimage} ${bet2maskpre} -f ${f} -g ${g} -m -n" # starting 8/5/2016, we feed the skull strippers with FOV normalized image instead of the raw image (N4 bc-ed)
		echo -e "\n${cmd}\n"
		#if [ "${runmode}" == "1" ]; then 
		#	pbsubmit -q p20 -c "${cmd}"
		#else
			${cmd}
		#fi # since each bet2 is very short, we let the main program run all bet2 jobs while other longer jobs (hwa, 3dSkullStrip, robex) are running in the cluster.
	  fi
	  echo ${bet2mask} >> ${skullstrippermaskstxt}
    done
done

# ----bse
# totally 121 parameter settings
echo ""
echo ------------
echo ""
echo "running bse skull stripper with varying parameters..."
mkdir -p ${fromSkullStripperDIR}/bse 2>/dev/null
chmod u+x ${fromSkullStripperDIR}/bse 2>/dev/null
chmod g+x ${fromSkullStripperDIR}/bse 2>/dev/null
for s in $(seq 0.42 0.04 0.82); do
  for d in $(seq 10 5 60); do
	  echo ""
	  echo "bse with varying parameters: s=${s}, d=${d}"
	  bsemask=${fromSkullStripperDIR}/bse/bse_${input_prefix}_s${s}_d${d}_mask.nii.gz
	  if [ ! -f ${bsemask} ]; then
		#cmd="${topdir}/lib/generateBrainMask_bse.sh -in ${input_fullpath} -out ${bsemask} -s ${s} -d ${d}"
		cmd="${topdir}/lib/generateBrainMask_bse.sh -in ${FOVimage} -out ${bsemask} -s ${s} -d ${d}" # starting 8/5/2016, we feed the skull strippers with FOV normalized images instead of raw image (N4 bc-ed)
		echo -e "\n${cmd}\n"
		#if [ "${runmode}" == "1" ]; then 
		#	pbsubmit -q p20 -c "${cmd}"
		#else
			${cmd}
		#fi # since each bse job is very short, we let the main program run all bse jobs while other longer jobs (hwa, 3dSkullStrip, robex) are running in the cluster
	  fi
	  echo ${bsemask} >> ${skullstrippermaskstxt}
  done
done

numCMBs=`cat ${skullstrippermaskstxt} |wc -l`
echo -e "\nAll candidate brain masks (CBMs) have been generated, N=${numCMBs}.\n"




#-------------------------------
### step 1: multi-atlas consensus as initialization

# make sure all atlas-to-target registrations finish
atlasoutput=${fromAtlasesDIR}/brainmask_${prefix}.nii.gz 
brainmask_consensus1=${consensus1DIR}/brainmask_consensus1_final_${prefix}.nii.gz
#mywait ${multiatlasdir}/reg_Template*_to_${input_prefix}.nii.gz ${numAtlasesToBeUsed}
#sleep 5
mywait "${multiatlasdir}/wrpedlabel_Template*_to_${input_prefix}.nii.gz" ${numAtlasesToBeUsed}
sleep 10
warpedmasktxt=${consensus1DIR}/warpedatlases.txt
ls ${fromAtlasesDIR}/2brainmask_bydeformableregistration_${prefix}/wrpedlabel_Template*_to_${input_prefix}.nii.gz > ${warpedmasktxt}
for labelimage in ${multiatlasdir}/wrpedlabel_Template*_to_${input_prefix}.nii.gz; do
        fslmaths ${labelimage} -bin ${labelimage}   # make sure the label is 1
done
echo -e "\n${numAtlasesToBeUsed} atlas-to-target registrations have finished. Now start label fusion.\n"

# generate 4D files
echo -e "\nMerge registered images into a 4D file; merge warped labels into a 4D file"
wrpedimage4D=${multiatlasdir}/4D_regimages_${input_prefix}.nii.gz
wrpedmask4D=${multiatlasdir}/4D_warpedmasks_${input_prefix}.nii.gz
FOVimage4D=${multiatlasdir}/4D_FOVimage_${input_prefix}.nii.gz
repeatedFOVimagename=""
for (( idx=1; idx<=${numAtlasesToBeUsed}; idx++ )); do
	repeatedFOVimagename="${repeatedFOVimagename} ${FOVimage}"
done
while true; do
	\rm ${wrpedmask4D} ${wrpedimage4D} ${FOVimage4D} 2>/dev/null
	fslmerge -t ${wrpedmask4D} ${multiatlasdir}/wrpedlabel_Template*_to_${input_prefix}.nii.gz 
	fslmerge -t ${wrpedimage4D} ${multiatlasdir}/reg_Template*_to_${input_prefix}.nii.gz 
	fslmerge -t ${FOVimage4D} ${repeatedFOVimagename} 
	#mywait ${wrpedmask4D}
	#mywait ${wrpedimage4D}
	#mywait ${FOVimage4D}
	NN1=`fslinfo ${wrpedmask4D} 2>/dev/null | grep dim4 | sed -n 1p | awk '{ print $2 }'`
	NN2=`fslinfo ${wrpedimage4D} 2>/dev/null | grep dim4 | sed -n 1p | awk '{ print $2 }'`
	NN3=`fslinfo ${FOVimage4D} 2>/dev/null | grep dim4 | sed -n 1p | awk '{ print $2 }'`
	if [ ${NN1} == ${NN2} ]&&[ ${NN2} == ${NN3} ]; then
		break;   # to make sure 4D files have the same number of 3D blocks
	fi
done
fslmaths ${wrpedmask4D} -bin ${wrpedmask4D}
echo -e "merge finished. Now start label fusion from multi-atlas skull stripping.\n"

# label fusion
cmd="${labelfuser} -inWarpedImage ${wrpedimage4D} -inWarpedLabel ${wrpedmask4D} -inTargetImage ${FOVimage4D} -out ${brainmask_consensus1} -dest ${consensus1DIR} -selectionRule 2"
echo -e "\n${cmd}\n"
if [ ${runmode} == 1 ]; then
        if [ ${pbscluster} == 1 ]; then
                pbsubmit -n 2 -q ${fusequeue} -c "${cmd}"
        elif [ ${mosixcluster} == 1 ]; then
                ${mos_fnndsc} -m15000 ${cmd} &
		sleep 5
	elif [ ${slurmcluster} == 1 ]; then
                cmdfile=${consensus1DIR}/cmd_brainmask_consensus1_${prefix}.sh
                echo -e '#!/bin/bash\n' "srun ${cmd}\n" > ${cmdfile}
                sbatch --partition=bch-largemem --mem=15000MB ${cmdfile}
        fi
else
	${cmd} 
fi
mywait "${brainmask_consensus1}"
sleep 10
echo -e "\nmulti-atlas-skull-stripping (modified version) finished"
echo -e "the result is ${brainmask_consensus1}\n"
consensus1mask=${brainmask_consensus1}
\rm ${wrpedimage4D}
\rm ${wrpedmask4D}
\rm ${FOVimage4D}
#\rm ${multiatlasdir}/reg_Template*_to_${input_prefix}.nii.gz

#-------------------------------
### step 2: selection of atlases and skull stripper parameters and form a second consensus

# check if all masks have been generated
nskullstrippermasks=`cat ${skullstrippermaskstxt} |wc -l`
for n in $(seq 1 1 ${nskullstrippermasks})
  do
    maskfile=`cat ${skullstrippermaskstxt} | sed -n ${n}p`
    if [ ! -f ${maskfile} ]; then
	fslmaths ${FOVimage} -mul 0 ${maskfile}
    fi
done
echo ""
echo "we have generated ${nskullstrippermasks} masks from skull-strippers with varying parameters"
#echo "out of which we will select a set of masks with good overlap with the tentative brain mask from consensus 1"

# select brain masks generated from skull strippers
[ "$(ls -A ${consensus2DIR})" ] && \rm -rf ${consensus2DIR}/*
allmaskstxt=${consensus2DIR}/allmasks.txt
leftmaskstxt=${consensus2DIR}/leftmasks.txt
cat ${skullstrippermaskstxt} ${warpedmasktxt} > ${allmaskstxt} 
#echo ${consensus1mask} >> ${allmaskstxt} # changed on Aug 30, 2014, adding the consensus1mask into candidate mask pool.
nallmasks=`cat ${allmaskstxt} | wc -l`
echo "including warped masks from atlases, we have totally ${nallmasks} masks"
echo "out of which we will select a set of masks with good overlap with the tentative brain mask from consensus 1"
echo "The assumption is that, candidate masks having high overlaps with the tentative mask are likely to have high overlaps with the unseen ground truth mask"
echo ""


iter=1
percent_thre0=0.91 #0.85
if [ ${usingPediatricAtlas} == 1 ]; then
        percent_thre0=0.95
fi
stopping_thre_withinitialization=0.9
if [ ${usingPediatricAtlas} -eq 1 ]; then
	stopping_thre_withinitialization=0.96  # added 12/11/2016, when skull stripping neonatal/pediatric cases, trust the atlas results only unless the robex result is over 0.95 overlap with atlas results.
fi
stopping_thre_withprevious=0.985
n_maxiter=6
thisbrainmask=${consensus2DIR}/brainmask_consensus2_iter0_${prefix}.nii.gz
\cp ${consensus1mask} ${thisbrainmask}


# we trust mass (modified version) results (consensus1), we may trust robex results if it is above 0.9 dice with mass result
trustrobex=0 # initialize
numForegroundVols=`fslstats ${robexmask} -V | awk '{ print $1 }'` # added on 4/10/2018, rerun robex if robex mask is empty up to now
if [ ${numForegroundVols} == 0 ]; then
	echo "re-run robex since the mask from robex is empty for now"
	${robexcmd}
fi
dice_mass_robex=`c3d -verbose ${consensus1mask} ${robexmask} -overlap 1 | grep Dice | awk '{print $4}'`
trustrobex=`echo ${dice_mass_robex} '>' ${stopping_thre_withinitialization} | bc -l`
if [ `echo ${dice_mass_robex} '>' 0.95 |bc -l` == 1 ]; then  # if mass results and robex results highly agree, it's a sign that mass result is great, we should trust this initial mask more
	percent_thre0=0.96
fi
if [ ${usingPediatricAtlas} == 1 ]; then
	trustrobex=0
	percent_thre0=0.96
fi

while [ ${iter} -le ${n_maxiter} ]; do
	# calculate dice of masks and the tentative mask, and then select warped masks.
	max_dice=0
	dicetxt=${consensus2DIR}/dice_allmasks_iter${iter}.txt
	sorteddicetxt=${consensus2DIR}/sorteddice_allmasks_iter${iter}.txt
	#dicetxt=${consensus2DIR}/dice_skullstrippermasks_iter${iter}.txt
	if [ -f ${dicetxt} ]; then
	    \rm ${dicetxt}
	fi
	percent_thre=`echo "scale=2; ${percent_thre0}+0.01*(${iter}-1)" | bc`
	if [ `echo ${percent_thre} '>' 0.98 | bc -l` == 1 ]; then percent_thre=0.98; fi
	lambda1=`echo "scale=2; 0.5+0.05*(${iter}-1)" |bc -l`
	if [ `echo ${lambda1} '>' 1 |bc -l` == 1 ]; then lambda1=1; fi
	lambda2=`echo "scale=2; 1-${lambda1}" |bc -l`

	
	if [ ${iter} == 1 ]; then
		nn=${nallmasks} #${nskullstrippermasks}   # in iter1, only look for best masks skull strippers
	else
		nn=`cat ${leftmaskstxt} |wc -l`             # in other iters, look for masks from either skull strippers or atlases
	fi
	echo -e "\nIn iteration ${iter}, ${nn} candidate brain masks to consider.\n"


	for idx in $(seq 1 1 ${nn})
	#for idx in $(seq 1 1 ${nskullstrippermasks})
	  do
		echo ""
		if [ ${iter} == 1 ]; then
			mask=`cat ${allmaskstxt} | sed -n ${idx}p`
		else
			mask=`cat ${leftmaskstxt} | sed -n ${idx}p`
		fi
		#mask=`cat ${skullstrippermaskstxt} | sed -n ${idx}p`
		dice_withpreviousmask=`c3d -verbose ${mask} ${thisbrainmask} -overlap 1 |grep Dice | awk '{print $4}'`
		dice_withconsensus1mask=`c3d -verbose ${mask} ${consensus1mask} -overlap 1 |grep Dice |awk '{print $4}'`
		dice=`echo "scale=2; ${lambda1}*${dice_withpreviousmask}+${lambda2}*${dice_withconsensus1mask}" |bc -l`
		if [ "${trustrobex}" == 1 ]; then
			dice_withrobex=`c3d -verbose ${mask} ${robexmask} -overlap 1 | grep Dice | awk '{print $4}'`
			if [ `echo ${dice_withrobex} |cut -d. -f1` -eq 1 ]; then
				dice_withrobex=${dice_withpreviousmask} # added on 12/29/2014 to avoid robex mask dominanting
			fi 
			#lambda2_foratlas=`echo "${lambda2}*0.5" |bc -l`
			#lambda2_forrobex=`echo "${lambda2}*0.5" |bc -l`
			stdev_foratlas=`fslstats ${input_fullpath} -k ${consensus1mask} -s`
			stdev_forrobex=`fslstats ${input_fullpath} -k ${robexmask} -s`
			lambda2_foratlas=`echo "${lambda2}*${stdev_forrobex}/(${stdev_foratlas}+${stdev_forrobex})" |bc -l`
			lambda2_forrobex=`echo "${lambda2}-${lambda2_foratlas}" |bc -l`
			dice=`echo "scale=2; ${lambda1}*${dice_withpreviousmask}+${lambda2_foratlas}*${dice_withconsensus1mask}+${lambda2_forrobex}*${dice_withrobex}" |bc -l`
		fi
		#dice=`echo "scale=2; 0.5*${dice_withpreviousmask}+0.5*${dice_withconsensus1mask}" | bc`
		#dice=${dice_withpreviousmask}
		echo "from ${mask}, dice=${dice}" 
		echo "${mask} ${dice} dice_with_previousmask ${dice_withpreviousmask} dice_with_consensus1 ${dice_withconsensus1mask} dice_with_robex ${dice_withrobex}" >> ${dicetxt}

		dice_greaterthan_max_dice=`echo ${dice} '>' ${max_dice} | bc -l`
		if [ "${dice_greaterthan_max_dice}" == 1 ]; then
			if [ $iter == 1 ]&&[ ${usingPediatricAtlas} == 0 ]; then
				if [ ${idx} -le ${nskullstrippermasks} ]; then
					max_dice=${dice} # max_dice among all candidate masks from skull strippers, excluding atlas-based candidate masks
				fi
			else
				max_dice=${dice}
			fi
		fi
	done
	thre_dice=`echo "scale=2; ${max_dice}*${percent_thre}" | bc`
	sort -k2rn ${dicetxt} > ${sorteddicetxt}  # descending order
	echo ------------------
	echo "in consensus 2, iteration ${iter}: max_dice=${max_dice} thre_dice=${thre_dice}  (i.e., ${percent_thre}*max_dice)"
	echo ------------------

	# start selecting warped masks based on dice overlaps with the tentative mask
	selectedmasktxt=${consensus2DIR}/selectedmask_iter${iter}.txt
	#selectedskullstrippermasktxt=${consensus2DIR}/selectedskullstrippermask_iter${iter}.txt
	nselectedmasks=0
	#nselectedskullstrippermasks=0
	selectedmasks=""
	selecteddices=""
	#selectedmasks=${selectedatlasmasks}  # to start with, include all the ones selected from warped atlases
	if [ -f ${selectedmasktxt} ]; then
	#if [ -f ${selectedskullstrippermasktxt} ]; then
		\rm ${selectedmasktxt}
		#\rm ${selectedskullstrippermasktxt}
	fi
	if [ ${iter} == 1 ]; then
		maxNallowedforoneskullstripper=1  # to avoid one skull-stripper dominanting, allow only a few parameter variations for a given skull stripper
		if [ ${usingPediatricAtlas} == 1 ]; then
			maxNallowedforatlases=3 # similarly, to avoid atlases dominating, which will give the same results as consensus 1. but trust atlases more when in pediatric cases, because robex performs less accurately in pediatric than in adult cases
		else
			maxNallowedforatlases=2
		fi
		if [ ${usingT1GadAtlas} == 1 ]; then
			maxNallowedforatlases=4
		fi
	else
		maxNallowedforoneskullstripper=1
		if [ ${usingPediatricAtlas} == 1 ]; then
			maxNallowedforatlases=3
		else
			maxNallowedforatlases=2
		fi
		if [ ${usingT1GadAtlas} == 1 ]; then
			maxNallowedforatlases=5
		fi
	fi
	diff=$(( ${maxNallowedforatlases} - ${maxNallowedforoneskullstripper} ))
	betpre=bet2_${input_prefix}
	bsepre=bse_${input_prefix}
	afni3dSkullStrippre=3dSkullStrip_${input_prefix}
	hwapre=hwa_${input_prefix}
	robexpre=robex_${input_prefix}
	atlaspre=wrpedlabel_Template #atlaspre=mask_InSpace_${input_prefix}
	countbet=0
	countbse=0
	count3dSkullStrip=0
	counthwa=0
	countrobex=0
	countatlas=0

	selectedbet=0
	selectedbse=0
	selected3dSkullStrip=0
	selectedhwa=0
	selectedrobex=0
	selectedatlas=0
	for n in $(seq 1 1 ${nn})
	#for n in $(seq 1 1 ${nskullstrippermasks})
	  do
		countinoneskullstripper=0
		line=`cat ${sorteddicetxt} | sed -n ${n}p`  # read the n-th line
		maskfile=`echo ${line} |cut -d' ' -f1`
		dice=`echo ${line} |cut -d' ' -f2`
	
		if [ `echo ${dice} '>' 0.8 | bc -l` == 1 ]&&[ ${iter} == 1 ]; then # dice>0.7 for this candidate mask to be considered in the next round
			echo "${line}" | awk '{ print $1 }' >> ${leftmaskstxt}
		fi
			
		dice_greaterthan_thre_dice=`echo ${dice} '>=' ${thre_dice} | bc -l`
		if [ "${dice_greaterthan_thre_dice}" -eq 1 ]; then	
			if [ `grep ${betpre} <<< ${maskfile}` ]; then
				countbet=$(( ${countbet} + 1 ))
				countinoneskullstripper=${countbet}
			fi
			if [ `grep ${bsepre} <<< ${maskfile}` ]; then
				countbse=$(( ${countbse} + 1 ))
				countinoneskullstripper=${countbse}
			fi
			if [ `grep ${afni3dSkullStrippre} <<< ${maskfile}` ]; then
				count3dSkullStrip=$(( ${count3dSkullStrip} + 1 )) 
				countinoneskullstripper=${count3dSkullStrip}
			fi
			if [ `grep ${hwapre} <<< ${maskfile}` ]; then
				counthwa=$(( ${counthwa} + 1 ))
				countinoneskullstripper=${counthwa}
			fi
			if [ `grep ${robexpre} <<< ${maskfile}` ]; then
				countrobex=$(( ${countrobex} + 1 ))
				countinoneskullstripper=${countrobex}
			fi
			if [ `grep ${atlaspre} <<< ${maskfile}` ]; then
				countatlas=$(( ${countatlas} + 1 ))
				countinoneskullstripper=$(( ${countatlas} - ${diff} )) 
			fi
		
			if [ "${countinoneskullstripper}" -le "${maxNallowedforoneskullstripper}" ]; then
				echo "${maskfile} ${dice}" >> ${selectedmasktxt}
				#echo "${maskfile} ${dice}" >> ${selectedskullstrippermasktxt}
				nselectedmasks=$(( ${nselectedmasks} + 1 ))
				#nselectedskullstrippermasks=$(( ${nselectedskullstrippermasks} + 1 ))
				selectedmasks="${selectedmasks} ${maskfile}"
				selecteddices="${selecteddices} ${dice}"

				if [ `grep ${betpre} <<< ${maskfile}` ]; then selectedbet=$(( ${selectedbet} + 1 )); fi
				if [ `grep ${bsepre} <<< ${maskfile}` ]; then selectedbse=$(( ${selectedbse} + 1 )); fi
				if [ `grep ${hwapre} <<< ${maskfile}` ]; then selectedhwa=$(( ${selectedhwa} + 1 )); fi
				if [ `grep ${robexpre} <<< ${maskfile}` ]; then selectedrobex=$(( ${selectedrobex} + 1 )); fi
				if [ `grep ${atlaspre} <<< ${maskfile}` ]; then selectedatlas=$(( ${selectedatlas} + 1 )); fi
				if [ `grep ${afni3dSkullStrippre} <<< ${maskfile}` ]; then selected3dSkullStrip=$(( ${selected3dSkullStrip} + 1 )); fi
			fi
		else
			break
		fi
	done
	echo "${nselectedmasks} masks selected in iteration ${iter} within consensus 2: ${selectedbet} from bet, ${selectedbse} from bse, ${selected3dSkullStrip} from 3dSkullStrip, ${selectedhwa} from HWA, ${selectedrobex} from ROBEX, ${selectedatlas} from atlases."
	echo -e "\n\n"
	for i in $(seq 1 1 ${nselectedmasks}); do
		echo "`echo ${selectedmasks} | cut -d' ' -f${i}` `echo ${selecteddices} | cut -d' ' -f${i}`"
	done
	echo -e "\n\n"

	thisbrainmask=${consensus2DIR}/brainmask_consensus2_iter${iter}_${prefix}.nii.gz
	if [ ${nselectedmasks} -gt 1 ]; then
		echo "we now fuse (staple) these selected masks into a refined mask"
		echo ""
		echo "--> form a 4D file"
		file4D=${consensus2DIR}/4D_consensus2_iter${iter}.nii.gz	
		cmd="fslmerge -t ${file4D} ${selectedmasks}"
		#if [ ${runmode} == 1 ]; then
		#	pbsubmit -q p30 -c "${cmd}"
		#else
			${cmd}
		#fi
		#mywait ${file4D}
		fslmaths ${file4D} -bin ${file4D}
		sleep 10
		echo ""
		echo "--> fuse the 4D file by SBA"
		cmd="seg_LabFusion -in ${file4D} -out ${thisbrainmask} -SBA -v 2"
		#if [ ${runmode} == 1 ]; then 
		#	pbsubmit -q p30 -c "${cmd}"
		#else
			${cmd}
		#fi
		#mywait ${thisbrainmask}
		\rm ${file4D}
	else
		cp ${selectedmasks} ${thisbrainmask}
	fi
	echo "in consensus 2, the fused mask of iteration ${iter} is saved as ${thisbrainmask}"

	# stop iteration in consensus 2 if the refined mask agrees with the previous mask with >0.98 dice overlap or if the maximum number of allowed iteration is reached.
	preiter=$(( ${iter} - 1 ))
	prebrainmask=${consensus2DIR}/brainmask_consensus2_iter${preiter}_${prefix}.nii.gz
	dicewithprevious=`c3d -verbose ${thisbrainmask} ${prebrainmask} -overlap 1 |grep Dice | awk '{print $4}'`
	dicewithconsensus1=`c3d -verbose ${thisbrainmask} ${consensus1DIR}/brainmask_consensus1_final_${prefix}.nii.gz -overlap 1 |grep Dice | awk '{ print $4 }'`
	echo "in consensus 2,"
	echo "the dice overlap between this refined mask and previous mask is ${dicewithprevious}"
	echo "the dice overlap between this refined mask and the mask from consensus 1 is ${dicewithconsensus1}"
#read a
	if [ "`echo ${dicewithconsensus1} '<' ${stopping_thre_withinitialization} | bc -l`" -eq 1 ]; then
		# too far away from the the mask in consensus 1, not good. 
		# re-do consensus 2 with stricter criterion of selecting skull-stripper-generated masksfromskullstrippers	
		# percent_thre=`echo "scale=2; ${percent_thre}+0.03" | bc`
		#thisbrainmask=${prebrainmask};
		cp ${prebrainmask} ${thisbrainmask}
		break;
	elif [ "`echo ${dicewithprevious} '>' ${stopping_thre_withprevious} | bc -l`" -eq 1 ]; then
		# not too far from the initialization, and stable compared to the previous iteration, save and quit
		break;
	fi
	iter=$(( ${iter} + 1 ))
done
\cp ${thisbrainmask} ${consensus2DIR}/brainmask_consensus2_final_${prefix}.nii.gz


### ------------------------
# modified on 2017/8/30, because we added a potential padding in the beginning, we need to accordingly crop the computed mask
#
if [ ${xpadN} != 0 ]||[ ${xpadP} != 0 ]||[ ${ypadN} != 0 ]||[ ${ypadP} != 0 ]||[ ${zpadN} != 0 ]||[ ${zpadP} != 0 ]; then
        cmd="3dZeropad -${xorientN} -${xpadN} -${xorientP} -${xpadP} -${yorientN} -${ypadN} -${yorientP} -${ypadP} -${zorientN} -${zpadN} -${zorientP} -${zpadP} -prefix ${ssoutput} ${thisbrainmask}"
	${cmd}
        echo -e "\n------------cropping-----------\n${cmd}\n"
	if [ ${keepIntermediateResultsOrNot} == 1 ]; then # also crop intermediate results
		for anyimage in ${tmpdir}/consensus?/*.nii.gz ${tmpdir}/fromAtlases/1FOVnormalization*/*.nii.gz ${tmpdir}/fromAtlases/1FOVnormalization*/intermediateresults*/*.nii.gz ${tmpdir}/fromAtlases/2brainmask*/*.nii.gz ${tmpdir}/fromSkullStrippers/*/*.nii.gz; do
			xxdim=`fslinfo ${anyimage} | grep dim1 | sed -n 1p | awk '{ print $2 }'`			
			yydim=`fslinfo ${anyimage} | grep dim2 | sed -n 1p | awk '{ print $2 }'`
			zzdim=`fslinfo ${anyimage} | grep dim3 | sed -n 1p | awk '{ print $2 }'`
			#echo "xrangepadded=$xrangepadded; yrangepadded=${yrangepadded}; zrangepadded=${zrangepadded}"
			if [ ${xxdim} -eq ${xrangepadded} ]&&[ ${yydim} -eq ${yrangepadded} ]&&[ ${zzdim} -eq ${zrangepadded} ]; then
				anyimagecropped=${anyimage%.nii.gz}_cropped.nii.gz
				cmd="3dZeropad -${xorientN} -${xpadN} -${xorientP} -${xpadP} -${yorientN} -${ypadN} -${yorientP} -${ypadP} -${zorientN} -${zpadN} -${zorientP} -${zpadP} -prefix ${anyimagecropped} ${anyimage}"
				${cmd}
				mv ${anyimagecropped} ${anyimage}
			fi
		done
	fi
else
	echo -e "\n------------no padding in the beginning, no need for cropping here------------\n${cmd}\n"
        cmd="cp ${thisbrainmask} ${ssoutput}" 
fi
fslcpgeom ${input} ${ssoutput}                        # copy the header of input image into this ssmask
fslmaths ${ssoutput} -mul 1 ${ssoutput} -odt char     # make sure ssmaks is a byte image


# apply the computed brain mask ($ssoutput) to skull strip the input image
if [ ! -z ${outimage} ]; then
	fslmaths ${input} -mul ${ssoutput} ${outimage}
fi 

echo ""
echo ""
echo "The final skull stripping binary brain mask has been saved into ${ssoutput}"
echo "=== End of the Program ==="



# ----------------------------------------------------------------------------
# clean up and exit
cleanup



