#-----------------------------
help()
{
cat << HELP

This script takes deformed atlases/labels and fuse them with atlas selection.

DEPENDENCIES:

	seg_LabFusion	: `which seg_LabFusion`
	3dcalc		: `which 3dcalc`
	3dmerge 	: `which 3dmerge`
	fslsplit	: `which fslsplit`
	c3d		: `which c3d`	
	3dmask_tool	: `which 3dmask_tool`

########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -inWarpedImage   <file>   : the 4D file stacking all warped images
	-inWarpedLabel   <file>   : the 4D file stacking all warped label images
	-inTargetImage   <file>   : the original 3D target image to be segmented, or 4D file stacking all variations (FOVnormalized) target images
	-out  <file>	          : the output fused label image
	-dest <path>		  : to save intermediate results into a path (by default, if no path is given, we will use a temporary folder)

Optional
	-thresholdratio  <float>  : to threshold at which percent of the maximum CC/dice/dice*CC (default: 0.95)
	-selectionRule   <int>    : 1. CC; 2. Dice with Tentative (iterative until convergence); 3. CC*Dice (iterative until convergence)  default: 1

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-inWarpedImage) 
				inWarpedImages=$2;
				shift 2;;			# input warped images (in a 4D stack)

			-inWarpedLabel) 
				inWarpedLabels=$2;
				shift 2;;			# input warped labels (in a 4D stack)

			-inTargetImage) 
				inTargetImage=$2;
				shift 2;;			# input target image 

			-out) 
				output=$2
				shift 2;;			# output label image

			-thresholdratio)
				thresholdratio=$2
				shift 2;;

			-dest)	
				destdir=$2
				shift 2;;

			-selectionRule)
				selectionRule=$2
				shift 2;;

			-*) 
				echo "ERROR: no such option $1";
				help;;
			 *) 
				break;;
		esac
	done
}



# --------------------------------
checkdependencies()
{
        echo -e "\nLoading software dependencies..."
        for program in fslmerge fslsplit fslmaths 3dcalc 3dmerge 3dmask_tool c3d seg_LabFusion; do
                p=`which ${program} 2>/dev/null`
                if [ -z "${p}" ]; then
                        echo -e "\nError loading dependencies: ${program} does not exist in the system path. Program exits.\n"
                        echo -e "\nThe dependencies include: fslmerge, fslsplit and fslmaths (FSL 5.0.9); seg_LabFusion (NiftySeg, 0.9.4); 3dmerge, 3dmask_tool and 3dcalc (AFNI 16.2.06); and c3d (ITK-SNAP 0.8.2)\n"
                        exit 1
                else
                        echo "    ${program}:   ${p}"
                fi
        done
}


#---------------------------------
getFilePrefix()
{
	bName=`basename $1`
	prefix=${bName%.nii.gz}
	prefix=${prefix%.nii}
	prefix=${prefix%.hdr}
	prefix=${prefix%.img}  # added by Yangming, right now only supports .nii, .nii.gz, .hdr, .img
}


################################################ MAIN BODY ################################################

### check dependencies
checkdependencies


########################
### main
########################

if [ $# -lt 1 ]
then
	help
fi


### Default parameters
thresholdratio=0.95
selectionRule=1


### Reading the arguments
parse $*

### check dependencies
checkdependencies


### get prefix
getFilePrefix ${inTargetImage}

# split the warped atlases
tmpdir=`mktemp -d "/tmp/LabelFusionWithAtlasSelection-${prefix}-XXXXXX"`
echo "intermediate results will be saved into ${tmpdir}"
echo "checking the numbers of images and label images..."
fslsplit ${inWarpedImages} ${tmpdir}/warpedimage
fslsplit ${inWarpedLabels} ${tmpdir}/warpedlabel
numTargetImages=`fslinfo ${inTargetImage} |grep dim4 |sed -n 1p |awk '{ print $2 }'`
if [ ${numTargetImages} -gt 1 ]; then
    fslsplit ${inTargetImage} ${tmpdir}/targetimage
fi

# get the number of atlases
numImages=`ls -l ${tmpdir}/warpedimage*.nii.gz |wc -l`
numLabelImages=`ls -l ${tmpdir}/warpedlabel*.nii.gz |wc -l`
if [ ${numImages} != ${numLabelImages} ]; then
    echo -e "\nError: the two input 4D images contain different number of 3D volume (${numImages} images and ${numLabelImages} label images). Please double check."
    echo -e "program ends.\n"
    \rm -rf ${tmpdir}
    exit
fi
if [ ${numTargetImages} -gt 1 ]; then
    if [ ${numImages} != ${numTargetImages} ]; then
	echo ""
	echo "Error: the input 4D images should contain the same number of 3D images."
	echo "Program ends."
	echo ""
	exit
    fi
fi
echo "Totally ${numImages} atlases."


### atlas selection
maxIter=10
if [ ${selectionRule} == 1 ]; then
	echo -e "\natlas selection by the CC between the warped atlas and the target\n"
	# get the maximum cc (or cc*dice when useDice is turned on)
	maxcc=0
	for (( i=0; i<${numImages}; i++ )); do
	    ni=$(printf %04d ${i})
	    image=${tmpdir}/warpedimage${ni}.nii.gz
	    labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
	    if [ ${numTargetImages} -gt 1 ]; then
		targetimage=${tmpdir}/targetimage${ni}.nii.gz
	    else
		targetimage=${inTargetImage}
	    fi

	    cc=`CalculateImageSimilarity ${targetimage} ${image} |grep CC | cut -d' ' -f3`
	    echo "${image} ${cc}"
	    echo "${image} ${cc}" >> ${tmpdir}/cc.txt
	    cc_greaterthan_maxcc=`echo ${cc} '>' ${maxcc} | bc -l`
	    if [ ${cc_greaterthan_maxcc} -eq 1 ]&&[ `echo ${cc} |cut -d. -f1` != 1 ]; then	
		maxcc=${cc}
	    fi
	done
	# select atlases according to cc
	thre=`echo "scale=2; ${maxcc}*${thresholdratio}" | bc`
	echo "maxcc=${maxcc}, thre_cc=${thre}"
	numSelected=0
	selectedlabelimages=""
	for (( i=0; i<${numImages}; i++ )); do
	    ni=$(printf %04d ${i})
	    image=${tmpdir}/warpedimage${ni}.nii.gz
	    labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
	    l=$(( $i + 1 ))
	    
	    cc=`cat ${tmpdir}/cc.txt | sed -n ${l}p |cut -d' ' -f2`
	    cc_greater_than_thre=`echo ${cc} '>=' ${thre} | bc -l`
	    if [ ${cc_greater_than_thre} -eq 1 ]; then	
		echo "${image} ${cc}" >> ${tmpdir}/selectedatlases.txt
		echo "selected: `basename ${image}` ${cc}"
		selectedlabelimages="${selectedlabelimages} ${labelimage}"
		numSelected=$(( ${numSelected} + 1 ))
	    fi
	done
	echo "Totally ${numSelected} atlases have been selected."
	if [ ${numSelected} -eq 1 ]; then
		thresholdratio=0.85;
	fi

elif [ ${selectionRule} == 2 ]; then
	echo -e "\natlas selection by the Dice overlap of warped labels and the tentatively fused label image.\n"
	iter=1
	maxdice=0
	thre=0
	# to initialize
	preiter=$(( ${iter} - 1 ))
	fused=${tmpdir}/tentativelyfused_iter${preiter}.nii.gz
	file4d=${tmpdir}/4dlabels_foriter${preiter}.nii.gz
	cp ${inWarpedLabels} ${file4d}
	cmd_fusion="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -STAPLE -v 2"
	echo ${cmd_fusion}
	${cmd_fusion}
	# STAPLE may fail due to memory or convergency issues in the niftiSeg implementation
	# plan B when STAPLE fails
	if [ ! -f ${fused} ]; then
		echo -e "\nSTAPLE fusion failed, using MV instead\n"
		cmd_fusion="seg_LabFusion -in ${file4d} -out ${fused} -MV"
		echo ${cmd_fusion}
		${cmd_fusion}
	fi

	if [ -f ${fused} ]; then
		\rm ${file4d}
	else
		echo "label fusion was not successful. Please double check. The command used was:"
		echo -e "\n${cmd_fusion}\n"
		echo "program exits."
		exit 1
	fi
	
	echo -e "\nnow iteratively update the selection pool and the fused label image... \n"
	while [ ${iter} -lt ${maxIter} ]; do
		echo -e "\n ===> iteration ${iter}, tentatively fused label image = ${fused}"
		preiter=$(( ${iter} - 1 ))
		# get the maximum dice
		maxdice=0
		for (( i=0; i<${numImages}; i++ )); do
		    	ni=$(printf %04d ${i})
		    	image=${tmpdir}/warpedimage${ni}.nii.gz
			labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
			#cmd_dice="${JaccardScore} ${labelimage} ${fused}"
			#dice=`${cmd_dice} | sed -n 3p | awk '{ print $3 }'`
			dice=`c3d -verbose ${labelimage} ${fused} -overlap 1 | sed -n 8p | awk '{ print $4}'`
			echo "       ${image} ${dice}"
			echo "${image} ${dice}" >> ${tmpdir}/dice_iter${iter}.txt
			dice_greaterthan_maxdice=`echo ${dice} '>' ${maxdice} | bc -l`
			if [ ${dice_greaterthan_maxdice} == 1 ]&&[ `echo ${dice} |cut -d. -f1` != 1 ]; then	
				maxdice=${dice}
			fi
		done
		# select atlases according to dice
		thre=`echo "scale=2; ${maxdice}*${thresholdratio}" | bc`
		echo "in iter ${iter}: maxdice=${maxdice}, thre_dice=${thre}"
		numSelected=0
		selectedlabelimages=""
		for (( i=0; i<${numImages}; i++ )); do
		    ni=$(printf %04d ${i})
		    image=${tmpdir}/warpedimage${ni}.nii.gz
		    labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
		    l=$(( $i + 1 ))
	    
		    dice=`cat ${tmpdir}/dice_iter${iter}.txt | sed -n ${l}p |cut -d' ' -f2`
		    dice_greater_than_thre=`echo ${dice} '>=' ${thre} | bc -l`
		    if [ ${dice_greater_than_thre} -eq 1 ]; then	
			echo "${image} ${dice}" >> ${tmpdir}/selectedatlases_iter${iter}.txt
			echo "       selected: `basename ${image}` ${dice}"
			selectedlabelimages="${selectedlabelimages} ${labelimage}"
			numSelected=$(( ${numSelected} + 1 ))
		    fi
		done
		echo "Iteration ${iter}: totally ${numSelected} atlases have been selected."

		if [ ${numSelected} -eq 1 ]; then
	                thresholdratio=0.85;
        	fi

		# tentatively fused label image
		echo "now fuse those selected to initialize the next iteration"
		fused=${tmpdir}/tentativelyfused_iter${iter}.nii.gz
		file4d=${tmpdir}/4dlabels_foriter${iter}.nii.gz
		fslmerge -t ${file4d} ${selectedlabelimages}
		if [ ${numSelected} -gt 1 ]; then
			cmd="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -STAPLE -v 2"
		else
			cmd="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -MV"
		fi
		echo ${cmd}
		${cmd}
		if [ ! -f ${fused} ]; then
			echo -e "\nSTAPLE fusion failed, using MV instead\n"
			cmd="seg_LabFusion -in ${file4d} -out ${fused} -MV"
			echo ${cmd}
			${cmd}
		else
			\rm ${file4d}
		fi


		# convergence criterion
		fused_preiter=${tmpdir}/tentativelyfused_iter${preiter}.nii.gz
		#dice=`${JaccardScore} ${fused_preiter} ${fused} | sed -n 3p | awk '{ print $3 }'`
		dice=`c3d -verbose ${fused_preiter} ${fused} -overlap 1 | sed -n 8p | awk '{ print $4}'`
		if [ `echo ${dice} '>' 0.99 | bc -l ` == 1 ]; then
			echo -e "\nAtlas selection converged.\n"
			break;
		else 
			iter=$(( ${iter} + 1 ))
		fi
	done
elif [ ${selectionRule} == 3 ]; then
	echo -e "\natlas selection by the Dice overlap of warped labels and the tentatively fused label image.\n"
	iter=1
	maxdice=0
	thre=0
	# to initialize
	preiter=$(( ${iter} - 1 ))
	fused=${tmpdir}/tentativelyfused_iter${preiter}.nii.gz
	file4d=${tmpdir}/4dlabels_foriter${preiter}.nii.gz
	cp ${inWarpedLabels} ${file4d}
	cmd_fusion="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -STAPLE -v 2"
	echo ${cmd_fusion}
	${cmd_fusion}
	if [ ! -f ${cmd_fusion} ]; then
		echo -e "\nSTAPLE fusion failed, using MV fusion instead\n"
		cmd_fusion="seg_LabFusion -in ${file4d} -out ${fused} -MV"
		echo ${cmd_fusion}
		${cmd_fusion}
	fi
	
	if [ -f ${fused} ]; then
		\rm ${file4d}
	else
		echo "label fusion was not successful. Please double check. The command used was:"
		echo -e "\n${cmd_fusion}\n"
		echo "program exits."
		exit 1
	fi
	
	echo -e "\nnow iteratively update the selection pool and the fused label image... \n"
	while [ ${iter} -lt ${maxIter} ]; do
		echo -e "\n ===> iteration ${iter}, tentatively fused label image = ${fused}"
		preiter=$(( ${iter} - 1 ))
		# get the maximum dice
		maxccdice=0
		for (( i=0; i<${numImages}; i++ )); do
		    	ni=$(printf %04d ${i})
		    	image=${tmpdir}/warpedimage${ni}.nii.gz
			labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
			if [ ${numTargetImages} -gt 1 ]; then
				targetimage=${tmpdir}/targetimage${ni}.nii.gz
			else
				targetimage=${inTargetImage}
			fi
			cmd_cc="CalculateImageSimilarity ${targetimage} ${image}" 
			cc=`${cmd_cc} |grep CC | cut -d' ' -f3`
			#cmd_dice="${JaccardScore} ${labelimage} ${fused}"
			#dice=`${cmd_dice} | sed -n 3p | awk '{ print $3 }'`
			dice=`c3d -verbose ${labelimage} ${fused} -overlap 1 | sed -n 8p | awk '{ print $4}'`

			ccdice=`echo "scale=2; ${cc}*${dice}" | bc -l`
			echo "       ${image} ${ccdice}"
			echo "${image} ${ccdice}" >> ${tmpdir}/ccdice_iter${iter}.txt
			ccdice_greaterthan_maxccdice=`echo ${ccdice} '>' ${maxccdice} | bc -l`
			if [ ${ccdice_greaterthan_maxccdice} == 1 ]&&[ `echo ${ccdice} |cut -d. -f1` != 1 ]; then	
				maxccdice=${ccdice}
			fi
		done
		# select atlases according to dice
		thre=`echo "scale=2; ${maxccdice}*${thresholdratio}" | bc`
		echo "in iter ${iter}: maxccdice=${maxdice}, thre_ccdice=${thre}"
		numSelected=0
		selectedlabelimages=""
		for (( i=0; i<${numImages}; i++ )); do
		    ni=$(printf %04d ${i})
		    image=${tmpdir}/warpedimage${ni}.nii.gz
		    labelimage=${tmpdir}/warpedlabel${ni}.nii.gz
		    l=$(( $i + 1 ))
	    
		    ccdice=`cat ${tmpdir}/ccdice_iter${iter}.txt | sed -n ${l}p |cut -d' ' -f2`
		    ccdice_greater_than_thre=`echo ${ccdice} '>=' ${thre} | bc -l`
		    if [ ${ccdice_greater_than_thre} -eq 1 ]; then	
			echo "${image} ${ccdice}" >> ${tmpdir}/selectedatlases_iter${iter}.txt
			echo "       selected: `basename ${image}` ${ccdice}"
			selectedlabelimages="${selectedlabelimages} ${labelimage}"
			numSelected=$(( ${numSelected} + 1 ))
		    fi
		done
		echo "Iteration ${iter}: totally ${numSelected} atlases have been selected."

		if [ ${numSelected} -eq 1 ]; then
        	        thresholdratio=0.85;
	        fi

		# tentatively fused label image
		echo "now fuse those selected to initialize the next iteration"
		fused=${tmpdir}/tentativelyfused_iter${iter}.nii.gz
		file4d=${tmpdir}/4dlabels_foriter${iter}.nii.gz
		fslmerge -t ${file4d} ${selectedlabelimages}
		if [ ${numSelected} -gt 1 ]; then
			cmd="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -STAPLE -v 2"
		else
			cmd="timeout 1h seg_LabFusion -in ${file4d} -out ${fused} -MV"
		fi
		echo ${cmd}
		${cmd}
		if [ ! -f ${fused} ]; then
			echo -e "\nSTAPLE fusion failed, using MV fusion instead\n"
			cmd="seg_LabFusion -in ${file4d} -out ${fused} -MV"
			echo ${cmd}
			${cmd}
		else
			\rm ${file4d}
		fi


		# convergence criterion
		fused_preiter=${tmpdir}/tentativelyfused_iter${preiter}.nii.gz
		#dice=`${JaccardScore} ${fused_preiter} ${fused} | sed -n 3p | awk '{ print $3 }'`
		dice=`c3d -verbose ${fused_preiter} ${fused} -overlap 1 | sed -n 8p | awk '{ print $4}'`
		if [ `echo ${dice} '>' 0.99 | bc -l ` == 1 ]; then
			echo -e "\nAtlas selection converged.\n"
			break;
		else 
			iter=$(( ${iter} + 1 ))
		fi
	done
else
	echo -e "\nError: the '-selectionRule' option has to be either 1, 2 or 3. Program exits.\n"
	exit 1
fi



# fuse the selected atlases
tmpout=${tmpdir}/fusedlabelimage_${prefix}.nii.gz
if [ ${selectionRule} == 1 ]; then
	fslmerge -t ${tmpdir}/4D.nii.gz ${selectedlabelimages}
	cmd="timeout 1h seg_LabFusion -in ${tmpdir}/4D.nii.gz -out ${tmpout} -STAPLE -v 2"
	echo ${cmd}
	${cmd}
	if [ ! -f ${tmpout} ]; then
		echo -e "\nSTAPLE fusion failed, using MV fusion instead\n"
		cmd="seg_LabFusion -in ${tmpdir}/4D.nii.gz -out ${tmpout} -MV"
		echo ${cmd}
		${cmd}
	else
		\rm ${tmpdir}/4D.nii.gz
	fi
else
	\mv ${fused} ${tmpout}
fi


# morphological dilation and erosion
if [ -f ${output} ]; then
    \rm ${output}
fi
cmd="3dmaskave -mask ${tmpout} ${tmpout}"
a=`$cmd | awk '{ print $1 }'`
if [ $a == 1 ]; then
	3dmask_tool -input ${tmpout} -prefix ${output} -dilate_input 2 -2
else
	\cp ${tmpout} ${output}
fi


if [ ! -z ${destdir} ]; then
    \cp ${tmpdir}/*.txt ${destdir}
fi

\rm -rf ${tmpdir}
