function mask = create_custom_mask(Params,inds,action)

% Create brain mask for FuSeISC to perform analysis within the desired ROI.
% As a default, we generate a mask which removes white-matter, ventricle,
% and brain-stem regions from the analysis. This default option is obtained
% as follows: custommask = create_custom_mask(Params);
%
% To create custom mask, you need to provide variable "inds" where you can 
% list ROIs you want to include/exclude. In this variable, atlas indices 
% (whole numbers) are provided in the first row and the atlas type in the 
% second row (1 = cortical atlas, 2 = subcortical atlas). Then you must 
% also provide variable "action" to denote whether you want to exclude 
% ('exclude') or include ('include') the given regions.
% Brain indices of the brain areas for both atlases are listed in the 
% file "atlas_labels_abbr.csv". As an example, see below have we generate 
% the default mask.

if ~( nargin == 1 || nargin == 3 )
    error('Number of input arguments must be 1 or 3!')
end
[brain,mask,atlasCort,atlasSub,AffMat,Labels] = loadAtlases(Params);

if nargin == 1
    % As a default, generate a mask removing white-matter, ventricle,
    % and brain-stem regions. The first row shows region index, the second
    % shows whether the index comes from cortical (1) or subcortical (2)
    % atlas. Indices of the brain areas are listed in variable "Labels"
    % from the file "atlas_labels_abbr.csv".
    inds = [101 103 108 112 114; 2 2 2 2 2];
    action = 'exclude';
end

if size(inds,1) ~= 2
   error('"inds" must have two rows!') 
end

switch action
    case 'include'
        custommask = zeros(size(atlasCort.img));
        for m = 1:size(inds,2)
            switch inds(2,m)
                case 1
                    atlas = atlasCort.img;
                case 2
                    atlas = atlasSub.img;                    
                otherwise
                    error('Every value in the second row of "inds" must be 1 or 2')
            end
            custommask(atlas==inds(1,m)) = 1;
        end        
    case 'exclude'
        % Load brain mask:
        if strcmp(Params.PublicParams.fileFormat,'nii')
            brainmask = load_nii([Params.PrivateParams.brainMask '.nii']);
            brainmask = brainmask.img;
        else
            brainmask = load(Params.PrivateParams.brainMask);
        end
        brainmask = double(brainmask);
        noisemask = zeros(size(brainmask));
        for m = 1:size(inds,2)
            switch inds(2,m)
                case 1
                    atlas = atlasCort.img;
                case 2
                    atlas = atlasSub.img;                    
                otherwise
                    error('Every value in the second row of "inds" must be 1 or 2')
            end            
            noisemask(atlas==inds(1,m)) = 1;
        end
        
        % find linear indices from both masks:
        noisemask_ind = find(noisemask);
        brainmask_ind = find(brainmask);
        
        % Generate a difference mask:
        
        % Indices of a difference mask
        diffmask_ind = setdiff(brainmask_ind,noisemask_ind);
        % Create a difference mask:
        custommask = zeros(size(brainmask));
        custommask(diffmask_ind) = 1;
        
    otherwise
        error('action must be "include" or "exclude"!')
end

mask = custommask;

