function statMap = computeUnpairedStats(Params,Data,GroupIndexA,GroupIndexB)

% This function returns statistical 3D-brain map for the unpaired ISC 
% comparison between two groups A and B. 
%
% Inputs:
%
% Params - Parameter-struct of the project. It can be loaded from the project
% destination directory.
% Data - 4D data matrix containing within group and possibly between group 
% subject-pair-wise (standardized) ISC values for each voxel (x,y,z). 
% GroupIndexA - index of group/session A (default is 1)
% GroupIndexB - index of group/session B (default is 2)
% Note: When just two groups/sessions are added to the project, use defaults. 
%
% Outputs:
%
% statMap - 3D (non-thresholded) statistical brain map
%
% Example:
% % Assume you have the project called "Test1" with two groups
% % and you have run unpaired test with EW permutations and voxel-based 
% MCC correction. After running the analysis, go to the destination 
% directory of the project and type:
% load Test1_params.mat
% load MemMaps
% Data = memMaps.PFMatMapSession.whole.band0.cor.sessComp1.Data.xyzc;
% statMap = computeUnpairedStats(Params,Data); 
% 
% % Then, threshold the data:
% cd PFsession
% load ThPFBand0sessComp1win0
% statMapPos = statMap;
% statMapPos(statMap < Th(1)) = 0; % group A > group B
% statMapNeg = statMap;
% statMapNeg(statMap > -Th(1)) = 0; % group B > group A
% 
% % Visualize one slice:
% figure,imagesc(rot90(statMapPos(:,:,45)))
% colormap(hot);axis equal;colorbar;
%

% Last modified by Jukka-Pekka Kauppi 10.2.2017


if ( nargin < 2 || nargin > 4 || nargin == 3 )
    error('Number of input arguments must be 2 or 4!')
end

if nargin == 2
    GroupIndexA = 1;
    GroupIndexB = 2;
end

Pub = Params.PublicParams;
Priv = Params.PrivateParams;

% total number of subject pairs for the largest group:
nrSubjectPairs = Priv.nrSubjects*(Priv.nrSubjects-1)/2;

% compute true number of subjects and subject pairs for each group:
trueNrSubjects = zeros(Priv.nrSessions,1);
trueNrSubjectPairs = zeros(Priv.nrSessions,1);
for ses = 1:Priv.nrSessions
    trueNrSubjects(ses) = sum(Priv.dummies(ses,:) == 0);
    trueNrSubjectPairs(ses) = trueNrSubjects(ses)*(trueNrSubjects(ses) - 1)/2;
end

% find indices of the (z-transformed) subject-pairwise-ISCs for groups A and B:
groupA_inds = 1:trueNrSubjectPairs(GroupIndexA);
groupB_inds = (nrSubjectPairs+1):(nrSubjectPairs+trueNrSubjectPairs(GroupIndexB));

% compute the difference of the means of the elements between the groups:
statMap = sum(Data(:,:,:,groupA_inds),4)/length(groupA_inds) + sum(Data(:,:,:,groupB_inds),4)/length(groupB_inds);
    