function [corMats,Params] = returnCorrMat(Params)

% function [corMats,Params] = returnCorrMat(Params)
% This function returns ISC correlation matrix data through the memory 
% mapping objects. 
%
% There are 3 input options how this function can be
% called, the options are:
%
% 1. Call without any input arguments: 
% [...] = returnCorrMat;
%
% Note! To call the function in this mode, you must be in your 
% ISC analysis project directory.
%
% 2. Call with memory mapping object:
% [...] = returnCorrMat(memMaps);
%
% Note! Memory mapping objects can be loaded in the analysis destination
% direction with the command: load memMaps
%
% 3. Call with params-struct:
% [...] = returnCorrMat(Params);
%
% Note! Params-variable can be loaded in the analysis destination
% direction with the following commands: 
% load('Tag.mat');
% load(Tag)
%
% Outputs:
% 
% corMats - the output is a struct containing vectorized ISC matrix data
% for each voxel in all sessions. For example, to access ISC matrix data
% in voxel [40, 30 , 50] in session 1, type:
% vectorizedISCmatrixData = squeeze(corMats.Session1(40,30,50,:))
%
% Params - parameter-struct of your ISC analysis project.
% 
% To create full correlation matrices from the vectorized data, use
% function returnFullCorrMat.m.
%
% EXAMPLE:
%
% In this example, ISCs/ISC matrices for two groups have been computed, 
% in which the first group had 3 subjects and the second group had 4
% subjects.
%
% Return all vectorized correlation matrices:
% >> corMats = returnCorrMat; 
%
% Return full ISC matrices for voxel [50,60,20]:
% >> ISCmatrix = returnFullCorrMat(corMats,[50,60,20]); 
%
% Show ISC matrices:
% >> ISCmatrix.Session1
%
% ans =
%
%    1.0000    0.0858    0.1492
%    0.0858    1.0000    0.2094
%    0.1492    0.2094    1.0000
%
% >> ISCmatrix.Session2
%
% ans =
%
%    1.0000   -0.1189   -0.1427   -0.1391
%   -0.1189    1.0000   -0.0369    0.2347
%   -0.1427   -0.0369    1.0000   -0.0181
%   -0.1391    0.2347   -0.0181    1.0000
% 
% SEE ALSO:
% 
% RETURNFULLCORRMAT

if nargin == 0
    Params.PublicParams = [];
    load('Tag.mat');
    load(Tag)
end

if isfield(Params,'PublicParams')

    % load memMaps:
    F = Params.PublicParams.dataDestination;
    load([F 'memMaps.mat'])      
else
    memMaps = Params;
    [dataDestination,B] = fileparts(memMaps.resultMap.whole.band0.Session1.cor.Filename);
    F = dataDestination(1:end-7);
    load([F 'Tag.mat']);
    load([F Tag]);
end
    
D = Params.PrivateParams.dummies;

clear corMats
for m = 1:size(D,1)
    nrS = sum(D(m,:) == 0);
    nrPairs = nrS*(nrS-1)/2;    
    C = memMaps.cormatMap.whole.band0.(['Session' num2str(m)]).cor.Data.xyzc;
    %DD.(['Session' num2str(m)]) = C;
    corMats.(['Session' num2str(m)]) = C(:,:,:,1:nrPairs);
end


