function Res = clusterInfo(Params,mapType,TH)

% Once you have run session comparison analysis (paired test) using the
% ISC toolbox, use this function to compute cluster information as follows:
%
% Res = clusterInfo(Params,'contrast');
%
% The function will extract cluster information and save it in the
% excel-file in the results-folder called "session_comparison_cluster_info".
% Results for different p-values are saved in different spreadsheets.
% The following information is saved:
% max: peak value of a cluster (sum ZPF statistic)
% x (MNI): x-MNI-coordinate of the peak value of a cluster
% y (MNI): y-MNI-coordinate of the peak value of a cluster
% z (MNI): z-MNI-coordinate of the peak value of a cluster
% size: Size of a cluster in voxels.
%
% Res-variable also contains the cluster information.

% Jukka-Pekka Kauppi 27.7.2018

% Note: switch compute_peak_location to 0 to save MNI-coordinates of the 
% center of mass of the clusters! When this is set to 1, MNI-coordinates 
% of the peak value of the clusters are extracted.
compute_peak_location = 1; 

freqBand = 1;
non_parametric_z = 0;
sessionComp = 1;
correction_type = 2; % FDR indep/dep
min_clust_size = 1;
save_result_table = 1;

critPvals = [];

if nargin == 1
    mapType = 'basic';
end
if nargin < 3
    TH = 'use_p_vals'; % use p-values by defaul
end

if isstr(TH) % use p-vals
    if strcmp(TH,'use_p_vals') == 0
        error('''Th'' must either be a numeric value, or it must be a string called ''use_p_vals''!')% use -p-values
    end
    use_pvals = 1;
else % use manual threshold
    use_pvals = 0;
end

win = 0;
str = {'whole','win'};
Priv = Params.PrivateParams;
Pub = Params.PublicParams;

[brain,mask,atlasCort,atlasSub,AffMat,Labels] = loadAtlases(Params);
brain_voxels = find(mask.img);

switch mapType
    case 'basic'
        
        
    case 'contrast'
        load([Pub.dataDestination 'memMaps.mat'])
        
        dPos = sum( memMaps.(Priv.PFmatMapSessionName).(str{win+1}...
            ).([Priv.prefixFreqBand num2str(freqBand-1)...
            ]).(Priv.simM{3}).([Priv.prefixSessComp...
            num2str(sessionComp)]).Data.xyzc,4);
        dNeg = -1*dPos;
        
        if use_pvals
            str_th = [Priv.prefixPF 'Band' num2str(freqBand-1) ...
                'sessComp' num2str(sessionComp) 'win' ...
                num2str(win) '.mat'];
            filen = [Priv.PFsessionDestination 'Th' str_th];
            ThAll = load(filen);
            Th = ThAll.Th;
            Th(Th==0) = NaN;
            critPvals = ThAll.critVals;
        else
            Th = TH;
        end
        
    case 'group'
        switch non_parametric_z
            case 1 % non-parametric z
                filen = [Priv.PFsessionDestination 'band0Z_nonparametric' num2str(sessionComp) '.mat'];
                dataT = load(filen);
                dPos = dataT.z_nonparametric;
                dNeg = -1*dPos; % flip values to change contrast map direction "<" to ">"
            case 0 % parametric z
                % if  %handles.Pub.permutationType == 1 || handles.Pub.permutationType == 2 || handles.Pub.permutationType == 3
                % elementwise (EW), subjectwise (SW) without
                % dummies, or full subjectwise (SW).
                if correction_type < 5 % FDR or FWE-correction -> load mat-file of result maps
                    filen = [Priv.PFsessionDestination 'band0ZsumStat' num2str(sessionComp) '.mat'];
                    dataT = load(filen);
                    dPos = dataT.PFdata3D;
                    dNeg = -1*dPos; % flip values to change contrast map direction "<" to ">"
                elseif correction_type == 5 % load cluster maps
                    if K == 1
                        pa = [Priv.resultsDestination 'ISCSessionCompPosClusters1Band' num2str(freqBand-1) '.nii'];
                    else
                        pa = [Priv.resultsDestination 'ISCSessionCompNegClusters1Band' num2str(freqBand-1) '.nii'];
                    end
                    dataT = load_nii(pa);
                    dPos = dataT.img;
                    dNeg = -1*dPos;
                else
                    error('Unknown correction type!!')
                end
        end
        if use_pvals
            switch non_parametric_z
                case 1
                    str = [Priv.prefixPF 'Band' num2str(freqBand-1) ...
                        'sessComp' num2str(sessionComp) 'win' ...
                        num2str(win) '.mat'];
                    filen = [Priv.PFsessionDestination 'fdr_Th' str];
                    ThAll = load(filen);
                    switch correction_type
                        case 1 % no correction:
                            Th = ThAll.nonparam_zth_uncorrected;
                        case 2 % FDR (indep/dep assump.)
                            Th = ThAll.nonparam_zthFDRid;
                    end
                    Th(Th==0) = NaN;
                    
                case 0
                    if correction_type < 4  % FDR
                        str = [Priv.prefixPF 'Band' num2str(freqBand-1) ...
                            'sessComp' num2str(sessionComp) 'win' ...
                            num2str(win) '.mat'];
                        filen = [Priv.PFsessionDestination 'fdr_Th' str];
                        ThAll = load(filen);
                        switch correction_type
                            case 1 % no correction:
                                Th = ThAll.zth_uncorrected;
                            case 2 % FDR (indep/dep assump.)
                                Th = ThAll.zthFDRid;
                            case 3 % FDR (indep/no assump.)
                                Th = ThAll.zthFDRid;
                        end
                        Th(Th==0) = NaN;
                        
                    elseif correction_type == 4 % FWE
                        load([Priv.PFsessionDestination 'Th' ...
                            Priv.prefixPF 'Band' ...
                            num2str(freqBand-1) 'sessComp' ...
                            num2str(sessionComp) 'win' num2str(win)])
                        Th(Th==0) = NaN;
                    elseif correction_type == 5 % cluster-based
                        load([Priv.PFsessionDestination 'clusterTh' ...
                            Priv.prefixPF 'Band' ...
                            num2str(freqBand-1) 'sessComp' ...
                            num2str(sessionComp) 'win' num2str(win)])
                        if dataset2 > dataset
                            Th = cThPos;
                        else
                            Th = cThNeg;
                        end
                        Th(Th==0) = NaN;
                    end
            end
        else
            Th = TH;
        end
end

for t = 1:length(Th) % repeat for each threshold: 
    TBL = [];
    results = [];
    pos_clust_found = 0;
    neg_clust_found = 0;
    
    for pos_neg = 1:2
        switch pos_neg
            case 1
                BW = dPos >= Th(t);
            case 2
                BW = dNeg >= Th(t);
        end
        CC = bwconncomp(BW);        
        c_mass = regionprops(CC,'Centroid');
        c_area = regionprops(CC,'Area');
        c_area = arrayfun(@(x) x.Area,c_area)';
        [c_area,c_area_ind] = sort(c_area,'descend');
        CC.PixelIdxList = CC.PixelIdxList(c_area_ind);
        n = 0;
        for nn = 1:length(c_area_ind)   % compute for each cluster:
            if c_area(nn) < min_clust_size
                continue
            end
            n = n + 1;
            
            if pos_neg == 1
                pos_clust_found = 1;
            end
            if pos_neg == 2
                neg_clust_found = 1;
            end
            
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            results.subclusters{n,pos_neg} = CC.PixelIdxList{nn};
            
            switch pos_neg
                case 1
                    vals_pos = dPos(results.subclusters{n,pos_neg});
                    results.mean_val(n,pos_neg) = mean(vals_pos);
                    [results.max_val(n,pos_neg),results.max_val_ind(n,pos_neg)] = max(vals_pos);     
                    
                    
                case 2
                    vals_neg = dNeg(results.subclusters{n,pos_neg});
                    results.mean_val(n,pos_neg) = mean(vals_neg);
                    [results.max_val(n,pos_neg),results.max_val_ind(n,pos_neg)] = max(vals_neg);                    
            end
            
            % hae aivojen sisn osuvien vokselien indeksi:
            [I,i1] = intersect(brain_voxels,CC.PixelIdxList{nn});
            results.sub_clust_size(n,pos_neg) = length(i1);
            Ce = round(c_mass(c_area_ind(nn)).Centroid([2 1 3]));
            siz = size(BW);
                        
            com_ind = sub2ind(siz, Ce(1), Ce(2), Ce(3));
            
            if compute_peak_location
                orig_ind_max = results.subclusters{n,pos_neg}(results.max_val_ind(n,pos_neg));
                [x,y,z] = ind2sub(siz,orig_ind_max);
            else    
                [x,y,z] = ind2sub(siz,com_ind);
            end
            % transform it to corresponding MNI coordinate:          
            VoxCoord = [x-1;y-1;z-1;1];
            MNI_com =  AffMat*VoxCoord;
            results.MNI_com(n,:,pos_neg) = round(MNI_com(1:3)*10)/10;
            results.com(n,pos_neg) = com_ind;
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        end
        if pos_clust_found
            [so,soi] = sort(results.max_val(:,1),'descend');
            soi = soi(find(so>0));
            TBL{1} = [results.max_val(soi,1) results.sub_clust_size(soi,1) results.MNI_com(soi,:,1)];
        end
        if neg_clust_found
            [so,soi] = sort(results.max_val(:,2),'descend');
            soi = soi(find(so>0));
            TBL{2} = [results.max_val(soi,2) results.sub_clust_size(soi,2) results.MNI_com(soi,:,2)];
        end        
    end    
    
    if ( pos_clust_found == 0 && neg_clust_found == 0 )
       disp('No clusters found!') 
       TBL = [];
       return
    end
    
    if pos_clust_found
        disp('  ')
        disp('  ')
        disp('Clusters in positive side (cond1 > cond2):')
        disp(['FWE-corrected p-value: ' num2str(critPvals(t))])
        for pp = 1:size(TBL{1},1)
            formatSpec = string('%-4d %-0.2f  %-6d %-4d %-4d %-4d');
            str = sprintf(formatSpec,pp,TBL{1}(pp,1),TBL{1}(pp,2),TBL{1}(pp,3),TBL{1}(pp,4),TBL{1}(pp,5));
            disp(str)
        end
    end
    
    if neg_clust_found
        disp('  ')
        disp('  ')
        disp('Clusters in negative side (cond2 > cond1):')
        disp(['FWE-corrected p-value: ' num2str(critPvals(t))])
        for pp = 1:size(TBL{2},1)
            formatSpec = string('%-4d %-0.2f  %-6d %-4d %-4d %-4d');
            str = sprintf(formatSpec,pp,TBL{2}(pp,1),TBL{2}(pp,2),TBL{2}(pp,3),TBL{2}(pp,4),TBL{2}(pp,5));
            disp(str)
        end
    end
    
    Res(t).clusterinfo = results;
    Res(t).table = TBL;
    Res(t).Pval = critPvals(t);
    
    if save_result_table
        fn = [Params.PublicParams.dataDestination 'results\session_comparison_cluster_info.xls'];
        nrClust = size(Res(t).table{1},1);
        ws = t;
        warning off
        ws = ['p = ' num2str(critPvals(t))];
        xlswrite(fn,{'positive'},ws,'A1')
        xlswrite(fn,{['p = ' num2str(critPvals(t))]},ws,'A2')
        xlswrite(fn,{'max';'x (MNI)';'y (MNI)';'z (MNI)';'size'}',ws,['A3'])
        xlswrite(fn,Res(t).table{1}(:,[1 3 4 5 2]),ws,['A4:E' num2str(nrClust+3)])

        nrClust = size(Res(t).table{2},1);
        xlswrite(fn,{'negative'},ws,'G1')
        xlswrite(fn,{['p = ' num2str(critPvals(t))]},ws,'G2')
        xlswrite(fn,{'max';'x (MNI)';'y (MNI)';'z (MNI)';'size'}',ws,['G3'])
        RR = Res(t).table{2}(:,[1 3 4 5 2]);
        RR(:,1) = -1*RR(:,1);
        xlswrite(fn,RR,ws,['G4:K' num2str(nrClust+3)])
        warning on        
    end
        
end
