#! /bin/bash



#-----------------------------
help()
{
cat << HELP

This script takes an input folder and use all images (*.nii.gz) inside that folder to generate an unbiased atlas through unbiased population-wise (aka group-wise) affine+deformable registration.

This program runs either parallelly in cluster (default, PBS or slurm) or parallely/sequentially on a linux machine (via the '-runmode 2' or '-runmode 3' option).

This program can generate normal-appearing atlases from either an normal cohort or an abnormal cohort. In the latter case, the algorithm will remove abnormality as (a) input from users masks (via the '-maskdir' option) and (b) detected by the algorithms as outlier values from the population distribution, at the voxel level.

This program can either be applied to non-standardized intensity images (e.g., structural MRIs where the intensity has an arbitrary unit), or to standardized absolution values (e.g., CT, FA, ADC). In the other case, intensity does carry meaning. So please use the '-histop 0' option if your input images have absolute intensity values.


########################################
USAGE: $0 [OPTIONS]
OPTIONS:

Reqd:   -indir   <folder>   : input folder, where all *.nii.gz files are stored (images must be in .nii.gz format).
	-out     <file>     : output atlas file.

Optional:
	-debug	            : debug mode, keep all intermediate results (default: not active)
	-maskdir <folder>   : mask folder, where all *.nii.gz files are masks corresponding to the images in the '-indir' folder. For instance, A.nii.gz, B.nii.gz and C.nii.gz in the input folder should have corresponding A_mask.nii.gz B_mask.nii.gz and C_mask.nii.gz in the mask folder. The mask images are binary images containing 1 for foreground that involves in the registration and 0 for background that will be masked out during registration. Note: the mask images must be binary images (unsigned char datatype).
	-histop  <int>      : 1 (default) histogram operation (bias correction, intensity normalization and histogram matching)
			      0 no histogram operation, especially when intensities are absolute values (eg, FA, ADC, CT, etc).
	-runmode <int> 	    : 1 (default) mutiple registrations run in parallel in the cluster; 
			      2 multiple registrations run in parallel but in the same linux machine;
			      3 multiple registrations run sequentially in the same linux machine;		    
	-affineonly	    : create the atlas using affine registration only (default: off, the program will create affine atlas and then proceed to creating deformable atlas).

########################################
Author:
	Yangming Ou, MGH, HMS (yangming.ou@mgh.harvard.edu), 2014-2016
	Yangming Ou, BCH, HMS (yangming.ou@childrens.harvard.edu), 2016-2020

HELP
exit 1
}


#-----------------------------
parse()
{
	while [ -n "$1" ]; do
		case $1 in
			-h) 
				help;
				shift 1;;			# help is called
			
		     	-indir) 
				inputfolder=$2;
				shift 2;;			# input folder

			-out) 
				outputfile=$2;
				shift 2;;			# output image

			-debug)
				debug=1;			# debug mode
				shift 1;;

			-maskdir)
				maskfolder=$2;
				shift 2;;
	
			-histop)
				histop=$2;
				shift 2;;

			-runmode)
				runmode=$2;			# default is 1, but can be 2 or 3
				shift 2;;

			-affineonly)
				affineonly=1;
				shift 1;;

			-*) 
				echo "ERROR: no such option $1";
				helpshort;;
			 *) 
				break;;
		esac
	done
}



#---------------------------------
# load library or executables, they are dependencies of this pipeline
loaddependency()
{
	AFNI_DIR=/usr/pubsw/packages/AFNI/current
	FSL_DIR=/usr/pubsw/packages/fsl/current
	DRAMMS_BIN=/autofs/homes/008/you2/Utilities/DRAMMS-1.4.3/bin
	DRAMMS_LIB=/autofs/homes/008/you2/Utilities/DRAMMS-1.4.3/lib 
	script_DIR=/autofs/homes/008/home/you2/Utilities/scripts/
	histmatch_DIR=/autofs/cluster/mi2b2/4/you2/Utilities/itkRelated/HistogramMatching/build/
	#c3d_DIR=/autofs/homes/008/you2/Utilities/c3d-0.8.2-Linux-x86_64/bin
	for dir in ${AFNI_DIR} ${FSL_DIR} ${DRAMMS_BIN} ${DRAMMS_LIB} ${script_DIR} ${histmatch_DIR} #${c3d_DIR}
	do
	    if [ -d ${dir} ]; then
	      export PATH=$PATH:${dir} 
	    fi
	done
}



##############################
##############################
## Main Script
##############################
##############################

# help if needed
if [ $# -lt 1 ]
then
	help
fi


#-------------------------------
### preparation 1: load dependencies
loaddependency
#check dependency
hasflirt=`which flirt`
hasfslmerge=`which fslmerge`
hasfslmaths=`which fslmaths`
hasdramms=`which dramms`
hasafni=`which 3dresample`
if [ -z "${hasflirt}" ]; then
	echo -e "\nError: flirt was not found.\nProgram exits.\n"
	exit 1
fi
if [ -z "${hasfslmerge}" ]; then
	echo -e "\nError: fslmerge was not found.\nProgram exits.\n"
	exit 1
fi
if [ -z "${hasfslmaths}" ]; then
	echo -e "\nError: fslmaths was not found.\nProgram exits.\n"
	exit 1
fi
if [ -z "${hasdramms}" ]; then
	echo -e "\nError: dramms was not found.\nProgram exits.\n"
	exit 1
fi
dramms_lib=`which dramms | rev | cut -d/ -f2- | rev`/../lib/
if [ -z "${hasafni}" ]; then
	echo -e "\nError: AFNI (specifically 3dresample) was not found.\nProgram exits.\n"
        exit 1
fi


#-------------------------------
### preparation 2: parse input argument
debug=0 # default
affineonly=0 # default
histop=1 # default
runmode=1 # default
maskfolder=""   # default
maxIter=8       # default, note that the ${gval_dramms} in later section is gradually changing as iterations go on. Changing maxIter may cause ${gval_dramms} to become negative, which should be avoided.
convergeThre=0.995 # default, changed from 0.999 to 0.995 on 10/16/2020
parse $*

if [ ${runmode} != 1 ]&&[ ${runmode} != 2 ]&&[ ${runmode} != 3 ]; then
	echo -e "\nError: the -runmode option must be either 1, 2 or 3. Program exits. Please reset."
	exit 1
fi

pbscluster=0
slurmcluster=0
if [ ! -z "`which pbsubmit 2>/dev/null`" ]; then
	pbscluster=1
elif [ ! -z "`which sbatch 2>/dev/null`" ]; then
	slurmcluster=1
fi

#-------------------------------
### preparation 3: get folder and prefix

outputfolder=`dirname ${outputfile}`
if [ ! -d ${outputfolder} ]; then
        mkdir -p ${outputfolder}
fi
outputfolder=`readlink -f ${outputfile}`
outputfolder=`dirname ${outputfolder}`
outputprefix=`basename ${outputfile}`
outputprefix=${outputprefix%.nii.gz}
outputfile_full=${outputfolder}/${outputprefix}.nii.gz

if [ ! -z "${maskfolder}" ]; then
	maskfolder_full=`readlink -f ${maskfolder}`
fi

inputfolder_full=`readlink -f ${inputfolder}`
echo inputfolder_full=${inputfolder_full}


#-------------------------------
### preparation 4: get the number of image files in this folder
cd ${inputfolder}
allfiles_original=`ls ${inputfolder_full}/*.nii.gz`  # this version only accepts .nii.gz files
numImg=`ls -l ${inputfolder_full}/*.nii.gz |wc -l`
echo "Totally ${numImg} image files (*.nii.gz) for atlas construction."

if [ ${numImg} -lt 1 ]; then
   echo "Error: please put at least one image file (*.nii.gz) in the folder ${folder}"
   exit 1
elif [ ${numImg} == 1 ]; then
   atlas=`ls *.nii.gz`
   echo "There is only one image file (*.nii.gz) in the folder ${folder}";
   echo "So it is used as the constructed atlas:"
   echo "Atlas = ${atlas}"
   exit
fi


# if there is a mask folder, check if every input image has a corresponding mask image
if [ ! -z "${maskfolder}" ]; then
	for i in ${allfiles_original}; do
		ibasename=`basename ${i}`
		maskfile=${maskfolder_full}/${ibasename%.nii.gz}_mask.nii.gz
		if [ ! -f ${maskfile} ]; then
			echo -e "\n\nError: the mask folder does not contain ${maskfile}. Program exits.\n\n"
			exit 1
		else
			echo "make sure ${maskfile} is binary."
			fslmaths ${maskfile} -bin ${maskfile} -odt char &
		fi
	done
	wait
fi


#-------------------------------
### preparation 4: get folder and prefix

#outputfolder=`dirname ${outputfile}`
#outputfolder=`readlink -f ${outputfile}`
#outputfolder=`dirname ${outputfolder}`
#outputprefix=`basename ${outputfile}`
#outputprefix=${outputprefix%.nii.gz}



#------------------------------
### preparation 5: histogram operations
if [ ${histop} == 1 ]; then
	tmpdir="${outputfolder}/intensitynormalizationandmatching-${outputprefix}"
	mkdir -p ${tmpdir} 2>/dev/null
	notefile=${tmpdir}/readme_intensitynormalization_histogrammatching.txt
	allmaxes=""
	allmins=""
	allranges=""
	mul=1;

	# skip ths preparation 5 if histogram operations have already been done (in case of re-run the program and resume from where it terminated)
	hmdone=1;
	for ((n=1;n<=${numImg};n++)); do
		image=`echo ${allfiles_original} | cut -d' ' -f ${n}`
	        image2=${tmpdir}/`basename ${image}`
		if [ ! -f ${image2} ]; then
			hmdone=0
			break;
		fi
	done


	if [ $hmdone == 1 ]; then # if done before (added 10/9/2020)
		echo -e "\nHistogram operations requested (-histop 1) but is already done in previous run (kept in intermediate folder $tmpdir), so skip this step in this run.\n"
		allfiles_hm=`ls ${tmpdir}/*.nii.gz` # allfiles_hm are after intensity normalization and histogram matching
	else
	# 5a. N4 bias correction and (added on 10/13/2020) reorient all images into LPI orientation, otherwise combining transformations from differnt orientations may have problems, and then reorient to MNI standard position while keeping the original image size (added on 10/13/2020)
	rn=0;
	for ((n=1;n<=${numImg};n++)); do
        	image=`echo ${allfiles_original} | cut -d' ' -f ${n}`
		oimage=${tmpdir}/`basename ${image}`
		oimage_tmp=${oimage%.nii.gz}_tmp.nii.gz
		echo "bias correction for ${image}"
		if [ ! -f ${oimage} ]; then
			echo -e "bias correction by N4\n"
			N4 -i ${image} -o ${oimage} >/dev/null 2>&1&
			echo -e "reorient to LPI orientation\n"
			3dresample -orient LPI -prefix ${oimage_tmp} -input ${oimage}
			fslreorient_mat=${tmpdir}/mat_`basename ${image%.nii.gz}`_fslreorient.mat
			fslreorient2std -m ${fslreorient_mat} ${oimage_tmp} ${oimage}
			\mv ${oimage_tmp} ${oimage}

			# if mask is input, also resample and reorient the mask file
			if [ ! -z "${maskfolder}" ]; then
				echo -e "doing the same reorientation for the input mask\n"
				inmaskfile=${maskfolder}/${subj_prefix}_mask.nii.gz
				inmaskfile_tmp=${inmaskfile%.nii.gz}_tmp.nii.gz
				inmaskfile_reoriented=${inmaskfile%.nii.gz}_reoriented.nii.gz
				3dresample -orient LPI -prefix ${inmaskfile_tmp} -input ${inmaskfile}
				flirt -in ${inmaskfile_tmp} -ref ${oimage} -out ${inmaskfile_reoriented} -init ${fslreorient_mat} -applyxfm
				\rm ${inmaskfile_tmp}
			fi


			# added on 11/20/2020. Pad a reoriented image into x-y square
			xsize=`fslinfo ${oimage} | grep dim1 | sed -n 1p | awk '{ print $2 }'`
			ysize=`fslinfo ${oimage} | grep dim2 | sed -n 1p | awk '{ print $2 }'`
			if [ $xsize -lt $ysize ]; then
				padsize=$(( ($ysize - $xsize) / 2 ))
				echo -e "pad the reoriented image in the LR direction, so xDim=${xsize} becomes ${ysize}\n"
				3dZeropad -L ${padsize} -R ${padsize} -prefix ${oimage_tmp} ${oimage}
				mv ${oimage_tmp} ${oimage}
				if [ ! -z "${maskfolder}" ]; then
					echo -e "also pad the input mask\n"
					inmaskfile_reoriented=${maskfolder}/${subj_prefix}_mask_reoriented.nii.gz
					3dZeropad -L ${padsize} -R ${padsize} -prefix ${oimage_tmp} ${inmaskfile_reoriented}
					mv ${oimage_tmp} ${inmaskfile_reoriented}
				fi					
			else		
				padsize=$(( ($xsize - $ysize) / 2 ))
				echo -e "pad the reoriented image in the PA direction, so yDim=${ysize} becomes ${xsize}\n"
				3dZeropad -P ${padsize} -A ${padsize} -prefix ${oimage_tmp} ${oimage}
                                mv ${oimage_tmp} ${oimage}
                                if [ ! -z "${maskfolder}" ]; then
					echo -e "also pad the input mask\n"
                                        inmaskfile_reoriented=${maskfolder}/${subj_prefix}_mask_reoriented.nii.gz
                                        3dZeropad -P ${padsize} -A ${padsize} -prefix ${oimage_tmp} ${inmaskfile_reoriented}
                                        mv ${oimage_tmp} ${inmaskfile_reoriented}
                                fi
			fi
			# end addition on 11/20/2020
			
			rn=$(( ${rn} + 1 ))
			if (( ${rn} % 15 == 0 )); then
				wait
			fi
		fi
	done
	wait
	# 5b. intensity normalization (normalized to geometric mean)
	for ((n=1;n<=${numImg};n++)); do
        	image=`echo ${allfiles_original} | cut -d' ' -f ${n}`
		image=${tmpdir}/`basename ${image}`
		if [ ! -f ${image} ]; then
			echo -e "\nError: please check whether N4 is functioning as expected.\n\n"
			exit 1
		fi
        	minmax=`fslstats ${image} -R`
	        min=`echo ${minmax} | awk '{ print $1 }'`
	        max=`echo ${minmax} | awk '{ print $2 }'`
	        range=`echo "${max} - ${min}" | bc -l`
	        echo -e "${image}: range=${range}\n"
	        allmins="${allmins} ${min}"
	        allmaxes="${allmins} ${max}"
	        allranges="${allranges} ${range}"
	        mul=`echo "${mul} * ${range}" | bc -l`
	done
	s=`echo "e(l(${mul})/${numImg})" |bc -l`  # ${mul}^(1/numImg), geometric mean
	if [ $s == 0 ]; then s=1; fi
	maxentropy=0
	for ((n=1;n<=${numImg};n++)); do
        	image=`echo ${allfiles_original} | cut -d' ' -f ${n}`  # original, in original input directory
	        image=${tmpdir}/`basename ${image}`                    # after N4 bias correction, in tmpdir
		image255=${image%.nii.gz}_scalingto0_255.nii.gz          # scaling to (0,255) and using dramms/ConvertImage to group long tails
		${dramms_lib}/ConvertImage -s -e --reset-scaling -t uchar -m 0 -M 255	${image} ${image255} # ${image255} has two purposes: (1) remove long tails, enhance contrast; (2) to calculate entropy using the CalculateImageSimilarity program

	        # map the (0,255) and long tail grouped histogram to the geometric mean range
		range=`echo ${allranges} | cut -d' ' -f ${n}`
	        min=`echo ${allmins} | cut -d' ' -f ${n}`
		max=`echo ${allmaxes} | cut -d' ' -f ${n}`
		ratio=`echo "${s}/${range}" | bc -l`
		oimage=${image}
		msg="${image}: s=${s}, ratio=${ratio}"
		echo -e "${msg}\n"
	        echo -e "${msg}\n" >> ${notefile}
		fslmaths ${image255} -div 255 -mul ${range} -add ${min} -mul ${ratio} ${oimage}  
		fslmaths ${image255} -abs -bin -mul ${oimage} ${oimage} # ${oimage} is the one finally used for subsequent analysis, = ratio * (${image255} * ${range}/255 + ${min})

		# calculate entropy when in (0,255) range
		entropy=`CalculateImageSimilarity ${image255} ${image255} -M | grep entropy | awk '{ print $3 }' | tr -d ,`
		\rm ${image255}
	        msg="${image}, entropy=${entropy}"
	        echo -e "${msg}\n\n\n"
	        echo -e "  ${msg}\n\n\n" >> ${notefile}
	        if [ `echo ${entropy} '>' ${maxentropy} | bc -l` == 1 ]; then
	                maxentropy=${entropy}
	                template_for_histmatch=${oimage} # use the image with the maximum entropy to provide the reference histogram
	        fi
	done
	# 5c. intensity histogram match
	msg="match histogram to the histogram of ${template_for_histmatch}"
	echo -e "\n${msg}"
	echo -e "\n${msg}" >> ${notefile}
	for ((n=1;n<=${numImg};n++)); do
	        image=`echo ${allfiles_original} | cut -d' ' -f ${n}`
	        image2=${tmpdir}/`basename ${image}`
		oimage=${image%.nii.gz}_hm.nii.gz
		MatchHistogram ${image2} ${template_for_histmatch} ${oimage}
		fslmaths ${image2} -abs -bin -mul ${oimage} ${oimage}
		\mv ${oimage} ${image2}
	done
	allfiles_hm=`ls ${tmpdir}/*.nii.gz` # allfiles_hm are after intensity normalization and histogram matching
	fi # if hm not done before (added 10/9/2020)
else # if no histogram operations
	allfiles_hm=${allfiles_original}
fi

#exit; # exit here if only to do intensity normalization and histogram matching	


# ----------------------------------------------------------------------
# if we have more than 1 images in the folder, proceed to atlas construction
# ----------------------------------------------------------------------
# ----------------------------------------------------------------------
# first, affine atlas
# ----------------------------------------------------------------------
# ----------------------------------------------------------------------

echo "========================================="
echo "========================================="
echo "Part 1/2: constructing affine atlas"
echo "========================================="
echo "========================================="

###
# before doing affine, first remove global rotation/translation differences in a **rigid** step (added 09/20/2014)
###
# determine an initial/tentative template as the one having the largest image size (until 10/9/2020)
# use the image whose size is at the 80% of all images (modified on 10/9/2020)
tmpdir="${outputfolder}/affineiteration0-${outputprefix}"
mkdir -p ${tmpdir} 2>/dev/null

sizeChart=${tmpdir}/imageSize_allImages.txt
\rm ${sizeChart} 2>/dev/null
#maximagesize=1
for ((n=1;n<=${numImg};n++))
  do
	image=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
	xsize=`fslinfo ${image} | grep "dim1" | sed -n 1p | awk '{ print $2 }'`
	ysize=`fslinfo ${image} | grep "dim2" | sed -n 1p | awk '{ print $2 }'` 
	zsize=`fslinfo ${image} | grep "dim3" | sed -n 1p | awk '{ print $2 }'`
	volx=`fslinfo ${image} | grep "dim1" | sed -n 2p | awk '{ print $2 }'`
        voly=`fslinfo ${image} | grep "dim2" | sed -n 2p | awk '{ print $2 }'`
        volz=`fslinfo ${image} | grep "dim3" | sed -n 2p | awk '{ print $2 }'`
	imagesize=$((${xsize}*${ysize}*${zsize}))
	ratio=`echo "${imagesize} / ( ${volx} * ${voly} * ${volz} )" | bc -l` # to penalize images with large slice thickness now that every image is in the LPI orientation and z direction is always the inferior-superior direction
	echo "${image} ${imagesize} ${ratio} $xsize $ysize $zsize ${volx} ${voly} ${volz}" >> ${sizeChart}
	#if [ ${imagesize} -gt ${maximagesize} ]; then
		#maximagesize=${imagesize}
		#temp=${image}
	#fi
done
idxprc80=$(( (${numImg} * 8 + 5 ) / 10  )); # find the index of the 80_prctile among all $numImg images.
temp=`cat ${sizeChart} | sort -n -k3 | sed -n ${idxprc80}p | awk '{ print $1 }'`  # this is the chosen image to start with

echo "----------------------"
echo "to remove differences in translation/rotation by rigid transformation: using the image which is at the 80-percentile for image size as the template: ${temp}..."
echo "register all subjects to the tentative template by rigid registration"

for ((n=1;n<=${numImg};n++)); do
	subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
	subj_basename=`basename ${subj}`
	subj_prefix=${subj_basename%.nii.gz}
	temp_basename=`basename ${temp}`
	temp_prefix=${temp_basename%.nii.gz}

	out=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.nii.gz
	omat=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.mat

	if [ "`basename ${subj}`" == "`basename ${temp}`" ]; then
		\cp ${subj} ${out}
		echo "1 0 0 0" > ${omat}
		echo "0 1 0 0" >> ${omat}
		echo "0 0 1 0" >> ${omat}
		echo "0 0 0 1" >> ${omat}
		temp0=${out}
	fi
	if [ ! -f ${out} ]; then
		echo ""
		#rigidcmd="flirt -in ${subj} -ref ${temp} -out ${out} -omat ${omat} -dof 6"
		rigidcmd="dramms -S ${subj} -T ${temp} -O ${out} -D ${omat} -a 4"
		echo ${rigidcmd}
		if [ ${runmode} == 1 ]; then
			if [ ${pbscluster} == 1 ]; then
				pbsubmit -q r_default -c "${rigidcmd}"
				sleep 5
			elif [ ${slurmcluster} == 1 ]; then
				cmdfile=${tmpdir}/cmd_${subj_prefix}_rigidto_${temp_prefix}.sh
				echo -e '#!/bin/bash\n' "srun ${rigidcmd}" > ${cmdfile}
				sbatch --mem=2000MB ${cmdfile}
				sleep 5
			fi
		elif [ ${runmode} == 2 ]; then
			${rigidcmd} &
		else
			${rigidcmd} 
		fi
   	fi
done
if [ ! -z "${temp0}" ]; then
	temp=${temp0}
fi

# wait until all rigid registrations finish, and then proceed to affine atlas construction
numRes=${numImg}
numOut=`ls -l ${tmpdir}/*_rigidto_${temp_prefix}.nii.gz 2>/dev/null |wc -l`
numOmat=`ls -l ${tmpdir}/*_rigidto_${temp_prefix}.mat   2>/dev/null |wc -l`
if [ ${numOut} -lt ${numRes} ]||[ ${numOmat} -lt ${numRes} ]; then
	for i in {1..1800}; do
		sleep 60    # maximum waiting time 24 hours, check every 1 minutes
		numOut=`ls -l ${tmpdir}/*_rigidto_${temp_prefix}.nii.gz 2>/dev/null |wc -l`
		numOmat=`ls -l ${tmpdir}/*_rigidto_${temp_prefix}.mat   2>/dev/null |wc -l`
		echo "${i} minutes passed, ${numOut} rigidly registered images generated, and ${numOmat} rigid matrices generated, expecting ${numRes}"
		if [ ${numOut} == ${numRes} ]&&[ ${numOmat} -ge ${numRes} ]; then 
			echo "all rigid registrations have finished, we can proceed to constructing the rigid atlas"
			break; 
		fi
	done
fi
# if by the end of the wait, not all the registration is generated, then exit with an error report.
if [ ${numOut} -lt ${numRes} ]||[ ${numOmat} -lt ${numRes} ]; then
	echo -e "\nAfter 30 hours of wait, there were only $numOut rigid registrations done while we expect $numRes. Please rerun, and the program will assume from where it terminated.\n"
	exit 1
fi

# construct rigid atlas space
# added on 10/11/2020 ---------------
# considering that some registrations may fail, then not all transformations/registered-images are useful
# remove outliers as those that had similarity with the current template outside the mean+/-2stdev (use SSD as a similarity metric since all images have been normalized histograms)
totalSSD=0
meanSSD=0
stdevSSD=0
allSSD=${tmpdir}/SSD_allSubjects_rigidto_${temp_prefix}.txt
rm $allSSD 2>/dev/null
echo -e "\nNow that all rigid registrations are done, remove outliers (cases that may have failed rigid registration)...\n"
if [ ! -f ${allSSD} ]; then
	# get all SSD for each subject's registered image, and compute meanSSD
	for ((n=1;n<=${numImg};n++)); do
		subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
	        subj_basename=`basename ${subj}`
        	subj_prefix=${subj_basename%.nii.gz}
		out=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.nii.gz
		SSD=`CalculateImageSimilarity ${out} ${temp} -D | sed -n 2p | awk ' { print $3 }'`
		echo "${out} ${SSD}" >> ${allSSD}
		#echo $n: SSD=$SSD
		totalSSD=`echo "$totalSSD + $SSD" | bc`
	done

	threSSD=$totalSSD  
	numImgUsedForAtlas=$numImg
	for dd in 1 2 3 4; do  # iterate twice to compute a compact mean and stdev, the first time removing outliers. 
		echo -e "\nget a compact mean and stdev SSD via 4 iterations. This is iteration $dd\n--------------\n\n"
		meanSSD=`echo "${totalSSD} / ${numImgUsedForAtlas}" | bc -l`	
		# compute stdev
		numImgUsedForAtlas=0
		totalSSD=0
		stdevSSD=0
		for ((n=1;n<=${numImg};n++)); do
			bsname=`cat ${allSSD} | sed -n ${n}p | awk '{ print $1 }' | rev | cut -d/ -f1 | rev`
			SSD=`cat ${allSSD} | sed -n ${n}p | awk '{ print $2 }'`
			if [ `echo "${SSD} < ${threSSD}" |bc` == 1 ]; then
				echo "$n: $bsname, SSD=$SSD, kept (in iteration $dd)"
				totalSSD=`echo "$totalSSD + $SSD" | bc`
		                stdevSSD=`echo "$stdevSSD + (${SSD} - ${meanSSD})^2" | bc`		
				numImgUsedForAtlas=$(( ${numImgUsedForAtlas} + 1 ))
			else
				echo "$n: $bsname, SSD=$SSD, not kept (in iteration $dd)"
			fi
        	done
		meanSSD=`echo "${totalSSD} / ${numImgUsedForAtlas}" | bc -l`
	        stdevSSD=`echo "sqrt(${stdevSSD} / (${numImgUsedForAtlas} - 1) )" | bc`
		threSSD=`echo "${meanSSD} + ( 2 + 0.1 * ${dd} ) * ${stdevSSD} " | bc`
		echo -e "\n iteration $dd, meanSSD=${meanSSD}, stdevSSD=${stdevSSD}, threSSD=${threSSD}, $numImgUsedForAtlas subjects left.\n"
	done
fi
#threSSD=`echo "${meanSSD} + 2 * ${stdevSSD}" | bc`
echo -e "\nmeanSSD=${meanSSD}, stdevSSD=${stdevSSD}, threSSD=${threSSD}\n"

suminvrigidmat=${tmpdir}/suminvrigidmat.mat
meaninvrigidmat=${tmpdir}/meaninvrigidmat.mat
numImgUsedForAtlas=0
allSSD_used=${tmpdir}/SSD_usedSubjects_rigidto_${temp_prefix}.txt
#rm ${allSSD_used} 2>/dev/null
if [ ! -f ${suminvrigidmat} ]||[ ! -f ${meaninvrigidmat} ]||[ ! -f ${allSSD_used} ]; then
	for ((n=1;n<=${numImg};n++)); do
        	subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
	        subj_basename=`basename ${subj}`
        	subj_prefix=${subj_basename%.nii.gz}

		out=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.nii.gz
	        omat=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.mat
		invomat=${tmpdir}/inv_${subj_prefix}_rigidto_${temp_prefix}.mat

		# added 10/12/2020, only use an image with a potentially successful registration (i.e., SSD < threSSD)
		useThisImg=0
		mat_legitimate=0
		SSD=`cat ${allSSD} | sed -n ${n}p | awk '{ print $2 }'`
		a11=`cat ${omat} | sed -n 1p | awk '{ print $1 }'`; a11=`printf '%5.5f' $a11`;
	        a22=`cat ${omat} | sed -n 2p | awk '{ print $2 }'`; a22=`printf '%5.5f' $a22`
	        a33=`cat ${omat} | sed -n 3p | awk '{ print $3 }'`; a33=`printf '%5.5f' $a33`
        	a11_legitimate=`echo "( ${a11} > 0.66 )&&( ${a11} < 1.5 )" | bc`
 	       	a22_legitimate=`echo "( ${a22} > 0.66 )&&( ${a22} < 1.5 )" | bc`
        	a33_legitimate=`echo "( ${a33} > 0.66 )&&( ${a33} < 1.5 )" | bc`
		if [ ${a11_legitimate} == 1 ]&&[ ${a22_legitimate} == 1 ]&&[ ${a33_legitimate} == 1 ]; then  mat_legitimate=1; fi
		if [ `echo " ${SSD} <= ${threSSD} " | bc` == 1 ]&&[ ${mat_legitimate} == 1 ]; then
			bsname=`cat ${allSSD} | sed -n ${n}p | awk '{ print $1 }' | rev | cut -d/ -f1 | rev`
			useThisImg=1
			numImgUsedForAtlas=$(( ${numImgUsedForAtlas} + 1 ))
			echo $bsname, SSD=$SSD, threSSD=$threSSD, kept
		fi
		if [ ${numImgUsedForAtlas} == 1 ];then 
			echo "${out} ${SSD} ${useThisImg}" > ${allSSD_used}
		else
			echo "${subj} ${SSD} ${useThisImg}" >> ${allSSD_used}  # the third column labels whether this registered image should be used for atlas
		fi


		if [ ! -f ${invomat} ]&&[ ${useThisImg} == 1 ]; then	
			dramms-defop -i ${omat} ${invomat}
		fi

		if [ $numImgUsedForAtlas == 1 ]&&[ ${useThisImg} == 1 ]; then
			cp ${invomat} ${suminvrigidmat}
		elif [ $numImgUsedForAtlas -gt 1 ]&&[ ${useThisImg} == 1 ]; then
			dramms-combine -a ${invomat} ${suminvrigidmat} ${suminvrigidmat}
		fi
	done
	#dramms-defop -d ${numImg} ${suminvrigidmat} ${meaninvrigidmat}
	dramms-defop -d ${numImgUsedForAtlas} ${suminvrigidmat} ${meaninvrigidmat}

	# --------------------------------------------
	# added on 10/14/2020 to check whether the mean inv rigid matrix is legitimate, otherwise set it identity
	# --------------------------------------------
	a11=`cat ${meaninvrigidmat} | sed -n 1p | awk '{ print $1 }'`
	a22=`cat ${meaninvrigidmat} | sed -n 2p | awk '{ print $2 }'`
	a33=`cat ${meaninvrigidmat} | sed -n 3p | awk '{ print $3 }'`
	a11_legitimate=`echo "( ${a11} > 0.9 )&&( ${a11} < 1.1 )" | bc`
	a22_legitimate=`echo "( ${a22} > 0.9 )&&( ${a22} < 1.1 )" | bc`
	a33_legitimate=`echo "( ${a33} > 0.9 )&&( ${a33} < 1.1 )" | bc`	
	if [ ${a11_legitimate} == 0 ]||[ ${a22_legitimate} == 0 ]||[ ${a33_legitimate} == 0 ]; then	# set it to the identity matrix
		echo "1 0 0 0" >  ${meaninvrigidmat}
		echo "0 1 0 0" >> ${meaninvrigidmat}
		echo "0 0 1 0" >> ${meaninvrigidmat}
 		echo "0 0 0 1" >> ${meaninvrigidmat}
	fi
	echo -e "\n-------------------\nmean inverse rigid transformation matrix\n------------------\n"
	cat ${meaninvrigidmat}
else
	numImgUsedForAtlas=`cat ${allSSD_used} | awk '{ print $3 }' | grep 1 | wc -l`
fi
echo -e "\n---------------Moving forward------------\n${numImgUsedForAtlas} out of all ${numImg} will be used for affine atlas construction \n-----------------------------------\n"
for ((n=1;n<=${numImg};n++)); do
        subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
        subj_basename=`basename ${subj}`
        subj_prefix=${subj_basename%.nii.gz}

        omat=${tmpdir}/${subj_prefix}_rigidto_${temp_prefix}.mat
	amat=${tmpdir}/${subj_prefix}_rigidto_rigidatlas.mat

	useThisImg=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
	if [ ! -f ${amat} ]&&[ ${useThisImg} == 1 ]; then 
		dramms-combine -c ${omat} ${meaninvrigidmat} ${amat}
	fi

	# warp the intensity_normalized and histogram_matched images and the mask (if given) to the rigid atlas space
	aout=${tmpdir}/${subj_prefix}_rigidto_rigidatlas.nii.gz
	if [ ! -f ${aout} ]&&[ ${useThisImg} == 1 ]; then
		dramms-warp ${subj} ${amat} ${aout} -t ${temp}  # ${temp} and the rigidatlas.nii.gz (not generated so far) share the same image and voxel sizes
	fi
	if [ ! -z "${maskfolder}" ]; then
		inmaskfile=${maskfolder_full}/${subj_prefix}_mask_reoriented.nii.gz
		amaskout=${tmpdir}/${subj_prefix}_rigidto_rigidatlas_mask.nii.gz
		if [ ! -f ${amaskout} ]&&[ ${useThisImg} == 1 ]; then
			dramms-warp ${inmaskfile} ${amat} ${amaskout} -t ${temp} -n
		fi
	fi	
done
allfiles_hm_rigid=`ls ${tmpdir}/*_rigidto_rigidatlas.nii.gz` # note on 10/12/2020: at this point, the discarded ones (due to failed registration) will not be included in the atlas and will not be included moving forward.
if [ ! -z "${maskfolder}" ]; then
	allmasks_rigid=`ls ${tmpdir}/*_rigidto_rigidatlas_mask.nii.gz`
fi
echo -e "\nrigid part of the atlas construction done!\n"
echo $allfiles_hm_rigid
#read a

#################################
# now start affine atlas
#################################
for (( iter=1; iter<=${maxIter}; iter++ )); do  
    # make a temporary directory for storing intermediate results in this iteration
    #if [ -d ${outputfolder}/affineiteration${iter}* ]; then
    #	\rm -rf ${outputfolder}/affineiteration${iter}*
    #fi
    tmpdir="${outputfolder}/affineiteration${iter}-${outputprefix}"
    mkdir -p ${tmpdir} 2>/dev/null

    # determine the initial/tentative template in this iteration
    if [ ${iter} == 1 ]; then
        echo "----------------------"
	echo "affine iteration 1: using the image having with the 80-percentile of the biggest image size as the template: temp1_init=${temp}..."
	\cp ${temp} ${tmpdir}/temp${iter}_init.nii.gz
	temp="temp${iter}_init.nii.gz"
    else
	previousiter=$((${iter}-1))
	\cp ${outputfolder}/affineiteration${previousiter}-${outputprefix}/mean_iter${previousiter}.nii.gz ${tmpdir}/temp${iter}_init.nii.gz
	temp="temp${iter}_init.nii.gz"
    echo "----------------------"
	echo "affine iteration ${iter}: using the atlas from the previous iteration as the tentative template, temp${int}_init=${temp}..."
    fi

    # affinely register all files onto the tentative template (the first iteration is rigid, to remove global difference
    echo "----------------------"
    echo "register all subjects to the tentative template"
    for ((n=1;n<=${numImgUsedForAtlas};n++))
      do
    	subj=`echo ${allfiles_hm_rigid} |cut -d' ' -f ${n}`
	if [ ${subj} != ${temp} ]; then
	   subj_basename=`basename ${subj}`
           subj_prefix=${subj_basename%.nii.gz}
	   temp_basename=`basename ${temp}`
           temp_prefix=${temp_basename%.nii.gz}

   	   out=${tmpdir}/${subj_prefix}_to_${temp_prefix}.nii.gz
	   omat=${tmpdir}/${subj_prefix}_to_${temp_prefix}.mat

	   #useThisImg=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
	   #if [ ! -f ${out} ]&&[ ${useThisImg} == 1 ]; then
	   if [ ! -f ${out} ]; then
		echo ""
		affinecmd="dramms -S ${subj} -T ${tmpdir}/${temp} -O ${out} -D ${omat} -a 2"
		echo ${affinecmd}
		if [ ${runmode} == 1 ]; then
			if [ ${pbscluster} == 1 ]; then
				pbsubmit -q max200 -c "${affinecmd}"
				sleep 5
                        elif [ ${slurmcluster} == 1 ]; then
                                cmdfile=${tmpdir}/cmd_${subj_prefix}_to_${temp_prefix}.sh
                                echo -e '#!/bin/bash\n' "srun ${affinecmd}" > ${cmdfile}
                                sbatch --mem=2000MB ${cmdfile}
				sleep 5
                        fi
		elif [ ${runmode} == 2 ]; then
			${affinecmd} &
		else
			${affinecmd}
		fi
	   fi
        fi
    done


    # wait until all affine registrations finish, and then calculate the mean
    #numRes=$((${numImg})) # modified on 10/12/2020, not all images will go into atlas, only those with successful rigid registration will be
    numRes=${numImgUsedForAtlas}
    for i in {1..2400}
      do
 	   sleep 60    # maximum waiting time 40 hours, check every 1 minute
	   numOut=`ls -l ${tmpdir}/*_to_${temp_prefix}.nii.gz 2>/dev/null |wc -l`
	   numOmat=`ls -l ${tmpdir}/*_to_${temp_prefix}.mat   2>/dev/null |wc -l`
	   echo "${i} minutes passed, ${numOut} affinely registered images generated, and ${numOmat} affine matrices generated, expecting $numRes"
	   if [ ${numOut} == ${numRes} ]&&[ ${numOmat} == ${numRes} ]; then 
		echo "all affine registrations in this iteration have finished, we can proceed to average all the affine matrix"
		break; 
	   fi
    done
    # exit with an error message if not all affine registration is done.	
    if [ $numOut -lt $numRes ]||[ ${numOmat} -lt ${numRes} ]; then
	echo -e "\nError: after 40 hours of wait, only $numOut affine registration were done out of the expected $numRes. Program terminates. Please re-run, and the program will resume from where it terminated.\n"
	exit 1
    fi


    # -----------now all affine registration have finished-------------
    # added on 11/20/2020, quality check, discard affine registrations that may have failed
    # this is done by looking at the overlap between the affinely-registered image and the tentative template image
    numImgUsedforAtlas_afteraffine=0
    #threDice=0.91
    allaffinemat=`ls ${tmpdir}/*_to_${temp_prefix}.mat`
    allaffinemat_used=""
    allAffineDiceAndSSD=${tmpdir}/DiceAndSSD_allSubjects_affineto_${temp_prefix}.txt
    allAffineDiceAndSSD_used=${tmpdir}/DiceAndSSD_usedSubjects_affineto_${temp_prefix}_used.txt
    \rm ${allAffineDiceAndSSD} ${allAffineDiceAndSSD_used} 2>/dev/null
    temp_mask=${tmpdir}/${temp%.nii.gz}_mask.nii.gz
    totalSSD=0
    totalDice=0
    if [ ! -f ${temp_mask} ]; then 
	#fslmaths ${tmpdir}/${temp} -bin ${temp_mask}; 
	prc25=`3dmaskave -mask SELF -perc 25 ${tmpdir}/${temp} | awk '{ print $1 }'` # modified on 12/2/2020, the mean image has many voxels outside the brain with very low intensities
	prc50=`3dmaskave -mask SELF -perc 50 ${tmpdir}/${temp} | awk '{ print $1 }'`
	prc95=`3dmaskave -mask SELF -perc 95 ${tmpdir}/${temp} | awk '{ print $1 }'`
	tt=`echo " ((${prc95} - ${prc25})/70 + (${prc50} - ${prc25})/25) * 4" | bc -l`
	fslmaths ${tmpdir}/${temp} -sub ${tt} -bin ${temp_mask}
    fi
    for ((n=1;n<=${numImgUsedForAtlas};n++)); do
	affinemat=`echo ${allaffinemat} |cut -d' ' -f ${n}`
	affineimage=${affinemat%.mat}.nii.gz
	affineimage_mask=${affineimage%.nii.gz}_mask.nii.gz
	if [ ! -f ${affineimage_mask} ]; then fslmaths ${affineimage} -bin ${affineimage_mask}; fi

	# calculate the dice overlap between template_mask and affineimage_mask
	dice=`c3d -verbose ${temp_mask} ${affineimage_mask} -overlap 1 | grep Dice | awk '{ print $4 }'`
	totalDice=`echo "${totalDice} + ${dice}" | bc`

	# calculate the SSD between template and affineimage
	SSD=`CalculateImageSimilarity ${affineimage} ${tmpdir}/${temp} -D | grep SSD | awk '{ print $3 }'`
	totalSSD=`echo "${totalSSD} + ${SSD}" | bc`

	msg="${affineimage} ${dice} ${SSD}"
	echo $msg
	echo "${msg}" >> ${allAffineDiceAndSSD}
	#if [ `echo "${dice} > ${threDice}" | bc` == 1 ]; then
	#	echo "`basename ${affineimage}`: dice with tentative template = $dice, dice_thre=${threDice}, kept"
	#	numImgUsedforAtlas_afteraffine=$(( ${numImgUsedforAtlas_afteraffine} + 1 ))
	#	useThisImg=1
	#	allaffinemat_used="${allaffinemat_used} ${affinemat}"
        #        echo "${affineimage} ${dice} ${useThisImg}" >> ${allAffineDiceAndSSD_used}  # the third column labels whether this affinely-registered image should be used for later affine and deformable atlas constructions
	#else
	#	echo "`basename ${affineimage}`: dice with tentative template = $dice, dice_thre=${threDice}, not kept"
	#	useThisImg=0
	#	echo "${affineimage} ${dice} ${useThisImg}" >> ${allAffineDiceAndSSD_used}
	#fi
    done

	# iteratively compute a threshold for SSD and a threshold for Dice, to remove outliers (failed affine registrations)
	threSSD=$totalSSD
	threDice=0
        numImgUsedforAtlas_afteraffine=$numImgUsedForAtlas
        for dd in 1 2 3 4; do  # iterate four times to compute a compact mean and stdev, the first time removing outliers.
                echo -e "\nget a compact mean and stdev SSD / Dice via 4 iterations. This is iteration $dd\n--------------\n\n"
                meanSSD=`echo "${totalSSD} / ${numImgUsedforAtlas_afteraffine}" | bc -l`
		meanDice=`echo "${totalDice} / ${numImgUsedforAtlas_afteraffine}" | bc -l`
                # compute stdev
                numImgUsedforAtlas_afteraffine=0
                totalSSD=0
                stdevSSD=0
		totalDice=0
		stdevDice=0
                for ((n=1;n<=${numImgUsedForAtlas};n++)); do
                        bsname=`cat ${allAffineDiceAndSSD} | sed -n ${n}p | awk '{ print $1 }' | rev | cut -d/ -f1 | rev`
			dice=`cat ${allAffineDiceAndSSD} | sed -n ${n}p | awk '{ print $2 }'` 
                        SSD=`cat ${allAffineDiceAndSSD} | sed -n ${n}p | awk '{ print $3 }'`
                        if [ `echo "${SSD} < ${threSSD}" |bc` == 1 ]&&[ `echo "${dice} > ${threDice}" |bc` == 1 ]; then
                                msg="$n: $bsname, dice=$dice, SSD=$SSD, kept (in iteration $dd)"
				echo $msg
                                totalSSD=`echo "$totalSSD + $SSD" | bc -l`
                                stdevSSD=`echo "$stdevSSD + (${SSD} - ${meanSSD})^2" | bc -l`
				totalDice=`echo "$totalDice + $dice" | bc -l`
                                stdevDice=`echo "$stdevDice + (${dice} - ${meanDice})^2" | bc -l`
                                numImgUsedforAtlas_afteraffine=$(( ${numImgUsedforAtlas_afteraffine} + 1 ))
				if [ $dd == 4 ]; then
					affinemat=`echo ${allaffinemat} |cut -d' ' -f ${n}`
					allaffinemat_used="${allaffinemat_used} ${affinemat}"
					echo "$n: $bsname, dice=$dice, SSD=$SSD, 1" >> ${allAffineDiceAndSSD_used}
				fi
                        else
                                msg="$n: $bsname, dice=$dice, SSD=$SSD, not kept (in iteration $dd)"
				echo $msg
				if [ $dd == 4 ]; then
					echo "$n: $bsname, dice=$dice, SSD=$SSD, 0" >> ${allAffineDiceAndSSD_used}
				fi
                        fi
                done

                meanSSD=`echo "${totalSSD} / ${numImgUsedforAtlas_afteraffine}" | bc -l`
                stdevSSD=`echo "sqrt(${stdevSSD} / (${numImgUsedforAtlas_afteraffine} - 1) )" | bc -l`
                threSSD=`echo "${meanSSD} + ( 2 + 0.1 * ${dd} ) * ${stdevSSD} " | bc -l`

		meanDice=`echo "${totalDice} / ${numImgUsedforAtlas_afteraffine}" | bc -l`
                stdevDice=`echo "sqrt(${stdevDice} / (${numImgUsedforAtlas_afteraffine} - 1) )" | bc -l`
                threDice=`echo "${meanDice} - ( 2 - 0.05 * ${dd} ) * ${stdevDice} " | bc -l`

                echo -e "\n iteration $dd, meanSSD=${meanSSD}, stdevSSD=${stdevSSD}, threSSD=${threSSD}"
		echo -e "                meanDice=${meanDice}, stdevDice=${stdevDice}, threDice=${threDice}, $numImgUsedforAtlas_afteraffine subjects kept.\n"
        done





    # calculate the mean inverse affine matrix
    echo "----------------------"
    echo "calculate the mean affine from tentative template to all the subjects, which will be offset in subsequent steps to get a unbiased template in this iteration."
    #allaffinemat=`ls ${tmpdir}/*_to_${temp_prefix}.mat`  #modified on 11/20/2020, because we added the above section, not all affine matrices are used, we will only use those ones stored in ${allaffinemat_used}
    #affinemat=`echo ${allaffinemat_used} |cut -d' ' -f 1`
    #affinemat_inv=${affinemat%.mat}_inv.mat
    #if [ ! -f ${affinemat_inv} ]; then
    #	    dramms-defop -i ${affinemat} ${affinemat_inv}
    #fi
    if [ ! -f ${tmpdir}/suminvaffine${iter}.mat ]; then
	    #\cp ${affinemat_inv} ${tmpdir}/suminvaffine${iter}.mat # initialize the sum affine matrix
    
    	    for ((n=1;n<=${numImgUsedforAtlas_afteraffine};n++))
		do
	     	    affinemat=`echo ${allaffinemat_used} |cut -d' ' -f ${n}`
		    echo ${affinemat}
		    if [ ! -f ${affinemat} ]; then
			echo "Error: affine matrix ${affinemat} does not exist in `pwd`"
			exit
		    fi
		    affinemat_inv=${affinemat%.mat}_inv.mat
		    dramms-defop -i ${affinemat} ${affinemat_inv}
		    if [ $n == 1 ]; then 
			\cp ${affinemat_inv} ${tmpdir}/suminvaffine${iter}.mat # initialize the sum affine matrix; 
		    else
			cmd="dramms-combine -a ${affinemat_inv} ${tmpdir}/suminvaffine${iter}.mat ${tmpdir}/tmp.mat"
		    	echo -e "\n${cmd}"
			${cmd}
	            	\mv ${tmpdir}/tmp.mat ${tmpdir}/suminvaffine${iter}.mat
		    fi
	    done
    fi
    if [ ! -f ${tmpdir}/meaninvaffine${iter}.mat ]; then
	    dramms-defop -d ${numImgUsedforAtlas_afteraffine} ${tmpdir}/suminvaffine${iter}.mat ${tmpdir}/meaninvaffine${iter}.mat
    fi


    # use the mean inverse affine matrix to move everyone to the template${iter} space
    echo "the mean inverse affine matrix at iteration ${iter} (defined in template${iter}_init space):"
    more ${tmpdir}/meaninvaffine${iter}.mat
    echo "now transform everyone to the mean geometry (template${iter} space)..."
    for ((n=1;n<=${numImgUsedForAtlas};n++))
      do
	image_rigid=`echo ${allfiles_hm_rigid} | cut -d' ' -f ${n}`
	bsname=`basename ${image_rigid}`
	mat_affine=${tmpdir}/${bsname%.nii.gz}_to_temp${iter}_init.mat
	mat_adjust=${tmpdir}/meaninvaffine${iter}.mat
	mat_affineadjust=${tmpdir}/${bsname%.nii.gz}_to_temp${iter}_init_adjusted.mat
	imageout=${tmpdir}/${bsname%.nii.gz}_to_temp${iter}_init_adjusted.nii.gz

	useThisImg=`cat ${allAffineDiceAndSSD_used} | sed -n ${n}p | awk '{ print $5 }'`
	#echo $useThisImg
	if [ $useThisImg == 1 ]; then
		# concatenation transformations (from rigid space to temp space to adjusted mean space)
		if [ ! -f ${imageout} ]; then
			cmd="dramms-combine -c ${mat_affine} ${mat_adjust} ${mat_affineadjust}"
			echo -e "\n${cmd}"
			${cmd}

			# warp the rigid image to the adjusted mean space
			cmd="dramms-warp ${image_rigid} ${mat_affineadjust} ${imageout} -t ${tmpdir}/${temp}"
			echo -e "\n${cmd}"
			${cmd}
		fi

		if [ ! -z "${maskfolder}" ]; then
			mask_rigid=${image_rigid%.nii.gz}_mask.nii.gz
			maskout=${tmpdir}/${bsname%.nii.gz}_to_temp${iter}_init_mask_adjusted.nii.gz
			if [ ! -f ${maskout} ]; then
				dramms-warp ${mask_rigid}  ${mat_affineadjust} ${maskout}  -t ${tmpdir}/${temp} -n
				fslmaths ${imageout} -mul ${maskout} ${imageout}
			fi
		fi
	fi
    done


    # compute the average intensity in the mean space in this iteration (will be mean_iter${iter}.nii.gz)
    echo "now compute the mean intensity in the mean geometry of this iteration (template${iter} space)"
    cmd="fslmerge -t ${tmpdir}/4D_iter${iter}.nii.gz ${tmpdir}/*_to_temp${iter}_init_adjusted.nii.gz"
    if [ ! -f ${tmpdir}/4D_iter${iter}.nii.gz ]; then
	    ${cmd}
    fi
    #fslmaths ${tmpdir}/4D_iter${iter}.nii.gz -Tmean ${tmpdir}/mean_iter${iter}.nii.gz
    #fslmaths ${tmpdir}/4D_iter${iter}.nii.gz -Tstd ${tmpdir}/stdev_iter${iter}.nii.gz
    if [ -f ${tmpdir}/4D_iter${iter}.nii.gz ]&&[ ! -f ${tmpdir}/mean_iter${iter}.nii.gz ]; then
	fusecmd="myTmeanTstd.sh ${tmpdir}/4D_iter${iter}.nii.gz ${tmpdir}/mean_iter${iter}.nii.gz ${tmpdir}/stdev_iter${iter}.nii.gz"
	echo -e "\n${fusecmd}\n"
	${fusecmd}
    elif [ ! -f ${tmpdir}/4D_iter${iter}.nii.gz ]; then 
	echo -e "\nFailed to merge into a 4D file. Possibly due to insufficient memory.\nThe program used was \n\n ${cmd} \n\nProgram exits.\n"
	exit 1
    fi
    if [ ! -f ${tmpdir}/mean_iter${iter}.nii.gz ]||[ ! -f ${tmpdir}/stdev_iter${iter}.nii.gz ]; then
	echo -e "\nFailed to generate mean and stdev maps. Possibly due to insufficient memory.\nThe program used was \n\n ${fusecmd} \n\nProgram exits.\n"
	exit 1
    fi
    #\rm ${tmpdir}/4D_iter${iter}.nii.gz

    # stopping criterion
    echo "-------------"
    echo "The constructed atlas of iteration ${iter} is ${tmpdir}/mean_iter${iter}.nii.gz"
    echo "-------------"
    if [ ${iter} -gt 1 ]; then
	previousatlas=`ls ${outputfolder}/affineiteration${previousiter}*/mean_iter${previousiter}.nii.gz`
	currentatlas="${tmpdir}/mean_iter${iter}.nii.gz"
	previousatlas255=`dirname ${previousatlas}`/byte_`basename ${previousatlas}`
	currentatlas255=`dirname ${currentatlas}`/byte_`basename ${currentatlas}`
	${dramms_lib}/ConvertImage -s -e --reset-scaling -t uchar -m 0 -M 255 ${previousatlas} ${previousatlas255} # ${image255} has two purposes: (1) remove long tails, enhance contrast; (2) to calculate entropy using the CalculateImageSimilarity program
	${dramms_lib}/ConvertImage -s -e --reset-scaling -t uchar -m 0 -M 255 ${currentatlas} ${currentatlas255}

	cmd="CalculateImageSimilarity ${currentatlas255} ${previousatlas255} -C"
	echo ""
	echo ${cmd}
	cc=`${cmd} | cut -d'=' -f2`
	echo "The atlas at iteration ${iter} is similar to the atlas at the previous iteration ${previousiter} with CC=${cc}"
	if [ `echo ${cc} '>' ${convergeThre} | bc -l` -eq 1 ]||[ ${iter} == ${maxIter} ]; then
		echo "------------"
		echo "iteration converges"

		# make a sub-directory to store the final results
		if [ -d ${outputfolder}/atlasconstruction_affine* ]; then
			\rm -rf ${outputfolder}/atlasconstruction_affine*
		fi
		affineatlasdir="${outputfolder}/atlasconstruction_affine-${outputprefix}"
		mkdir -p ${affineatlasdir}

		# get the concatenated affine matrix for all subjects and use it to warp the image and mask (if given)
		template=${tmpdir}/${temp}
		nn=0
		allaffineimage_used=""
		allaffineimage_masked_used=""
		for ((n=1;n<=${numImg};n++))
		  do
			subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
			useThisImg_afterrigid=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
			echo -e "\n${n}: this image ${subj} will be used after rigid?(0/1) -- ${useThisImg_afterrigid}\n"
			if [ ${useThisImg_afterrigid} == 1 ]; then
				nn=$(( $nn + 1 ))
				subj_basename=`basename ${subj}`
				mat_orig2rigidatlas=`ls ${outputfolder}/affineiteration0-${outputprefix}/${subj_basename%.nii.gz}_rigidto_rigidatlas.mat`
				mat_subj2tentatemp=`echo ${allaffinemat} |cut -d' ' -f ${nn}`
				mat_tentatemp2temp=${tmpdir}/meaninvaffine${iter}.mat
				mat_subj2temp=${affineatlasdir}/${subj_basename%.nii.gz}_inrigidatlas_to_affineatlas.mat
				mat_orig2temp=${affineatlasdir}/${subj_basename%.nii.gz}_to_affineatlas.mat
				subjaffined=${affineatlasdir}/${subj_basename%.nii.gz}_to_affineatlas.nii.gz
				#if [[ "${allaffinemat_used}" == *${mat_subj2tentatemp}* ]]; then
				#	useThisImg_afteraffine=1
				#fi
				useThisImg_afteraffine=`cat ${allAffineDiceAndSSD_used} | sed -n ${nn}p | awk '{ print $5 }'`
				if [ ! -f ${subjaffined} ]&&[ ${useThisImg_afteraffine} == 1 ]; then
					echo -e "${n}: this image ${subj} will be used after affine?(0/1) -- ${useThisImg_afteraffine}\n"
					cmd="dramms-combine -c ${mat_subj2tentatemp} ${mat_tentatemp2temp} ${mat_subj2temp}"
					echo -e "\n${cmd}\n"
					$cmd
					cmd="dramms-combine -c ${mat_orig2rigidatlas} ${mat_subj2temp} ${mat_orig2temp}"
					echo -e "\n${cmd}\n"
					$cmd
	
					echo ""
					echo "affinely transform ${subj} into atlas space by affine matrix ${mat_subj2temp}"
					cmd="flirt -in ${subj} -ref ${tmpdir}/${temp} -init ${mat_orig2temp} -applyxfm -out ${subjaffined}"
					echo -e "\n${cmd}\n"
					${cmd}

					allaffineimage_used="${allaffineimage_used} ${subjaffined}"
				fi

				# if mask exists, apply the mask
				if [ ! -z "${maskfolder}" ]&&[ ! -f ${subjaffined%.nii.gz}_masked.nii.gz ]&&[ ${useThisImg_afteraffine} == 1 ]; then
					inmaskfile=${maskfolder_full}/${subj_basename%.nii.gz}_mask_reoriented.nii.gz
					outmaskfile=${affineatlasdir}/${subj_basename%.nii.gz}_to_affineatlas_mask.nii.gz
					cmd="flirt -in ${inmaskfile} -ref ${tmpdir}/${temp} -init ${mat_orig2temp} -applyxfm -out ${outmaskfile} -interp nearestneighbour"
					echo -e "\n${cmd}\n"
					${cmd}
				
					echo -e "\napply the mask onto the affinely registered image\n"
					cmd="fslmaths ${subjaffined} -mul ${outmaskfile} ${subjaffined%.nii.gz}_masked.nii.gz"
					echo -e "\n${cmd}\n"
					${cmd}

					allaffineimage_masked_used="${allaffineimage_masked_used} ${subjaffined%.nii.gz}_masked.nii.gz"
				fi
			fi
		done

		# calculate mean and stdev in the atlas space if not already exist
		if [ ! -f ${affineatlasdir}/atlas_affine.nii.gz ]||[ ! -f ${affineatlasdir}/stdev_affine.nii.gz ]; then
			echo -e "\nconstructing the mean and stdev in the atlas space"
			if [ -z "${maskfolder}" ]; then
				#cmd="fslmerge -t ${affineatlasdir}/4D.nii.gz ${affineatlasdir}/*_to_affineatlas.nii.gz"
				cmd="fslmerge -t ${affineatlasdir}/4D.nii.gz ${allaffineimage_used}"
			else
				#cmd="fslmerge -t ${affineatlasdir}/4D.nii.gz ${affineatlasdir}/*_to_affineatlas_masked.nii.gz"
				cmd="fslmerge -t ${affineatlasdir}/4D.nii.gz ${allaffineimage_masked_used}"
			fi
			if [ ! -f ${affineatlasdir}/4D.nii.gz ]; then
				${cmd}
			fi
			#fslmaths ${affineatlasdir}/4D.nii.gz -Tmean ${affineatlasdir}/atlas_affine.nii.gz
			#fslmaths ${affineatlasdir}/4D.nii.gz -Tstd  ${affineatlasdir}/stdev_affine.nii.gz
			if [ -f ${affineatlasdir}/4D.nii.gz ]&&[ ! -f ${affineatlasdir}/atlas_affine.nii.gz ]; then
				fusecmd="myTmeanTstd.sh ${affineatlasdir}/4D.nii.gz ${affineatlasdir}/atlas_affine.nii.gz ${affineatlasdir}/stdev_affine.nii.gz"
				echo -e "\n${fusecmd}\n"
				${fusecmd}
			elif [ ! -f ${affineatlasdir}/4D.nii.gz ]; then
				echo -e "\nFailed to merge into a 4D file. Possibly due to insufficient memory. \nThe program used was \n\n ${cmd} \n\nProgram exits.\n"
				exit 1
			fi
		fi
		if [ ! -f ${affineatlasdir}/atlas_affine.nii.gz ]||[ ! -f ${affineatlasdir}/stdev_affine.nii.gz ]; then
			echo -e "\nFailed to generate mean and stdev maps. Possibly due to insufficient memory. \nThe program used was \n\n ${fusecmd} \n\nProgram exits.\n"
			exit 1
		fi
		#\rm ${affineatlasdir}/4D*

		#if [ ${debug} == 0 ]; then
		#	\rm -rf ${outputfolder}/affineiteration*
		#fi
		echo "affine atlas = ${affineatlasdir}/atlas_affine.nii.gz"
		echo "std dev map  = ${affineatlasdir}/stdev_affine.nii.gz"
		break
	fi
    fi
done # iters


if [ ${affineonly} == 1 ]; then
	echo -e "\nThe user chose to create the atlas using affine registration only ('through the -affineonly' argument).\n"
        \cp ${affineatlasdir}/atlas_affine.nii.gz ${outputfile_full}
        \cp ${affineatlasdir}/stdev_affine.nii.gz ${outputfile_full%.nii.gz}_stdev.nii.gz
	if [ ${debug} == 0 ]; then
                \rm -rf ${outputfolder}/affineiteration*
        fi
        echo ""
        echo "====================================================="
        echo "affine atlas construction has finished successfully:"
        echo "=> atlas = ${outputfile_full}"
        echo "=> std dev map  = ${outputfile_full%.nii.gz}_stdev.nii.gz"
        echo "program ends."
        exit 1
fi


# ----------------------------------------------------------------------
# ----------------------------------------------------------------------
# second, deformable atlas
# ----------------------------------------------------------------------
# ----------------------------------------------------------------------

echo "========================================="
echo "========================================="
echo "Part 2/2: constructing deformable atlas"
echo "========================================="
echo "========================================="

# get the number of image files in this folder
allfiles_hm_rigid_affine=`ls ${affineatlasdir}/*_to_affineatlas.nii.gz`  # there should be only ${numImgUsedforAtlas_afteraffine} of them
if [ ! -z "${maskfolder}" ]; then
	allmasks_rigid_affine=`ls ${affineatlasdir}/*_to_affineatlas_mask.nii.gz`
fi
echo "The input folder has ${numImg} images."
echo "allfiles (after histogram matching and rigid aligned, N=${numImgUsedForAtlas}):"
echo "${allfiles_hm_rigid}"
echo ""
echo "allaffinefiles: (N=${numImgUsedforAtlas_afteraffine})"
echo "$allfiles_hm_rigid_affine"

for (( iter=1; iter<=${maxIter}; iter++ )); do
    # make a temporary directory for storing intermediate results in this iteration
    #if [ -d ${outputfolder}/defiteration${iter}* ]; then
	#\rm -rf ${outputfolder}/defiteration${iter}*
    #fi
    tmpdir="${outputfolder}/defiteration${iter}-${outputprefix}"
    mkdir -p ${tmpdir} 2>/dev/null

    # determine the initial/tentative template in this iteration
    if [ ${iter} == 1 ]; then
	# in iteration 1, using the file having the 80th-percentile greatest image size (after histogram matching, rigid and affine registration) as the initial template
	# actually, since all images are now in the same affine atlas space, all of them share the same size, so the following will pick up the first image as the initial space to register every other image into.
	# we did not use the affine atlas since it may be too blurred.
	maximagesize=1
	for ((n=1;n<=${numImgUsedforAtlas_afteraffine};n++))
	  do
		image=`echo ${allfiles_hm_rigid_affine} |cut -d' ' -f ${n}`
		if [ -z ${image} ]; then
			echo -e "\nError: the ${n}-th out of all expected ${numImgUsedforAtlas_afteraffine} images do not exist. Program exits.\n"
			exit 1
		fi
		xsize=`fslinfo ${image} | grep "dim1" | sed -n 1p`
		xsize=`echo ${xsize} |cut -d' ' -f2`
		ysize=`fslinfo ${image} | grep "dim2" | sed -n 1p`
		ysize=`echo ${ysize} |cut -d' ' -f2`
		zsize=`fslinfo ${image} | grep "dim3" | sed -n 1p`
		zsize=`echo ${zsize} |cut -d' ' -f2`
		imagesize=$((${xsize}*${ysize}*${zsize}))
		if [ ${imagesize} -gt ${maximagesize} ]; then
			maximagesize=${imagesize}
			temp=${image}
		fi
	done
    echo "----------------------"
	echo "deformable iteration 1: using the image having the 80th-percentile biggest image size as the template: temp1_init=${temp}..."
	\cp ${temp} ${tmpdir}/temp${iter}_init.nii.gz
	temp="temp${iter}_init.nii.gz"
    else
	previousiter=$((${iter}-1))
	cp ${outputfolder}/defiteration${previousiter}-${outputprefix}/mean_iter${previousiter}.nii.gz ${tmpdir}/temp${iter}_init.nii.gz
	temp="temp${iter}_init.nii.gz"
	echo "----------------------"
	echo "deformable iteration ${iter}: using the atlas from the previous iteration as the tentative template, temp${int}_init=${temp}..."
    fi

    # deformably register all files onto the tentative template
    echo "----------------------"
    echo "register all subjects to the tentative template using dramms"
    gval_dramms=`echo "0.3 - (${iter} - 1)*0.03" | bc -l`
    for ((n=1;n<=${numImgUsedforAtlas_afteraffine};n++))
      do
    	subj=`echo ${allfiles_hm_rigid_affine} |cut -d' ' -f ${n}`
	subjbase=`basename ${subj}`
	if [ ${subj} != ${temp} ]; then
		subj_prefix=${subjbase%.nii.gz}
		temp_prefix=${temp%.nii.gz}

		out=${tmpdir}/reg_${subj_prefix}_to_${temp_prefix}.nii.gz
		def=${tmpdir}/def_${subj_prefix}_to_${temp_prefix}.nii.gz
		log=${tmpdir}/log_${subj_prefix}_to_${temp_prefix}.txt

		if [ ! -f ${out} ]; then
			if [ -z "${maskfolder}" ]; then # if no cost-function masking
				defcmd="dramms -S ${subj} -T ${tmpdir}/${temp} -O ${out} -D ${def} -g 0.35 -a 0 "
			else
				mask=${subj%.nii.gz}_mask.nii.gz
				defcmd="dramms -S ${subj} -T ${tmpdir}/${temp} -O ${out} -D ${def} -g 0.35 -a 0 --bs ${mask}"
			fi
			echo -e "\n${defcmd}\n"
			if [ ${runmode} == 1 ]; then
				if [ ${pbscluster} == 1 ]; then
					pbsubmit -n 2 -q max200 -c "${defcmd}"
					sleep 5
				elif [ ${slurmcluster} == 1 ]; then
					cmdfile=${tmpdir}/cmd_reg_${subj_prefix}_to_${temp_prefix}.sh
					echo -e '#!/bin/bash\n' "srun ${defcmd}" > ${cmdfile}
					sbatch --partition=bch-largemem --mem=17000MB ${cmdfile}
					sleep 5
				fi
			elif [ ${runmode} == 2 ]; then
				${defcmd} &
			else
				${defcmd}
			fi
		fi
	fi
    done


    # wait until all registrations finish, and then calculate the mean
    numRes=${numImgUsedforAtlas_afteraffine}
    for i in {1..600}
      do
 	   sleep 180    # maximum waiting time 30 hours, check every 3 minutes
	   numOut=`ls -l ${tmpdir}/reg_*_to_${temp_prefix}.nii.gz 2>/dev/null |wc -l`
	   numDef=`ls -l ${tmpdir}/def_*_to_${temp_prefix}.nii.gz 2>/dev/null |wc -l`
	   echo "$((${i}*3)) minutes passed, ${numOut} registered images generated, and ${numDef} deformations generated, expecting ${numImgUsedforAtlas_afteraffine}"
	   if [ ${numOut} == ${numRes} ]&&[ ${numDef} == ${numRes} ]; then 
		echo "all registrations in this iteration have finished, we can proceed to average all the deformations."
		break; 
	   fi
	   if [ $i == 600 ]&&[ ${numOut} != ${numRes} ]; then
		echo -e "\nError: We have reached the max time (30 hours), but not all ${numRes} registrations were done (indeed ${numOut} done). So, the program exits.\n"
		exit 1
	   fi
    done

    # calculate the mean deformation fields
    echo "----------------------"
    echo "calculate the mean deformation from the tentative template to all the subjects, which will be offset in subsequent steps to get a unbiased template in this iteration."
    alldeformations=`ls ${tmpdir}/def*_to_${temp_prefix}.nii.gz`
    echo -e "\nall deformations:\n"
    echo $alldeformations
    deformation=`echo ${alldeformations} |cut -d' ' -f 1`
    \cp ${deformation} ${tmpdir}/sumdeformation.nii.gz # initialize the sum deformation
    for ((n=2;n<=${numImgUsedforAtlas_afteraffine};n++))
      do
     	deformation=`echo ${alldeformations} |cut -d' ' -f ${n}`
	    echo ${deformation}
	    if [ ! -f ${deformation} ]; then
			echo "Error: deformation ${deformation} does not exist in `pwd`. Program exits."
			exit
	    fi
	echo ""
        echo dramms-combine -a ${deformation} ${tmpdir}/sumdeformation.nii.gz ${tmpdir}/tmp.nii.gz
	dramms-combine -a ${deformation} ${tmpdir}/sumdeformation.nii.gz ${tmpdir}/tmp.nii.gz
        \mv ${tmpdir}/tmp.nii.gz ${tmpdir}/sumdeformation.nii.gz
    done
    dramms-defop -d ${numImgUsedforAtlas_afteraffine} ${tmpdir}/sumdeformation.nii.gz ${tmpdir}/meandeformation.nii.gz


    # use the inverse of the mean deformation to move everyone to the template${iter} space
    echo "invert the mean deformation at iteration ${iter}, which was defined in template${iter}_init space:"
    dramms-defop -i ${tmpdir}/meandeformation.nii.gz ${tmpdir}/meandeformation_inv.nii.gz
    echo "now transform everyone to the mean geometry (template${iter} space)..."
    alldeformedimages=`ls ${tmpdir}/reg_*_to_${temp_prefix}.nii.gz`
    nn=0
    for ((n=1;n<=${numImg};n++))
      do
	image_hm=`echo ${allfiles_hm} | cut -d' ' -f ${n}`
	useThisImg_afterrigid=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
	if [ ${useThisImg_afterrigid} == 1 ]; then
		nn=$(( $nn + 1 ))
		useThisImg_afteraffine=`cat ${allAffineDiceAndSSD_used} | sed -n ${nn}p | awk '{ print $5 }'`
		if [ ${useThisImg_afteraffine} == 1 ]; then
			bsname=`basename ${image_hm}`
			mat_affine=${affineatlasdir}/${bsname%.nii.gz}_to_affineatlas.mat # original image (after intensity normalization and histogram matching) to the affine space
			def_affineatlas2temp=${tmpdir}/def_${bsname%.nii.gz}_to_affineatlas_to_temp${iter}_init.nii.gz
			def_adjust=${tmpdir}/meandeformation_inv.nii.gz
			def_affineatlas2meanspace=${tmpdir}/concdef_${bsname%.nii.gz}_to_affineatlas_to_meanspace.nii.gz
			def_hm2meanspace=${tmpdir}/concdef_${bsname%.nii.gz}_to_meanspace.nii.gz

		        imageout=${tmpdir}/warped_${bsname%.nii.gz}_to_meanspace.nii.gz

        		# concatenate transformations (from affine atlas space to temp space to adjusted mean space)
	        	cmd="dramms-combine -c ${def_affineatlas2temp} ${def_adjust} ${def_affineatlas2meanspace}"
	        	echo -e "\n${cmd}"
		        ${cmd}

			# concatenate transformations (from original space to adjusted mean space)
			cmd="dramms-combine -c ${mat_affine} ${def_affineatlas2meanspace} ${def_hm2meanspace} -f ${image_hm} -t ${affineatlasdir}/atlas_affine.nii.gz"
			echo -e "\n${cmd}"
			${cmd}

	        	# warp the original image (int_normalized and hist_matched) to the adjusted mean space
		        cmd="dramms-warp ${image_hm} ${def_hm2meanspace} ${imageout} -t ${tmpdir}/${temp}"
        		echo -e "\n${cmd}"
		        ${cmd}

       		 	if [ ! -z "${maskfolder}" ]; then
                		mask=${maskfolder_full}/${bsname%.nii.gz}_mask.nii.gz
	                	maskout=${tmpdir}/warped_${bsname%.nii.gz}_to_meanspace_mask.nii.gz
	        	        dramms-warp ${mask}  ${def_hm2meanspace} ${maskout}  -t ${tmpdir}/${temp} -n
        	        	fslmaths ${imageout} -mul ${maskout} ${imageout}
	        	fi
		fi
	fi
    done

    # compute the average intensity in the mean space in this iteration (will be mean_iter${iter}.nii.gz)
    echo -e "now compute the mean intensity in the mean geometry of this iteration (template${iter} space)"
    file4D=${tmpdir}/4D_iter${iter}.nii.gz
    cmd="fslmerge -t ${file4D} ${tmpdir}/warped*_to_meanspace.nii.gz"
    if [ ! -f ${file4D} ]; then
	    ${cmd}
    fi
    #fslmaths ${tmpdir}/4D_iter${iter}.nii.gz -Tmean ${tmpdir}/mean_iter${iter}.nii.gz
    #fslmaths ${tmpdir}/4D_iter${iter}.nii.gz -Tstd ${tmpdir}/stdev_iter${iter}.nii.gz
    if [ -f ${file4D} ]&&[ ! -f ${tmpdir}/mean_iter${iter}.nii.gz ]; then
	    fusecmd="myTmeanTstd.sh ${file4D} ${tmpdir}/mean_iter${iter}.nii.gz ${tmpdir}/stdev_iter${iter}.nii.gz"
	    echo -e "\n${fusecmd}\n"
	    ${fusecmd}
    elif [ ! -f ${file4D} ]; then
	    echo -e "\nFailed to generate ${file4D}. Possibly due to the memory issue.\n"
	    echo -e "The program was \n\n ${cmd} \n\n Program exits.\n"
	    exit 1
    fi
    if [ ! -f ${tmpdir}/mean_iter${iter}.nii.gz ]||[ ! -f ${tmpdir}/stdev_iter${iter}.nii.gz ]; then
	    echo -e "\nFailed to generate mean and stdev images in this iteration. Possibly due to insufficient memory.\n"
	    echo -e "The program was \n\n ${fusecmd} \n\n Program exits. \n"
	    exit 1
    fi
    #\rm ${file4D}

    # stopping criterion
    echo "-------------"
    echo "The constructed atlas of iteration ${iter} is ${tmpdir}/mean_iter${iter}.nii.gz"
    echo "-------------"
    if [ ${iter} -gt 1 ]; then
	previousatlas=`ls ${outputfolder}/defiteration${previousiter}*/mean_iter${previousiter}.nii.gz`
	currentatlas="${tmpdir}/mean_iter${iter}.nii.gz"
	previousatlas255=`dirname ${previousatlas}`/byte_`basename ${previousatlas}`
        currentatlas255=`dirname ${currentatlas}`/byte_`basename ${currentatlas}`
        ${dramms_lib}/ConvertImage -s -e --reset-scaling -t uchar -m 0 -M 255 ${previousatlas} ${previousatlas255} # ${image255} has two purposes: (1) remove long tails, enhance contrast; (2) to calculate entropy using the CalculateImageSimilarity program
        ${dramms_lib}/ConvertImage -s -e --reset-scaling -t uchar -m 0 -M 255 ${currentatlas} ${currentatlas255}

	cmd="CalculateImageSimilarity ${currentatlas255} ${previousatlas255} -C"
	echo ""
	echo ${cmd}
	cc=`${cmd} | cut -d'=' -f2`
	echo "The atlas at iteration ${iter} is similar to the atlas at the previous iteration ${previousiter} with CC=${cc}"
	if [ `echo ${cc} '>' ${convergeThre} | bc -l` -eq 1 ]||[ ${iter} == ${maxIter} ]; then
		echo "------------"
		echo "iteration converges"

		# make a sub-directory to store the final results
		if [ -d ${outputfolder}/atlasconstruction_deformable* ]; then
			\rm -rf ${outputfolder}/atlasconstruction_deformable*
		fi
		atlasdir="${outputfolder}/atlasconstruction_deformable-${outputprefix}"
		mkdir -p ${atlasdir}

		# get the concatenated deformation for all subjects
		# also calculate the mean deformation of all subjects
		template=${tmpdir}/${temp}
		sumdef=${atlasdir}/sumdef.nii.gz
		sumdeftemp=${atlasdir}/sumdeftemp.nii.gz
		nn=0
		nn2=0
		for ((n=1;n<=${numImg};n++))
		  do
			subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
			useThisImg_afterrigid=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
			if [ ${useThisImg_afterrigid} == 1 ]; then
				nn=$(( $nn + 1 ))
				useThisImg_afteraffine=`cat ${allAffineDiceAndSSD_used} | sed -n ${nn}p | awk '{ print $5 }'`
				if [ ${useThisImg_afteraffine} == 1 ]; then
				nn2=$(( ${nn2} + 1 ))
				subjbase=`basename ${subj}`
				affsubj=`echo ${allfiles_hm_rigid_affine} |cut -d' ' -f ${nn2}`
				affsubjbase=`basename ${affsubj}`
				mat_orig2affsubj=${affineatlasdir}/${subjbase%.nii.gz}_to_affineatlas.mat
				def_affsubj2tentatemp=`echo ${alldeformations} |cut -d' ' -f ${nn2}`
				def_tentatemp2temp=${tmpdir}/meandeformation_inv.nii.gz
				def_affsubj2temp=${atlasdir}/concdef_${affsubjbase%.nii.gz}_to_affineatlas_to_atlas.nii.gz
				def_subj2temp=${atlasdir}/concdef_${subjbase%.nii.gz}_to_atlas.nii.gz
				subjwarped=${atlasdir}/${subjbase%.nii.gz}_warpedto_atlas.nii.gz

				cmd="dramms-combine -c ${def_affsubj2tentatemp} ${def_tentatemp2temp} ${def_affsubj2temp}"
				echo -e "\n${cmd}"
				${cmd}			
			
				cmd="dramms-combine -c -f ${subj} -t ${affsubj} ${mat_orig2affsubj} ${def_affsubj2temp} ${def_subj2temp}"
				echo -e "\n${cmd}"
				${cmd}

				echo -e "\ndeform ${subj} into atlas space by the concatenated deformation ${def_subj2temp}"
				cmd="dramms-warp ${subj} ${def_subj2temp} ${subjwarped}"
				echo -e "\n${cmd}"
				${cmd}

				if [ ! -z "${maskfolder}" ]; then
					echo -e "\nalso deform ${subj%.nii.gz}_mask.nii.gz into the atlas space by the concatenated deformation ${def_subj2temp}"
					cmd="dramms-warp ${maskfolder_full}/${subjbase%.nii.gz}_mask.nii.gz ${def_subj2temp} ${subjwarped%.nii.gz}_mask.nii.gz -n"
					echo -e "\n${tmp}"
					${cmd}

					echo -e "\napply the warped mask to the warped images"
					cmd="fslmaths ${subjwarped} -mul ${subjwarped%.nii.gz}_mask.nii.gz ${subjwarped%.nii.gz}_masked.nii.gz"
					echo -e "\n${cmd}"
					${cmd}
				fi


				# calculate the sum deformation
				if [ $nn == 1 ]; then
					\cp ${def_subj2temp} ${sumdef}
				else
					dramms-combine -a ${sumdef} ${def_subj2temp} ${sumdeftemp}
					\mv ${sumdeftemp} ${sumdef}
				fi
				fi
			fi
		done


		# calculate mean and stdev intensity in the atlas space
		if [ ! -f ${atlasdir}/4D.nii.gz ]; then
			if [ -z "${maskfolder}" ]; then
				cmd="fslmerge -t ${atlasdir}/4D.nii.gz ${atlasdir}/*_warpedto_atlas.nii.gz"
			else
				cmd="fslmerge -t ${atlasdir}/4D.nii.gz ${atlasdir}/*_warpedto_atlas_masked.nii.gz"
			fi
		fi
		if [ ! -f ${atlasdir}/4D.nii.gz ]; then
			${cmd}	
		fi
		#fslmaths ${atlasdir}/4D.nii.gz -Tmean ${atlasdir}/atlas.nii.gz
		#fslmaths ${atlasdir}/4D.nii.gz -Tstd  ${atlasdir}/stdev.nii.gz
		if [ -f ${atlasdir}/4D.nii.gz ]&&[ ! -f ${atlasdir}/atlas.nii.gz ]; then
			fusecmd="myTmeanTstd.sh ${atlasdir}/4D.nii.gz ${atlasdir}/atlas.nii.gz ${atlasdir}/stdev.nii.gz"
			echo -e "\n${fusecmd}\n"
			${fusecmd}
		elif [ ! -f ${atlasdir}/4D.nii.gz ]; then
			echo -e "\nFailed to merge into a 4D file ${atlasdir}/4D.nii.gz. Possibly due to insufficient memory.\nThe program used was\n${cmd}\nProgram exits.\n"
			exit 1
		fi
		if [ ! -f ${atlasdir}/atlas.nii.gz ]||[ ! -f ${atlasdir}/stdev.nii.gz ]; then
			echo -e "\nFailed to generate the mean and stdev map. Possibly due to insufficient memory. \nThe program used was \n\n ${fusecmd} \n\nProgram exits.\n"
			exit 1
		fi
		#\rm ${atlasdir}/4D*

		# calculate mean and stdev geometry/deformation in the atlas space
		meandef=${atlasdir}/meandef.nii.gz
		stdevdef=${atlasdir}/stdevdef.nii.gz
		diffdef=${atlasdir}/diffdef.nii.gz
		sqdiffdef=${atlasdir}/sqdiffdef.nii.gz
		sumsqdiffdef=${atlasdir}/sumsqdiffdef.nii.gz
		sumsqdiffdeftemp=${atlasdir}/sumsqdiffdeftemp.nii.gz
		# mean def
		dramms-defop -d ${numImgUsedforAtlas_afteraffine} ${sumdef} ${meandef}
		nn=0
		for ((n=1;n<${numImg};n++)); do
				subj=`echo ${allfiles_hm} |cut -d' ' -f ${n}`
				useThisImg_afterrigid=`cat ${allSSD_used} | sed -n ${n}p | awk '{ print $3 }'`
				if [ ${useThisImg_afterrigid} == 1 ]; then
					nn=$(( $nn + 1 ))
					useThisImg_afteraffine=`cat ${allAffineDiceAndSSD_used} | sed -n ${nn}p | awk '{ print $5 }'`
					if [ ${useThisImg_afteraffine} == 1 ]; then
					subjbase=`basename ${subj}`
					def_subj2temp=${atlasdir}/concdef_${subjbase%.nii.gz}_to_atlas.nii.gz
					dramms-combine -s ${def_subj2temp} ${meandef} ${diffdef}
					dramms-defop -s ${diffdef} ${sqdiffdef}
					if [ $n == 1 ]; then
						\cp ${sqdiffdef} ${sumsqdiffdef}
					else
						dramms-combine -a ${sqdiffdef} ${sumsqdiffdef} ${sumsqdiffdeftemp}
						\mv ${sumsqdiffdeftemp} ${sumsqdiffdef}
					fi
					fi
				fi
		done
		dramms-defop -d ${numImgUsedforAtlas_afteraffine} ${sumsqdiffdef} ${sumsqdiffdeftemp}
		dramms-defop -r ${sumsqdiffdeftemp} ${stdevdef}
		\rm ${diffdef} ${sqdiffdef} ${sumsqdiffdef} ${sumsqdiffdeftemp}
		

		# move the atlas and stdev deformation and intensity images to the user-specified destination
		\cp ${atlasdir}/atlas.nii.gz ${outputfile_full}
		\cp ${atlasdir}/stdev.nii.gz ${outputfile_full%.nii.gz}_stdev.nii.gz
		\cp ${atlasdir}/meandef.nii.gz ${outputfile_full%.nii.gz}_meandef.nii.gz
		\cp ${atlasdir}/stdevdef.nii.gz ${outputfile_full%.nii.gz}_stdevdef.nii.gz
                if [ ${debug} == 0 ]; then
                        \rm -rf ${outputfolder}/affineiteration*
                        \rm -rf ${outputfolder}/defiteration*
                fi
		echo ""
		echo "====================================================="
		echo "affine+deformable atlas construction has finished successfully:"
		echo "=> atlas = ${outputfile_full}"
		echo "=> std dev map  = ${outputfile_full%.nii.gz}_stdev.nii.gz"
		echo "=> mean def = ${outputfile_full%.nii.gz}_meandef.nii.gz"
		echo "=> std def  = ${outputfile_full%.nii.gz}_stdevdef.nii.gz"
		echo "program ends."
		exit 1


	fi
    fi
done # iters

