#ifndef CSIgrid_h
#define CSIgrid_h

#include "CSIgrid.h"


/*********************************************************************/
CSIgrid::CSIgrid()
{
	this->DisplayName = new char[100];
	this->spacing = new double[3];
	this->rowVector = new double[3];
	this->colVector = new double[3];
	this->q = new double[3];
	this->FOV = new double[3];
	this->VOI = new double[3];
	this->dimensions = new int[3];
	this->positionVector = new double[3];
	this->GridOrigin = new double[3];
	this->OutputGrid = vtkStructuredGrid::New();
	this->rdaHdr  = new SiemensRdaReader();
	this->VoiPosition = new double[3];

	this->GridMatrix = vtkMatrix4x4::New();

	this->CellCollection = vtkActorCollection::New();

	//this->ChemicalShift = vtkTransform::New();

}
/*********************************************************************/
CSIgrid::~CSIgrid() 
{

}
/*********************************************************************/
void CSIgrid::SetFileName(char * fileName)
{
	this->rdaHdr->ReadRdaHeader(fileName);
}
/*********************************************************************/
void CSIgrid::ReadHeader()
{
	this->spacing[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_PIXEL_SIZE_ROW).c_str());
	this->spacing[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_PIXEL_SIZE_COL).c_str());
	this->spacing[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_PIXEL_SIZE_SLICE).c_str());
	
	this->dimensions[0] = atoi(this->rdaHdr->GetHeaderTag(RDA_FOV_NUM_ROWS).c_str());
	this->dimensions[1] = atoi(this->rdaHdr->GetHeaderTag(RDA_FOV_NUM_COLS).c_str());
	this->dimensions[2] = atoi(this->rdaHdr->GetHeaderTag(RDA_FOV_NUM_3D).c_str());
		
	this->FOV[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_FOV_HEIGHT).c_str());
	this->FOV[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_FOV_WIDTH).c_str());
	this->FOV[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_FOV_3D).c_str());

	this->VOI[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_READ_FOV).c_str());
	this->VOI[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_PHASE_FOV).c_str());
	this->VOI[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_THICK).c_str());

	this->positionVector[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_POS_X_VECTOR).c_str());
	this->positionVector[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_POS_Y_VECTOR).c_str());
	this->positionVector[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_POS_Z_VECTOR).c_str());

	this->rowVector[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_ROW_X_VECTOR).c_str());
	this->rowVector[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_ROW_Y_VECTOR).c_str());
	this->rowVector[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_ROW_Z_VECTOR).c_str());

	this->colVector[0] = ((double) atof(this->rdaHdr->GetHeaderTag(RDA_COL_X_VECTOR).c_str()));
	this->colVector[1] = ((double) atof(this->rdaHdr->GetHeaderTag(RDA_COL_Y_VECTOR).c_str()));
	this->colVector[2] = ((double) atof(this->rdaHdr->GetHeaderTag(RDA_COL_Z_VECTOR).c_str()));

	this->VoiPosition[0] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_POS_SAG).c_str());
	this->VoiPosition[1] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_POS_COR).c_str());
	this->VoiPosition[2] = (double) atof(this->rdaHdr->GetHeaderTag(RDA_VOI_POS_AXI).c_str());
	
	this->q[0] = this->rowVector[1] * this->colVector[2] - this->rowVector[2] * this->colVector[1];
	this->q[1] = this->rowVector[2] * this->colVector[0] - this->rowVector[0] * this->colVector[2];
	this->q[2] = this->rowVector[0] * this->colVector[1] - this->rowVector[1] * this->colVector[0];	
	
	/********** Registrate Grid to Scanner Coordinate System ************/
	/**********		Rotation Matrix		***********/
	this->GridDirCos[0][0] = this->colVector[0];	this->GridDirCos[0][1] = this->rowVector[0];	this->GridDirCos[0][2] = this->q[0];	
	this->GridDirCos[1][0] = this->colVector[1];	this->GridDirCos[1][1] = this->rowVector[1];	this->GridDirCos[1][2] = this->q[1];		
	this->GridDirCos[2][0] = this->colVector[2];	this->GridDirCos[2][1] = this->rowVector[2];	this->GridDirCos[2][2] = this->q[2];	

	ImageType::PointType temp;
	temp[0] = this->FOV[0];
	temp[1] = this->FOV[1];
	temp[2] = this->FOV[2];
	
	ImageType::PointType temp2;
	temp2 = this->GridDirCos * temp;
	
	/**********	Translation Matrix ***********/
	this->GridOrigin[0] = this->VoiPosition[0] - (temp2[0]/2.0);
	this->GridOrigin[1] = this->VoiPosition[1] - (temp2[1]/2.0);
	this->GridOrigin[2] = this->VoiPosition[2] - (temp2[2]/2.0);
	this->GridOrigin[3] = 0;		//coherent with 4x4 Matrix

	this->GridMatrix->SetElement(0,0,this->rowVector[0]);
	this->GridMatrix->SetElement(0,1,this->colVector[0]);
	this->GridMatrix->SetElement(0,2,this->q[0]);
	this->GridMatrix->SetElement(0,3,0);
	
	this->GridMatrix->SetElement(1,0,this->rowVector[1]);
	this->GridMatrix->SetElement(1,1,this->colVector[1]);
	this->GridMatrix->SetElement(1,2,this->q[1]);
	this->GridMatrix->SetElement(1,3,0);
	
	this->GridMatrix->SetElement(2,0,this->rowVector[2]);
	this->GridMatrix->SetElement(2,1,this->colVector[2]);
	this->GridMatrix->SetElement(2,2,this->q[2]);
	this->GridMatrix->SetElement(2,3,0);

	this->GridMatrix->SetElement(3,0,0);
	this->GridMatrix->SetElement(3,1,0);
	this->GridMatrix->SetElement(3,2,0);
	this->GridMatrix->SetElement(3,3,1);

}
/*********************************************************************/
void CSIgrid::Update()
{
		
	ReadHeader();

	/**********	Build Spectroscopy Grid ***********/
	double x[3]; 
	x[0] = 0;
	x[1] = 0;
	x[2] = 0;
	
	// VTK requires this to be the number of points not cells (ie 1 more point / dimension)
	int vertices[3];
	vertices[0] = this->dimensions[0]+1;
	vertices[1] = this->dimensions[1]+1;
	vertices[2] = this->dimensions[2]+1;

	// Offset here is the absolute index of each point.x->y->z	
	vtkPoints *points =	vtkPoints::New();
	points->Allocate(vertices[0]*vertices[1]*vertices[2]);
 
	for (int k=0; k < vertices[2]; k++)
	{
		for (int j=0; j < vertices[1]; j++)
		{
			for (int i=0; i < vertices[0]; i++)  //Changed j to associate with colVector and RowVector to associate with i
			{
				x[0] = (j) * this->spacing[0] * this->colVector[0] + (i) * this->spacing[1] * this->rowVector[0] + (k) * this->spacing[2] * this->q[0]; 
				x[1] = (j) * this->spacing[0] * this->colVector[1] + (i) * this->spacing[1] * this->rowVector[1] + (k) * this->spacing[2] * this->q[1]; 
				x[2] = (j) * this->spacing[0] * this->colVector[2] + (i) * this->spacing[1] * this->rowVector[2] + (k) * this->spacing[2] * this->q[2]; 

				points->InsertPoint( (i + j * vertices[0] + k * vertices[0] * vertices[1]), x[0] + this->GridOrigin[0],
										x[1] + this->GridOrigin[1], x[2] + this->GridOrigin[2]);  //Debugging, took out j direction minus signs
			}	
		}
	}
	
	vtkStructuredGrid * tempGrid = vtkStructuredGrid::New();
	tempGrid->SetDimensions(vertices);
	tempGrid->SetPoints(points);
	tempGrid->Update();

	this->OutputGrid = tempGrid;

	CreateActors();
}
/*********************************************************************/
void CSIgrid::Shift(double deltaR)
{
	double translation[3];
	translation[0] = deltaR * this->GridDirCos[0][2]; 
	translation[1] = (deltaR * this->GridDirCos[1][2]); //Took out negative sign
	translation[2] = deltaR * this->GridDirCos[2][2]; 
	
	this->ChemicalShift = vtkTransform::New();
	this->ChemicalShift->Translate(translation);
	this->ChemicalShift->Update();

	vtkTransformFilter * csFilter = vtkTransformFilter::New();
	csFilter->SetTransform(this->ChemicalShift);
	csFilter->SetInput(this->OutputGrid);
	csFilter->Update();
	
	this->OutputGrid = (vtkStructuredGrid *) csFilter->GetOutput();
}
/*********************************************************************
void CSIgrid::RigidMotion(TransformType::Pointer transform)
{
	
	vtkTransformFilter * csFilter = vtkTransformFilter::New();
	csFilter->SetTransform(transform);
	csFilter->SetInput(this->OutputGrid);
	csFilter->Update();
	
	this->OutputGrid = (vtkStructuredGrid *) csFilter->GetOutput();
}
/*********************************************************************/
double * CSIgrid::GetSpacing()
{
	return this->spacing;
}
/*********************************************************************/
char * CSIgrid::GetName()
{
	return this->DisplayName;
}
/*********************************************************************/
void CSIgrid::SetDisplayName(char * name)
{
	strcpy(this->DisplayName, name);
}
/*********************************************************************/
double * CSIgrid::GetPosition()
{
	return this->GridOrigin;
}
/*********************************************************************/
double * CSIgrid::GetGridCenter()
{
	return this->VoiPosition;
}
/*********************************************************************/
double * CSIgrid::GetRowVector()
{
	return this->rowVector;
}
/*********************************************************************/
double * CSIgrid::GetColVector()
{
	return this->colVector;
}
/*********************************************************************/
double * CSIgrid::GetFOV()
{
	return this->FOV;
}
/*********************************************************************/
MatrixType CSIgrid::GetDirCos()
{
	return this->GridDirCos;
}
/*********************************************************************/
vtkMatrix4x4 * CSIgrid::GetTransformMatrix()
{
	return this->GridMatrix;
}
/*********************************************************************/
vtkStructuredGrid * CSIgrid::GetOutput()
{
	return this->OutputGrid;
}
/*********************************************************************/
void CSIgrid::RemovePickedCell(int cellID)
{
	//iterate through PickedCell vector, find cellID, remove it
}
/*********************************************************************/
void CSIgrid::AddPickedCell(int cellID)
{
	this->PickedCellIDs.push_back(cellID);
}
/*********************************************************************/
void CSIgrid::AddPickedCells(std::vector<int> cellIDs)
{
	this->PickedCellIDs = cellIDs;
}
/*********************************************************************/
std::vector<int> CSIgrid::GetPickedCells()
{
	return this->PickedCellIDs;
}
/*********************************************************************/
void CSIgrid::CreateActors()
{

	this->CellCollection = vtkActorCollection::New();

	for (int i = 0; i < this->PickedCellIDs.size(); i++)
	{
		this->CellCollection->AddItem( GenerateCell(this->PickedCellIDs[i]) );
	}

}
/*********************************************************************/
vtkActorCollection * CSIgrid::GetActorCollection()
{
	return	this->CellCollection; 
}

/*********************************************************************/
vtkActor * CSIgrid::GenerateCell(int cellID)
{
	vtkIdList *pointids = vtkIdList::New();
	this->OutputGrid->GetCellPoints(cellID, pointids);

	int Ids[8],temp;
	for(int i=0; i < pointids->GetNumberOfIds(); i++)
	{
		Ids[i] = pointids->GetId(i);
	}

	temp = Ids[0];
	Ids[0] = Ids[1];
	Ids[1] = temp;

	temp = Ids[4];
	Ids[4] = Ids[5];
	Ids[5] = temp;

	vtkFloatingPointType v[3];
	vtkPoints *newCellPoints = vtkPoints::New();
	newCellPoints->SetNumberOfPoints(pointids->GetNumberOfIds());
	
	for (int i=0; i<pointids->GetNumberOfIds(); i++)
	{
		this->OutputGrid->GetPoint(Ids[i], v);
		newCellPoints->InsertPoint(i,v[0],v[1],v[2]); // v[0]+this->GridOrigin[0], -v[1]-this->GridOrigin[1], v[2]+this->GridOrigin[2]); //neg for display?
	}

	int dimension[3];
	dimension[0] = 2;
	dimension[1] = 2;
	dimension[2] = 2;

	/**********		Build Hexahedron and Transformation		***********/
	vtkStructuredGrid *newCell = vtkStructuredGrid::New();
	newCell->SetDimensions(dimension);
	newCell->SetPoints(newCellPoints);

	vtkGeometryFilter *geometryfilter = vtkGeometryFilter::New();
	geometryfilter->SetInput(newCell);  

	vtkMatrix4x4 * inverse = vtkMatrix4x4::New();
	inverse->Invert(this->GridMatrix,inverse);

	vtkTransform *newTransform = vtkTransform::New();
	newTransform->Identity(); // SetMatrix(this->GridMatrix);

	vtkTransformPolyDataFilter *polyData = vtkTransformPolyDataFilter::New();
	polyData->SetTransform(newTransform);
	polyData->SetInput(geometryfilter->GetOutput());
	polyData->Update();

	vtkPolyDataMapper *mapper = vtkPolyDataMapper::New();
	mapper->SetInput(polyData->GetOutput());

	//Keep track of 
	vtkActor * actor = vtkActor::New();
	actor->SetMapper(mapper);
	actor->GetProperty()->SetColor(1,1,0);
	actor->GetProperty()->SetOpacity(0.4);

	/**********		Visualization.		***********/
	/*m_Renderer->AddActor(m_CellActors[index]);	*/

	return actor; //newCell;
	
}

/*********************************************************************/
#endif
