/*=========================================================================

Program:   BRAINS (Brain Research: Analysis of Images, Networks, and Systems)
Module:    $RCSfile: $
Language:  C++
Date:      $Date: 2008/11/12 14:53:40 $
Version:   $Revision: 1.9 $

Copyright (c) Iowa Mental Health Clinical Research Center. All rights reserved.
See BRAINSCopyright.txt or http://www.psychiatry.uiowa.edu/HTML/Copyright.html
for details.

This software is distributed WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the above copyright notices for more information.

Input Example:

./MixtureStatisticOptimizer --inputFirstVolume T1.nii.gz --inputSecondVolume T2.nii.gz --inputMaskVolume brainMask.nii.gz[optional] --desiredMean 10000 [optional] --desiredVariance 0 [optional] --seed "128,128,128" [optional]  --outputImage mush_2.nii.gz [optional] --outputMask mask_2.nii.gz [optional] --outputWeightsFile weights.txt [optional] --boundingBoxSize "90,60,75" [optional] --boundingBoxStart "83,113,80" [optional]

Minimal Input Example:
./MixtureStatisticOptimizer --inputFirstVolume T1.nii.gz --inputSecondVolume T2.nii.gz


=========================================================================*/


#include "BRAINSMushPrimary.h"
//#include "BRAINSMushPrimaryCLP.h"

//#include "itkBrains2MaskImageIOFactory.h"
//
#include "itkBinaryDilateImageFilter.h"
#include "itkBinaryErodeImageFilter.h"
#include "itkBinaryThresholdImageFilter.h"
#include "itkConnectedComponentImageFilter.h"
#include "itkConnectedThresholdImageFilter.h"
//#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIterator.h"
#include "itkLabelStatisticsImageFilter.h"
#include "itkRelabelComponentImageFilter.h"
#include "itkThresholdImageFilter.h"

#include "FindLargestForgroundFilledMask.h"
#include "itkMixtureStatisticCostFunction.h"
#include "itkLevenbergMarquardtOptimizer.h"

#include "itkCastImageFilter.h"
#include <itkAndImageFilter.h>

//
//#include <fstream>
//#include <math.h>
//#include <string>

#define PR(x) std::cout << #x " = " << x << "\n"; //a simple print macro for use when debugging


MushImageType::Pointer BRAINSMushPrimary(FloatImageType::Pointer firstImage, FloatImageType::Pointer secondImage)
{
 
  double lowerThresholdFactor = 1.2431;
  double upperThresholdFactor = 0.71956;
  double lowerThresholdFactorPre = 1.485;
  double upperThresholdFactorPre = 0.62;
  int boundingBoxSize[3] = {90,60,75};
  int boundingBoxStart[3] = {83,113,80};
  int seed[3] = {128,128,128};

  
 
  typedef itk::CastImageFilter<FloatImageType,MaskType>							FloatMaskFilterType;

  MaskType::Pointer maskImage = MaskType::New();
  MaskType::Pointer firstImageMask = MaskType::New();
  MaskType::Pointer secondImageMask = MaskType::New();
  MushImageType::Pointer resultImage = MushImageType::New();
  //ImageType::Pointer resultImage2 = ImageType::New();
    
  FloatMaskFilterType::Pointer maskfilter1 = FloatMaskFilterType::New();
  FloatMaskFilterType::Pointer maskfilter2 = FloatMaskFilterType::New();
  
  maskfilter1->SetInput(firstImage);
  maskfilter1->Update();
  //firstImageMask = maskfilter1->GetOutput();

  maskfilter2->SetInput(secondImage);
  maskfilter2->Update();
 // secondImageMask = maskfilter2->GetOutput();

  firstImageMask = FindLargestForgroundFilledMask<MaskType>(maskfilter1->GetOutput(), 0, 5);
  secondImageMask = FindLargestForgroundFilledMask<MaskType>(maskfilter2->GetOutput(), 0, 5);

  typedef itk::AndImageFilter<MaskType,MaskType,MaskType>    ANDFilterType;

  ANDFilterType::Pointer andFilter = ANDFilterType::New();
  andFilter->SetInput1(firstImageMask);
  andFilter->SetInput2(secondImageMask);
  andFilter->Update();
  maskImage = andFilter->GetOutput();




  
  //Steve said this step is no longer needed.
  //maskImage = resultImage;
  //GenerateBrainVolume(firstImage, secondImage, maskImage, lowerThresholdFactor, upperThresholdFactor, resultImage2);
  
  //typedef itk::ImageFileWriter<MaskType>					WriterType;	
  //WriterType::Pointer maskWriter = WriterType::New();
  //maskWriter->SetInput(firstImageMask);
  //maskWriter -> SetFileName( "E:\\Development\\output\\mask1.nii");
  //
  //try
  //  {
  //  maskWriter->Update();
  //  }
  //catch( itk::ExceptionObject exp )
  //  {
  //  std::cerr << "Exception caught !" << std::endl;
  //  std::cerr << exp << std::endl;
  //  }

  ////typedef itk::ImageFileWriter<MaskImageType>					WriterType;	
  //WriterType::Pointer maskWriter2 = WriterType::New();
  //maskWriter2->SetInput(secondImageMask);
  //maskWriter2->SetFileName( "E:\\Development\\output\\mask2.nii");
  //
  //try
  //  {
  //  maskWriter2->Update();
  //  }
  //catch( itk::ExceptionObject exp )
  //  {
  //  std::cerr << "Exception caught !" << std::endl;
  //  std::cerr << exp << std::endl;
  //  }

  GenerateBrainVolume(firstImage, secondImage, maskImage, lowerThresholdFactorPre, upperThresholdFactorPre, resultImage);

  return resultImage;

 }

void GenerateBrainVolume(FloatImageType::Pointer &firstImage, FloatImageType::Pointer &secondImage, 
						 MaskType::Pointer &maskImage, double lowerThresholdFactor, 
						 double upperThresholdFactor, MushImageType::Pointer &resultImage)
{
  double desiredMean = 10000.0;
  double desiredVariance = 0.0;
  //------------------------------------------------------------------------------------
  // Send to Optimizer
  FloatImageType::Pointer mixtureImage = FloatImageType::New();
  mixtureImage = MixtureOptimizer(firstImage, secondImage, maskImage, desiredMean, desiredVariance);

  //------------------------------------------------------------------------------------
 
  typedef itk::ImageFileWriter<MushImageType>					WriterType;	
  WriterType::Pointer maskWriter = WriterType::New();

  double mean;
  double upper;
  double lower;

  //if(inputMaskVolume == "no_mask_exists")
  //  {
  //  //------------------------------------------------------------------------------------
  //  //Draw a cuboid box of fixed dimensions inside the brain and use it to determine mean and bounds

  //  int voxelCount = 0;
  //  double signalTotal = 0.0;

  //  MaskImageType::RegionType regionOfInterest;
  //  MaskImageType::RegionType::SizeType regionOfInterestSize;
  //  MaskImageType::RegionType::IndexType regionOfInterestStart;

  //  regionOfInterestSize[0] = boundingBoxSize[0]; 
  //  regionOfInterestSize[1] = boundingBoxSize[1];
  //  regionOfInterestSize[2] = boundingBoxSize[2];   

  //  regionOfInterest.SetSize(regionOfInterestSize);
 
  //  regionOfInterestStart[0] = boundingBoxStart[0];
  //  regionOfInterestStart[1] = boundingBoxStart[1];
  //  regionOfInterestStart[2] = boundingBoxStart[2];

  //  regionOfInterest.SetIndex(regionOfInterestStart);

  //  ConstFloatIteratorType volumeIt( mixtureImage, regionOfInterest);
  //  ConstMaskIteratorType labelIt ( maskImage, regionOfInterest);

  //  for (volumeIt.GoToBegin(), labelIt.GoToBegin(); !volumeIt.IsAtEnd(), !labelIt.IsAtEnd(); ++volumeIt, ++labelIt)
  //    {
  //    MaskPixelType labelValue = labelIt.Get();
  //    if (labelValue == 1)
  //      {
  //      PixelType signalValue = volumeIt.Get( );
  //      signalTotal += signalValue;
  //      }
  //    voxelCount++;
  //    }
  //  PR(voxelCount);
  //  PR(signalTotal);
  //  mean = signalTotal/voxelCount;
  //  //these definitions use magic numbers obtained through manual thresholding and experimentation
  //  lower = (mean/lowerThresholdFactor);
  //  upper = (mean/upperThresholdFactor);
  //  }
  //else
  //  {
    //------------------------------------------------------------------------------------	
    //Binary erosion to generate initial brain mask



    //------------------------------------------------------------------------------------
    //Perform binary threshold on image
    //std::cout << "---------------------------------------------------" << std::endl;
    //std::cout << "Performing Initial Binary Threshold..." << std::endl;
    //std::cout << "---------------------------------------------------" << std::endl << std::endl;


    typedef itk::BinaryThresholdImageFilter <MaskType, MushImageType> BinaryThresholdMaskFilterType;
    BinaryThresholdMaskFilterType::Pointer threshToBrainCoreMask = BinaryThresholdMaskFilterType::New();
    threshToBrainCoreMask->SetInput( maskImage );
    threshToBrainCoreMask->SetLowerThreshold(1);
    threshToBrainCoreMask->SetUpperThreshold(1);
    threshToBrainCoreMask->SetInsideValue(1);
    threshToBrainCoreMask->SetOutsideValue(0);

    try
      {
      threshToBrainCoreMask->Update();
      }
    catch( itk::ExceptionObject & excp )
      {
      std::cerr << "Exception caught ! " << std::endl;
      std::cerr << excp << std::endl;
      //return EXIT_FAILURE;
      }


	
    typedef itk::BinaryErodeImageFilter < MushImageType, MushImageType, StructuringElementType> binaryErodeFilterType;

    int erosionValue = 7;

	binaryErodeFilterType::Pointer initialMaskImage = binaryErodeFilterType::New();

    StructuringElementType structuringElement;
    structuringElement.SetRadius(erosionValue);
    structuringElement.CreateStructuringElement();
    initialMaskImage->SetKernel(structuringElement);
    initialMaskImage->SetInput( threshToBrainCoreMask->GetOutput() );

    //Templating requires different writer to output mask images (type short for initialMaskImage vs type float for mixtureImage)

    try
      {
      initialMaskImage->Update();
      }
    catch( itk::ExceptionObject exp )
      {
      std::cerr << "Exception caught !" << std::endl;
      std::cerr << exp << std::endl;
      }

    //------------------------------------------------------------------------------------
    //Obtain mean of image; calculate lower and upper bounds

    typedef itk::LabelStatisticsImageFilter<FloatImageType, MushImageType > LabelFilterType;
    LabelFilterType::Pointer labelFilter = LabelFilterType::New();
    labelFilter->SetInput (mixtureImage);
    labelFilter->SetLabelInput (initialMaskImage->GetOutput());

    try
      {
      labelFilter->Update();
      }
    catch( itk::ExceptionObject & excp )
      {
      std::cerr << "Exception caught ! " << std::endl;
      std::cerr << excp << std::endl;
      //return EXIT_FAILURE;
      }

    //typedef LabelFilterType::RealType  StatisticRealType;
    mean     =  labelFilter->GetMean( 1 );

    //these definitions use magic numbers obtained through manual thresholding and experimentation
    lower = (mean/lowerThresholdFactor);
    upper = (mean/upperThresholdFactor);
   // }


  typedef itk::BinaryThresholdImageFilter <FloatImageType, MushImageType> BinaryThresholdFilterType;
  BinaryThresholdFilterType::Pointer threshToHeadMask = BinaryThresholdFilterType::New();
  threshToHeadMask->SetInput(mixtureImage);
  threshToHeadMask->SetLowerThreshold(lower);
  threshToHeadMask->SetUpperThreshold(upper);
  threshToHeadMask->SetInsideValue(1);
  threshToHeadMask->SetOutsideValue(0);

  try
    {
    threshToHeadMask->Update();
    }
  catch( itk::ExceptionObject & excp )
    {
    std::cerr << "Exception caught ! " << std::endl;
    std::cerr << excp << std::endl;
    //return EXIT_FAILURE;
    }


  typedef itk::BinaryDilateImageFilter      < MushImageType, MushImageType, StructuringElementType > BinaryDilateFilterType;
  typedef itk::BinaryErodeImageFilter       < MushImageType, MushImageType, StructuringElementType > BinaryImageErodeFilterType;

  double ClosingSize = 6;

  const FloatImageType::SpacingType& spacing = mixtureImage->GetSpacing();

  // Compute minumum object size as the number of voxels in the structuring element, an ellipsoidal ball.
  const double FourThirdsPi = 3.141592653589793238459 * 1.333333333333333333333;
  double ClosingElementVolume = FourThirdsPi * ClosingSize * ClosingSize * ClosingSize;
  double VoxelVolume = spacing[0] * spacing[1] * spacing[2];
  int MinimumObjectSize=static_cast<int>( ClosingElementVolume / VoxelVolume );

  //Define binary erosion and dilation structuring element
  StructuringElementType ball;
  StructuringElementType::SizeType ballSize;
  for(int d=0;d<3;d++)
    {
    ballSize[d]=static_cast<int>((0.5*ClosingSize)/spacing[d]);
    }
  ball.SetRadius(ballSize);
  ball.CreateStructuringElement();



  BinaryImageErodeFilterType::Pointer binaryErodeFilter = BinaryImageErodeFilterType::New();
  binaryErodeFilter->SetErodeValue(1);
  binaryErodeFilter->SetKernel( ball );

  //------------------------------------------------------------------------------------
  //Update the ITK pipeline
  try
    {
    MushImageType::Pointer thresholdOutput = threshToHeadMask->GetOutput();
    thresholdOutput->DisconnectPipeline();

    binaryErodeFilter->SetInput(thresholdOutput);
    binaryErodeFilter->Update();
    }
  catch( itk::ExceptionObject exp )
    {
    std::cerr << "Exception caught !" << std::endl;
    std::cerr << exp << std::endl;
    }



  typedef itk::BinaryThresholdImageFilter <MaskType, MushImageType> BinaryThresholdMaskFilterType;
  BinaryThresholdMaskFilterType::Pointer threshToBrainCoreMask2 = BinaryThresholdMaskFilterType::New();
  threshToBrainCoreMask2->SetInput( binaryErodeFilter->GetOutput() );
  threshToBrainCoreMask2->SetLowerThreshold(1);
  threshToBrainCoreMask2->SetUpperThreshold(1);
  threshToBrainCoreMask2->SetInsideValue(1);
  threshToBrainCoreMask2->SetOutsideValue(0);

  try
    {
    threshToBrainCoreMask2->Update();
    }
  catch( itk::ExceptionObject & excp )
    {
    std::cerr << "Exception caught ! " << std::endl;
    std::cerr << excp << std::endl;
    //return EXIT_FAILURE;
    }

  typedef itk::ConnectedComponentImageFilter< MushImageType, MushImageType > ConnectedComponentFilterType;
  typedef itk::RelabelComponentImageFilter  < MushImageType, MushImageType > RelabelComponentFilterType;
  typedef itk::ConnectedThresholdImageFilter< MushImageType, MushImageType > ConnectedThresholdFilterType;
  typedef itk::ThresholdImageFilter <MushImageType> ThresholdFilterType;
  //alternate definition to allow for binary thresholding of a mask image
  typedef itk::BinaryThresholdImageFilter <MushImageType, MushImageType> MaskBinaryThresholdFilterType;

  BinaryThresholdFilterType::Pointer threshold = BinaryThresholdFilterType::New();
  ConnectedComponentFilterType::Pointer filter = ConnectedComponentFilterType::New();
  RelabelComponentFilterType::Pointer relabel = RelabelComponentFilterType::New();

  ThresholdFilterType::Pointer LargestFilter = ThresholdFilterType::New();
  LargestFilter->SetOutsideValue(0);
  LargestFilter->ThresholdAbove(1);
  LargestFilter->ThresholdBelow(1);

  //------------------------------------------------------------------------------------
  //Update the ITK pipeline

  try
    {
      {
      MushImageType::Pointer threshToBrainCoreOutput = threshToBrainCoreMask2->GetOutput();
      threshToBrainCoreOutput->DisconnectPipeline();

      filter->SetInput (threshToBrainCoreOutput);  //ConnectedComponentFilter
      filter->Update();
      }


      {
      MushImageType::Pointer filterOutput = filter->GetOutput();
      filterOutput->DisconnectPipeline();

      relabel->SetInput( filterOutput );

      if (MinimumObjectSize > 0)
        {
        relabel->SetMinimumObjectSize( MinimumObjectSize );
        std::cerr << "MinimumObjectSize: " << MinimumObjectSize << std::endl;
        }

      relabel->Update();

      unsigned short numObjects = relabel->GetNumberOfObjects();
      std::cout << "Removed " << numObjects-1 << " smaller objects." << std::endl << std::endl;
      }


      {
      MushImageType::Pointer relabelOutput = relabel->GetOutput();
      relabelOutput->DisconnectPipeline();

      LargestFilter->SetInput(relabelOutput);
      LargestFilter->Update();
      }
    }
  catch( itk::ExceptionObject exp )
    {
    std::cerr << "Exception caught !" << std::endl;
    std::cerr << exp << std::endl;
    }


  //------------------------------------------------------------------------------------
  //Binary dilation
  //std::cout << "---------------------------------------------------" << std::endl;
  //std::cout << "Dilating largest filled region..." << std::endl;
  //std::cout << "---------------------------------------------------" << std::endl << std::endl;
  //
  
  BinaryDilateFilterType::Pointer binaryDilateFilter = BinaryDilateFilterType::New();
  binaryDilateFilter->SetDilateValue(1);
  binaryDilateFilter->SetKernel( ball );

  //------------------------------------------------------------------------------------
  //Update the ITK pipeline
  try
    {
    MushImageType::Pointer largestFilterOutput = LargestFilter->GetOutput();
    largestFilterOutput->DisconnectPipeline();

    binaryDilateFilter->SetInput(largestFilterOutput);
    binaryDilateFilter->Update();
    }
  catch( itk::ExceptionObject exp )
    {
    std::cerr << "Exception caught !" << std::endl;
    std::cerr << exp << std::endl;
    }


  //------------------------------------------------------------------------------------
  //Perform binary threshold on image
  //std::cout << "---------------------------------------------------" << std::endl;
  //std::cout << "Performing Special Binary Threshold..." << std::endl;
  //std::cout << "---------------------------------------------------" << std::endl << std::endl;
  //
  
  BinaryThresholdMaskFilterType::Pointer threshToClosureMask = BinaryThresholdMaskFilterType::New();
  threshToClosureMask->SetInput( binaryDilateFilter->GetOutput() );
  threshToClosureMask->SetLowerThreshold(1);
  threshToClosureMask->SetUpperThreshold(1);
  threshToClosureMask->SetInsideValue(1);
  threshToClosureMask->SetOutsideValue(0);

  try
    {
    threshToClosureMask->Update();
    }
  catch( itk::ExceptionObject & excp )
    {
    std::cerr << "Exception caught ! " << std::endl;
    std::cerr << excp << std::endl;
    //return EXIT_FAILURE;
    }

  MushImageType::Pointer dilatedOutput = threshToClosureMask->GetOutput();
  dilatedOutput->DisconnectPipeline();
  
  resultImage = FindLargestForgroundFilledMask<MushImageType>(dilatedOutput, 0, 5);
  return;
}

MushImageType::Pointer GenerateInitializerRegion( FloatImageType::Pointer &referenceImage)
{
  MushImageType::Pointer initializeMask = MushImageType::New();

  initializeMask->SetRegions(referenceImage->GetLargestPossibleRegion());
  //initializeMask->SetSpacing(referenceImage->GetSpacing());
  //initializeMask->SetOrigin(referenceImage->GetOrigin());
  //initializeMask->SetDirection(referenceImage->GetDirection());
  initializeMask->CopyInformation(referenceImage);
  initializeMask->Allocate();

  MushImageType::RegionType regionOfInterest;
  MushImageType::RegionType::IndexType regionStart;
  MushImageType::RegionType::SizeType regionSize;

  FloatImageType::RegionType::SizeType imageSize;
  imageSize = referenceImage->GetLargestPossibleRegion().GetSize();
 
  regionSize[0] = 90;
  regionSize[1] = 60;
  regionSize[2] = 76;

  regionStart[0] = imageSize[0]/2 - regionSize[0]/2; //83;
  regionStart[1] = imageSize[1]/2 - regionSize[1]/2; //113;
  regionStart[2] = imageSize[2]/2 - regionSize[2]/2; //80;



  regionOfInterest.SetSize(regionSize);
  regionOfInterest.SetIndex(regionStart);

  typedef itk::ImageRegionIterator<MushImageType>		IteratorType;
  IteratorType initializeIt ( initializeMask, regionOfInterest);

  //sets a binary cuboid mask
  for (initializeIt.GoToBegin(); !initializeIt.IsAtEnd(); ++initializeIt)
    {
    initializeIt.Set(1);
    }

  return initializeMask;
}


FloatImageType::Pointer MixtureOptimizer(FloatImageType::Pointer &firstImage,
  FloatImageType::Pointer &secondImage,
  MaskType::Pointer &maskImage,
  double desiredMean,
  double desiredVariance)
{
  typedef itk::MixtureStatisticCostFunction<FloatImageType, FloatImageType> MixtureStatisticCostFunctionType;
  MixtureStatisticCostFunctionType::Pointer twoByTwoCostFunction = MixtureStatisticCostFunctionType::New();
  twoByTwoCostFunction->SetDesiredMean( desiredMean );
  twoByTwoCostFunction->SetDesiredVariance( desiredVariance );
  twoByTwoCostFunction->SetFirstImage( firstImage );
  twoByTwoCostFunction->SetSecondImage( secondImage );
  twoByTwoCostFunction->SetImageMask( maskImage );
  twoByTwoCostFunction->Initialize( 1 );

  std::cout << "---------------------------------------------------" << std::endl;
  std::cout << "Initialized MixtureStatisticCostFunction! " << std::endl;
  std::cout << "---------------------------------------------------" << std::endl << std::endl;

  double firstMean = twoByTwoCostFunction->GetSumOfFirstMaskVoxels() / twoByTwoCostFunction->GetNumberOfMaskVoxels();
  double secondMean = twoByTwoCostFunction->GetSumOfSecondMaskVoxels() / twoByTwoCostFunction->GetNumberOfMaskVoxels();
  double jointFactor = 1.0 / (firstMean + secondMean);

  typedef itk::LevenbergMarquardtOptimizer LevenbergMarquardtOptimizerType;
  LevenbergMarquardtOptimizerType::Pointer twoByTwoOptimizer = LevenbergMarquardtOptimizerType::New();
  twoByTwoOptimizer->SetUseCostFunctionGradient(false);
  twoByTwoOptimizer->SetCostFunction( twoByTwoCostFunction );
  LevenbergMarquardtOptimizerType::ParametersType initialParameters(2);
  initialParameters[0] = firstMean * jointFactor;
  initialParameters[1] = secondMean * jointFactor;
  twoByTwoOptimizer->SetInitialPosition( initialParameters );

  std::cout << "---------------------------------------------------" << std::endl;
  std::cout << "Updating Levenberg-Marquardt Optimizer... " << std::endl;
  std::cout << "---------------------------------------------------" << std::endl << std::endl;

  try
    {
    twoByTwoOptimizer->StartOptimization();
    }
  catch( itk::ExceptionObject exp )
    {
    std::cerr << "LMO FAIL" << std::endl;
    std::cerr << "Exception caught !" << std::endl;
    std::cerr << exp << std::endl;
    }
  // End of Sending to Optimizer
  //------------------------------------------------------------------------------------

  LevenbergMarquardtOptimizerType::ParametersType optimalParameters = twoByTwoOptimizer->GetCurrentPosition();
  LevenbergMarquardtOptimizerType::MeasureType optimalMeasures = twoByTwoOptimizer->GetValue();

  std::cout << "---------------------------------------------------" << std::endl;
  std::cout << "Obtained Output from Levenberg-Marquardt Optimizer!" << std::endl;
  std::cout << "---------------------------------------------------" << std::endl << std::endl;

  //------------------------------------------------------------------------------------
  // Save the optimization:  one line with the image coeficients in order,
  // and one line with the error measure ito desired mean and variance.  Print it all out, too.
  const double firstWeight = optimalParameters[0];
  const double secondWeight = optimalParameters[1];

  std::cout << "First Weight:  " << firstWeight << std::endl;
  std::cout << "Second Weight:  " << secondWeight << std::endl;
  std::cout << "Optimality of Mean:  " << optimalMeasures[0] << std::endl;
  std::cout << "Optimality of Variance:  " << optimalMeasures[1] << std::endl << std::endl;

  // write a text file named outputWeightsFile


  //declare and compute mixtureImage.

  FloatImageType::Pointer mixtureImage =  FloatImageType::New();
  mixtureImage->SetRegions( firstImage->GetLargestPossibleRegion() );
  mixtureImage->SetSpacing( firstImage->GetSpacing() );
  mixtureImage->SetOrigin( firstImage->GetOrigin() );
  mixtureImage->SetDirection( firstImage->GetDirection() );
  mixtureImage->Allocate();

	typedef itk::ImageRegionConstIterator<FloatImageType> ConstFloatIteratorType;
  ConstFloatIteratorType firstIt( firstImage, firstImage->GetRequestedRegion() );
  ConstFloatIteratorType secondIt( secondImage, secondImage->GetRequestedRegion() );

  typedef itk::ImageRegionIterator<FloatImageType> MixtureIteratorType;
  MixtureIteratorType mixtureIt( mixtureImage, mixtureImage->GetRequestedRegion() );

  for (mixtureIt.GoToBegin(), firstIt.GoToBegin(), secondIt.GoToBegin(); !mixtureIt.IsAtEnd(); ++mixtureIt, ++firstIt, ++secondIt )
    {
    MushPixelType firstValue = firstIt.Get( );
    MushPixelType secondValue = secondIt.Get( );

    double mixtureValue = firstWeight * firstValue + secondWeight * secondValue;

    mixtureIt.Set( mixtureValue );
    }

  return mixtureImage;
}


