#ifndef CSIDisplay_cxx
#define CSIDisplay_cxx

#include "CSIDisplay.h"
#include "vtkGeometryFilter.h"
#include "vtkPolyDataMapper.h"
#include "vtkRenderer.h"
#include "itkFlipImageFilter.h"
#include "itkFixedArray.h"	
#include "vtkMarchingCubes.h"
#include "vtkRenderWindow.h"
#include "vtkKWImage.h"
#include "vtkKWImageIO.h"

#include "vtkProperty.h"

#include "vtkDoubleArray.h"
#include "vtkDataArray.h"
#include "vtkSmartPointer.h"
#include "vtkCellData.h"
#include "vtkLookupTable.h"
#include "vtkCallbackCommand.h"
#include "vtkRenderWindowInteractor.h"


/**************************************************************************************************************************/
CSIDisplay::CSIDisplay()
{

	this->PlaneX	= vtkImagePlaneWidget::New();
	this->PlaneY	= vtkImagePlaneWidget::New();
	this->PlaneZ	= vtkImagePlaneWidget::New();
	
	this->RenderWidget = vtkKWRenderWidget::New();
	
	this->Picker = vtkCellPicker::New();
	this->Picker->SetTolerance(0.005);

	this->Pickcallback = NULL;

	this->Image = ImageType::New();
	this->vtkImage = vtkImageData::New();
	this->Grid = vtkStructuredGrid::New();

	this->Metabolite = NULL;
	this->RatioButton = NULL;
	this->UseRatio = 0;
	this->MetabWholeRange = new double[2];
	this->MetabRange = new double[2];
	this->CRLBRange = new double[2];

	this->MetabWholeRange[0] = -1;
	this->MetabWholeRange[1] = -1;
	this->MetabRange[0] = -1;
	this->MetabRange[1] = -1;
	this->CRLBRange[0] = 0;
	this->CRLBRange[1] = 20;

	this->Component = 3;

	this->GridActor = vtkActor::New();

	this->PickActorCollection = vtkActorCollection::New();

	this->Spectroscopy = NULL;
	
}
/**************************************************************************************************************************/
CSIDisplay::~CSIDisplay()
{
	if (this->RenderWidget)
    {
		this->RenderWidget->Delete();
    }
}

/***************************************************************/
void CSIDisplay::DisplayPickedCells(int state)
{
	if (state)
	{
		this->GridActor->GetProperty()->SetOpacity(0.05);
		this->Spectroscopy->AddPickedCells(this->Pickcallback->GetCellIDs() );
		this->Spectroscopy->Update();

		this->PickActorCollection = this->Spectroscopy->GetActorCollection();
		vtkActor * actor = vtkActor::New();

		this->PickActorCollection->InitTraversal();
		actor = this->PickActorCollection->GetNextActor();

		while(actor)
		{
			actor->GetProperty()->SetColor(1,0,0);
			actor->GetProperty()->SetRepresentationToWireframe();
			actor->SetVisibility(state);

			this->RenderWidget->GetRenderer()->AddActor( actor );
			actor = this->PickActorCollection->GetNextActor();
		}
	}
	else
	{
		this->GridActor->GetProperty()->SetOpacity(1);
		this->PickActorCollection = this->Spectroscopy->GetActorCollection();
		vtkActor * actor = vtkActor::New();

		this->PickActorCollection->InitTraversal();
		actor = this->PickActorCollection->GetNextActor();
		
		while(actor)
		{
			actor->SetVisibility(state);	
			actor = this->PickActorCollection->GetNextActor();
		}
	}

	this->RenderWidget->Render();

}
/***************************************************************/
void CSIDisplay::DisplayGrids()
{
	this->Grid = this->Spectroscopy->GetOutput();

	/***********		Visualization		***********/
	vtkGeometryFilter *SGGfilter = vtkGeometryFilter::New();
	SGGfilter->SetInput(this->Grid);  

	vtkPolyDataMapper *sgridMapper = vtkPolyDataMapper::New();
	sgridMapper->SetInput( SGGfilter->GetOutput() );	
	
	this->GridActor->SetMapper(sgridMapper);
	this->GridActor->GetProperty()->SetColor(1,1,0);

	this->GridActor->GetProperty()->SetRepresentationToWireframe();

	this->RenderWidget->GetRenderer()->AddActor(this->GridActor);

	////Set Transform	
	//typedef itk::AffineTransform<double,3>						AffineTransformType;
	//AffineTransformType::Pointer transform = AffineTransformType::New();
	//transform->SetMatrix( this->Spectroscopy->GetDirCos());
	
	this->Picker->AddObserver( vtkCommand::EndPickEvent, this->Pickcallback);
	this->Picker->AddPickList(this->GridActor);
	this->RenderWidget->Render();
	
}
/**********************************************************************/
void CSIDisplay::SetCellPickerCallback( PickCellCallBack * callback)
{
	this->Pickcallback = callback;
}
/**********************************************************************/
void CSIDisplay::DisplayMasks()
{

	vtkMarchingCubes *surfaceFilter = vtkMarchingCubes::New();
	surfaceFilter->SetInput( this->Mask);
	surfaceFilter->SetValue(0,1);
	surfaceFilter->Update();
	
	vtkPolyDataMapper *surfaceMapper = vtkPolyDataMapper::New();
	surfaceMapper->SetInput(surfaceFilter->GetOutput());
	
	vtkActor	*surfaceActor = vtkActor::New();
	surfaceActor->SetMapper(surfaceMapper);
	surfaceActor->GetProperty()->SetRepresentationToSurface();
	
	this->RenderWidget->GetRenderer()->AddActor(surfaceActor);
	this->RenderWidget->ResetCamera();

}
/**************************************************************************************************************************/
void CSIDisplay::DisplayImage()
{
	typedef itk::FlipImageFilter<ImageType>						FlipImageFilterType;
	typedef itk::FixedArray<bool,3>								AxesType;

	//Flip y axes of DICOM to get coherent coordinate system with vtk
	AxesType flipaxes;
	flipaxes[0] = 0;
	flipaxes[1] = 1;
	flipaxes[2] = 0;
	
	FlipImageFilterType::Pointer flipfilter = FlipImageFilterType::New();
	flipfilter->SetInput( this->Image);
	flipfilter->SetFlipAxes(flipaxes);
	flipfilter->Update();
	
	vtkKWImage * image = vtkKWImage::New();
	image->SetITKImageBase(	this->Image); //flipfilter->GetOutput()); 
	
	ImageViewer(image->GetVTKImage());

}
/**************************************************************************************************************************/
void CSIDisplay::Update()
{
	if (this->Spectroscopy)	DisplayGrids();
	if (this->Image) DisplayImage();
	if (this->Metabolite) DisplayMetabolites();
	this->RenderWidget->Render();
}
/**************************************************************************************************************************/
void CSIDisplay::ImageViewer( vtkImageData * vtkImage)
{
	int * size = vtkImage->GetExtent();

	//Create the three plane widgets.
	this->PlaneX->SetInput(vtkImage);
	this->PlaneX->DisplayTextOn();
	this->PlaneX->SetPlaneOrientationToXAxes();
	this->PlaneX->RestrictPlaneToVolumeOn();
	this->PlaneX->SetSliceIndex(static_cast<int>(size[1]/2));
	this->PlaneX->SetPicker(this->Picker);
	this->PlaneX->SetKeyPressActivationValue('x');
	this->PlaneX->GetPlaneProperty()->SetColor(1,0,0);
	this->PlaneX->SetInteractor(this->RenderWidget->GetRenderWindow()->GetInteractor());
	this->PlaneX->On();

	this->PlaneY->SetInput(vtkImage);
	this->PlaneY->DisplayTextOn();
	this->PlaneY->SetPlaneOrientationToYAxes();
	this->PlaneY->RestrictPlaneToVolumeOn();
	this->PlaneY->SetSliceIndex(static_cast<int>(size[3]/2));
	this->PlaneY->SetPicker(this->Picker);
	this->PlaneY->SetKeyPressActivationValue('y');
	this->PlaneY->GetPlaneProperty()->SetColor(0,1,0);
	this->PlaneY->SetInteractor(this->RenderWidget->GetRenderWindow()->GetInteractor());
	this->PlaneY->On();

	this->PlaneZ->SetInput(vtkImage);
	this->PlaneZ->DisplayTextOn();
	this->PlaneZ->SetPlaneOrientationToZAxes();
	this->PlaneZ->RestrictPlaneToVolumeOn();
	this->PlaneZ->SetSliceIndex(static_cast<int>(size[5]/2));
	this->PlaneZ->SetPicker(this->Picker);
	this->PlaneZ->SetKeyPressActivationValue('z');
	this->PlaneZ->GetPlaneProperty()->SetColor(0,0,1);
	this->PlaneZ->SetInteractor(this->RenderWidget->GetRenderWindow()->GetInteractor());
	this->PlaneZ->On();

	// Compute Scalar Range for window and level of image
	double *range = vtkImage->GetScalarRange();	
	this->PlaneX->SetWindowLevel( range[1] - range[0], 0.5 * (range[1] + range[0]) );
	this->PlaneY->SetWindowLevel( range[1] - range[0], 0.5 * (range[1] + range[0]) );
	this->PlaneZ->SetWindowLevel( range[1] - range[0], 0.5 * (range[1] + range[0]) );

	this->RenderWidget->Reset();

}//end of ImageViewer
/**************************************************************************************************************************/
void CSIDisplay::SetMetabolite( const char * name)
{
	this->Metabolite = new char[100];
	strcpy(this->Metabolite,name);

}
/**************************************************************************************************************************/
double * CSIDisplay::GetMetabWholeRange()
{
	double  test;
	
	test = this->MetabWholeRange[1];

	return this->MetabWholeRange;
}
/**************************************************************************************************************************/
void CSIDisplay::SetMetabWholeRange(double * range)
{
	this->MetabWholeRange = range;
}
/**************************************************************************************************************************/
void CSIDisplay::SetMetabRange(double * range)
{
	this->MetabRange = range;
}
/**************************************************************************************************************************/
void CSIDisplay::SetCRLBRange(double range1, double range2)
{
	this->CRLBRange[0] = range1;
	this->CRLBRange[1] = range2;
}
/**************************************************************************************************************************/
double * CSIDisplay::GetMetabRange()
{
	return this->MetabRange;
}
/**************************************************************************************************************************/
double * CSIDisplay::GetCRLBRange()
{
	return this->CRLBRange;
}
/**************************************************************************************************************************/
void CSIDisplay::SetRatioState( int ratio)
{
	this->UseRatio = ratio;
}
/**************************************************************************************************************************/
void CSIDisplay::SetImage( vtkKWImage * image)
{
	ImageProcessing * processing = new ImageProcessing();
	processing->SetImage(image);
	processing->ResampleImage();
	
	this->KWImage = processing->GetOutput();
	this->vtkImage = this->KWImage->GetVTKImage();
	
	this->Image = GetITK(this->KWImage);
}
/**************************************************************************************************************************/
void CSIDisplay::SetGrid(vtkStructuredGrid * grid)
{
	this->Grid = grid;
}
/**************************************************************************************************************************/
void CSIDisplay::AddSpectroscopy(CSIgrid *csi)
{
	this->Spectroscopy = csi;
	
}
/**************************************************************************************************************************/
void CSIDisplay::RemoveSpectroscopy(char * name)
{
	//std::vector<CSIgrid *>::iterator iter;
	//for (iter = this->Spectroscopy.begin(); iter != this->Spectroscopy.end(); ++iter)
	//{
	//	if (strcmp(name, (*iter)->GetName()) )
	//	{
	//		this->Spectroscopy.erase(iter);
	//	}
	//}
}
/**************************************************************************************************************************/
void CSIDisplay::SetMask( vtkKWImage * image)
{
	ImageProcessing * processing = new ImageProcessing();
	processing->SetImage(image);
	processing->ResampleImage();

	this->Mask =  processing->GetOutput()->GetVTKImage();
}
/**************************************************************************************************************************/
ImageType::Pointer CSIDisplay::GetITK(vtkKWImage * image)
{	
	ImageType::Pointer output = ImageType::New();

	const ImageType * testImage = static_cast< const ImageType * >(image->GetITKImageBase() );
	output = const_cast<ImageType *>(testImage);
	
	return output;
}
/**************************************************************************************************************************/
void CSIDisplay::PickEvent()
{
	std::ofstream debugStream;
	debugStream.open("E:\\Development\\output\\pickEvent.txt");

	debugStream << "Test" <<std::endl;

	debugStream.close();	
}
/**************************************************************************************************************************/
void CSIDisplay::PickingOn()
{
	this->GridActor->SetPickable(1);
	
}
/**************************************************************************************************************************/
void CSIDisplay::PickingOff()
{
	this->GridActor->SetPickable(0);
	
}
/**************************************************************************************************************************/
std::vector<int> CSIDisplay::GetPickedCells()
{
	std::ofstream debugStream;
	debugStream.open("E:\\Development\\output\\pickEvent.txt");

	
	this->PickedCellIDs = this->Pickcallback->GetCellIDs();

	this->PickedCells = this->Pickcallback->GetCellCollection();

	std::vector<vtkStructuredGrid *>::iterator iter;

	debugStream << this->PickedCells.size()<<std::endl;

	
	return this->PickedCellIDs;
	debugStream.close();	
}
/**************************************************************************************************************************/
double CSIDisplay::Median(std::vector<double> * data)
{

	double median;

	int size = data->size();
	if (size >0)
	{
		std::sort( data->begin(), data->end() );

		int middle = size / 2;

		if ( middle % 2)
		{
			median = ( (data->at(middle)) + (data->at(middle + 1) )) / 2;
		}
		else
		{
			median = data->at(middle);
		}
		
		return median;
	}
	else return 0;

}
/**************************************************************************************************************************/
double CSIDisplay::STDev(std::vector<double> * data)
{

	double std = 0.0;
	double dev = 0.0;
	double sumDev = 0.0;
	double sum = 0.0;
	double ave = 0.0;

	int size = data->size();
	
	for (int i = 0; i < size; i++)
	{
		sum += data->at(i);
	}
	ave = sum/size;

	for (int j = 0; j < size; j++)
	{
		dev = ave - data->at(j);
		sumDev= sumDev + dev*dev;
	}

	std = sqrt(sumDev/size);

	return std;

}
/**************************************************************************************************************************/
void CSIDisplay::SetComponent(int a)
{
	this->Component = a;
}
/**************************************************************************************************************************/
int CSIDisplay::GetComponent()
{
	return this->Component;
}
/**************************************************************************************************************************/
void CSIDisplay::CalcCRLBRange()
{
	/**************************************************************/
	//Get the metabolite data from cell data
	vtkDataArray * metabArray = vtkDoubleArray::New();
	metabArray = this->Grid->GetCellData()->GetArray(this->Metabolite);

	/**************************************************************/
	//Determine Range of CRLB
	std::vector<double> * medianCRLB = new std::vector<double>;
	std::vector<double> * medianMetab = new std::vector<double>;

	double tempCRLB;
	double sumCRLB = 0.0;
	double aveCRLB = 0.0;
	double sumMetab = 0.0;
	double aveMetab = 0.0;
	int count = 0;
	int count2 = 0;

	for (int i = 0; i < metabArray->GetNumberOfTuples(); i++)
	{
		tempCRLB = metabArray->GetComponent(i,1); 

		if (tempCRLB > 0 && tempCRLB < 20)
		{
			medianCRLB->push_back(tempCRLB);	
			sumCRLB += tempCRLB;
			count++;
		}
	}

	aveCRLB = sumCRLB/count; 
	double stdCRLB = STDev(medianCRLB);
	/**************************************************************/
	//Determine Range of Metabolite Concentrations to map
	this->CRLBRange[0] = 0; 
	this->CRLBRange[1] = aveCRLB + 2*stdCRLB;

}
/**************************************************************************************************************************/
void CSIDisplay::CalcMetabRange()
{
	/**************************************************************/
	//Get the metabolite data from cell data
	vtkDataArray * metabArray = vtkDoubleArray::New();
	metabArray = this->Grid->GetCellData()->GetArray(this->Metabolite);

	double aveMetab = -1.0;
	double stdMetab = 0.0;
	/**************************************************************/
	//Determine Range of CRLB
	std::vector<double> * medianCRLB = new std::vector<double>;
	std::vector<double> * medianMetab = new std::vector<double>;

	double tempCRLB, tempMetab;
	double sumCRLB = 0.0;
	double aveCRLB = 0.0;
	double sumMetab = 0.0;

	int count = 0;
	int count2 = 0;

	/**************************************************************/
	//Determine Range of Filtered Metabolite Data
	for (int j = 0; j < metabArray->GetNumberOfTuples(); j++)
	{
		tempCRLB = metabArray->GetComponent(j,1);
		if (this->Component == 2 || this->Component == 1)
		{
			tempMetab = metabArray->GetComponent(j,this->Component);
		}
		else
		{
			tempMetab = metabArray->GetComponent(j,0); 
		}
		double t1 = this->CRLBRange[0];
		double t2 = this->CRLBRange[1];
		if (  (tempCRLB > this->CRLBRange[0] ) && (tempCRLB < this->CRLBRange[1] ) ) 
		{
			medianMetab->push_back(tempMetab);	
			this->Grid->GetCellData()->GetArray(this->Metabolite)->SetComponent(j,3,tempMetab);
			sumMetab += tempMetab;
			count2++;
		}
		else
		{
			this->Grid->GetCellData()->GetArray(this->Metabolite)->SetComponent(j,3,0);
		}
	}

	aveMetab = sumMetab/count2;
	stdMetab = STDev(medianMetab);
	double medMetab = Median(medianMetab);

	/**************************************************************/
	//Determine Range of Metabolite Concentrations to map
	if (medMetab > 0)
	{
		if (aveMetab > stdMetab)
		{
			this->MetabRange[0] = aveMetab - stdMetab; 
			this->MetabRange[1] = aveMetab + stdMetab;
		}
		else
		{
			this->MetabRange[0] = 0;
			this->MetabRange[1] = 2 * medMetab;
		}

		if ( aveMetab > 2*stdMetab)
		{
			this->MetabWholeRange[0] = aveMetab - 2*stdMetab; 
			this->MetabWholeRange[1] = aveMetab + 2*stdMetab;
		}
		else
		{
			this->MetabWholeRange[0] = 0;
			this->MetabWholeRange[1] = 2 * medMetab;
		}
	}
	else 
	{
		this->MetabRange[0] = 0; 
		this->MetabRange[1] = 1;
		this->MetabWholeRange[0] = 0;
		this->MetabWholeRange[1] = 1;
	}
	/**************************************************************/
	
}
/**************************************************************************************************************************/
void CSIDisplay::DisplayMetabolites()
{
	this->GridActor->GetMapper()->SetScalarModeToUseCellFieldData();
	this->GridActor->GetMapper()->ColorByArrayComponent(this->Metabolite, this->Component); 
	char * name = this->GridActor->GetMapper()->GetArrayName();
	this->GridActor->GetMapper()->SetScalarRange(this->MetabRange[0], this->MetabRange[1]); 
			
	vtkLookupTable * lut = vtkLookupTable::New();
	lut->SetHueRange(0.667,0.0);
			
	this->GridActor->GetMapper()->SetLookupTable(lut);

	this->GridActor->GetProperty()->SetRepresentationToSurface();
	this->GridActor->GetProperty()->EdgeVisibilityOn();
	this->GridActor->GetProperty()->SetOpacity(0.25);

	this->RenderWidget->GetRenderer()->AddActor(this->GridActor);

}
#endif